#include "synthetic_attrs.h"

#include "db_fetcher.h"
#include "exception.h"
#include "strings.h"
#include "utils.h"

#include <passport/infra/daemons/blackbox/src/helpers/display_name_helper.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_encryptor.h>
#include <passport/infra/daemons/blackbox/src/totp/totp_profile.h>

#include <passport/infra/libs/cpp/phonenumber/phonenumber.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

using namespace std::placeholders;

namespace NPassport::NBb {
    namespace {
        // minimal format check
        bool ValidBirthDate(const TString& date) {
            return date.size() == 10 && date[4] == '-' && date[7] == '-';
        }
    }

    TSyntheticAttributes::TSyntheticAttributes(time_t fullFioStartTime, TProxyAttrsMap&& proxyAttributes)
        : FullFioStartTime_(fullFioStartTime)
        , ProxyAttributes_(std::move(proxyAttributes))
    {
    }

    // parse yyyy-mm-dd birthdate and return mm-dd part
    TString TSyntheticAttributes::AttrBirthDayFunc(const TDbProfile* profile, TDbIndex birthdate) {
        const TDbValue* date = profile->Get(birthdate);
        if (date->Exists && ValidBirthDate(date->Value)) {
            return date->Value.substr(5);
        }

        return TStrings::EMPTY;
    }

    // parse yyyy-mm-dd birthdate and return yyyy if it is meaningful
    TString TSyntheticAttributes::AttrBirthYearFunc(const TDbProfile* profile, TDbIndex birthdate) {
        const TDbValue* date = profile->Get(birthdate);
        if (date->Exists && ValidBirthDate(date->Value)) {
            TString year = date->Value.substr(0, 4);
            if (year != "0000") {
                return year;
            }
        }

        return TStrings::EMPTY;
    }

    // check that 2FA is ON (i.e. totp secret exists)
    TString TSyntheticAttributes::Attr2FaOnFunc(const TDbProfile* profile, TDbIndex secret) {
        const TDbValue* totpSecret = profile->Get(secret);
        return totpSecret->Value.empty() ? TStrings::EMPTY : TStrings::ONE;
    }

    // return 2FA pin if any
    TString TSyntheticAttributes::Attr2FaPinFunc(const TDbProfile* profile, TDbIndex secret, const TTotpEncryptor* totpEncryptor) {
        const TDbValue* totpSecret = profile->Get(secret);
        if (!totpEncryptor || totpSecret->Value.empty()) { // no way to decrypt the secret
            return TStrings::EMPTY;
        }

        ui64 uid = TUtils::ToUInt(profile->Uid(), TStrings::UID);
        return totpEncryptor->Decrypt(uid, totpSecret->Value).Pin();
    }

    // check that user has password (plain old one or totp secret)
    TString TSyntheticAttributes::AttrHavePasswordFunc(const TDbProfile* profile, TDbIndex passwd, TDbIndex secret) {
        const TDbValue* val1 = profile->Get(passwd);
        const TDbValue* val2 = profile->Get(secret);
        return val1->Value.empty() && val2->Value.empty() ? TStrings::EMPTY : TStrings::ONE;
    }

    // check that user has hinta & hintq (and hintq has valid format)
    TString TSyntheticAttributes::AttrHaveHintFunc(const TDbProfile* profile, TDbIndex hintq, TDbIndex hinta) {
        TStringBuf hq = profile->Get(hintq)->Value;
        TStringBuf ha = profile->Get(hinta)->Value;
        NUtils::Trim(hq);
        NUtils::Trim(ha);

        if (hq.empty() || ha.empty()) {
            return TStrings::EMPTY;
        }

        size_t colon = hq.find(':');
        // check sanity of hintq/hinta data
        if (colon != TStringBuf::npos && colon > 0) { // should have : inside
            TStringBuf id = hq.substr(0, colon);
            if (NUtils::DigitsOnly(id) && id[0] != '0') { // don't allow junk or disabled
                // for custom hintq allow only non-empty ones
                if (id == "99" && colon == hq.size() - 1) {
                    return TStrings::EMPTY;
                }
                return TStrings::ONE;
            }
        }

        return TStrings::EMPTY;
    }

    // return user fio as lastname + ' ' + firstname
    TString TSyntheticAttributes::AttrFioFunc(const TDbProfile* profile, TDbIndex firstname, TDbIndex lastname) {
        const TString& fname = profile->Get(firstname)->Value;
        const TString& lname = profile->Get(lastname)->Value;

        if (!fname && !lname) {
            return {}; // no FIO
        }

        return NUtils::CreateStr(lname, ' ', fname);
    }

    // return normalized login (as in accounts.login.uid dbfield)
    TString TSyntheticAttributes::AttrNormalizedLogin(const TDbProfile* profile,
                                                      TDbIndex pdd,
                                                      TDbIndex portal,
                                                      TDbIndex alt,
                                                      TDbIndex social,
                                                      TDbIndex lite,
                                                      TDbIndex phne,
                                                      TDbIndex mailish,
                                                      TDbIndex uber,
                                                      TDbIndex yambot,
                                                      TDbIndex colonkish,
                                                      TDbIndex kpId,
                                                      TDbIndex neophonish,
                                                      TDbIndex scholar) {
        const TDbValue* login = profile->Get(portal);
        if (login->Exists) {
            return login->Value;
        }
        const TDbValue* altLogin = profile->Get(alt);
        if (altLogin->Exists) {
            return altLogin->Value;
        }
        const TDbValue* socialLogin = profile->Get(social);
        if (socialLogin->Exists) {
            return socialLogin->Value;
        }
        const TDbValue* liteLogin = profile->Get(lite);
        if (liteLogin->Exists) {
            return liteLogin->Value;
        }
        const TDbValue* pddLogin = profile->Get(pdd);
        if (pddLogin->Exists) {
            return pddLogin->Value;
        }
        const TDbValue* scholarLogin = profile->Get(scholar);
        if (scholarLogin->Exists) {
            return scholarLogin->Value;
        }
        const TDbValue* phneLogin = profile->Get(phne);
        if (phneLogin->Exists) {
            return phneLogin->Value;
        }
        const TDbValue* neophneLogin = profile->Get(neophonish);
        if (neophneLogin->Exists) {
            return neophneLogin->Value;
        }
        const TDbValue* mailishLogin = profile->Get(mailish);
        if (mailishLogin->Exists) {
            return mailishLogin->Value;
        }
        const TDbValue* uberId = profile->Get(uber);
        if (uberId->Exists) {
            return uberId->Value;
        }
        const TDbValue* botId = profile->Get(yambot);
        if (botId->Exists) {
            return botId->Value;
        }
        const TDbValue* colonkishId = profile->Get(colonkish);
        if (colonkishId->Exists) {
            return colonkishId->Value;
        }
        const TDbValue* kpid = profile->Get(kpId);
        if (kpid->Exists) {
            return kpid->Value;
        }

        return TStrings::EMPTY;
    }

    TString TSyntheticAttributes::AttrIsAvailable(const TDbProfile* profile, TDbIndex disabled) {
        const bool accDisabled = profile->Get(disabled)->AsBoolean();
        if (accDisabled) {
            return TStrings::EMPTY;
        }

        if (profile->Pdd() && !profile->PddDomItem().Ena()) {
            return TStrings::EMPTY;
        }

        return TStrings::ONE;
    }

    TString TSyntheticAttributes::AttrSecretIds(const TDbProfile* profile,
                                                TDbIndex totpSecrets,
                                                const TTotpEncryptor* totpEncryptor) {
        const TDbValue* totpSecretsValue = profile->Get(totpSecrets);
        if (!totpEncryptor || totpSecretsValue->Value.empty()) {
            return TStrings::EMPTY;
        }

        TTotpProfile tp = totpEncryptor->Decrypt(TUtils::ToUInt(profile->Uid(), TStrings::UID),
                                                 totpSecretsValue->Value);

        return tp.GetSecretInfo();
    }

    TString TSyntheticAttributes::AttrHaveOrganizationName(const TDbProfile* profile) {
        return profile->PddDomItem().OrganizationName().empty() ? TStrings::EMPTY : TStrings::ONE;
    }

    TString TSyntheticAttributes::Attr2FaPinLength(const TDbProfile* profile, TDbIndex secret, const TTotpEncryptor* totpEncryptor) {
        std::size_t pinLength = Attr2FaPinFunc(profile, secret, totpEncryptor).size();
        if (pinLength == 0) {
            return TStrings::EMPTY;
        }

        return IntToString<10>(pinLength);
    }

    // find given phone attributes, if any, or return nullptr
    static const TDbProfile::TAttrs* GetPhoneAttrs(const TDbProfile* profile, const TString& phoneid) {
        const TDbProfile::TExtendedEntities& entities = profile->ExtendedPhoneAttrs();
        TDbProfile::TExtendedEntities::const_iterator p = entities.find(phoneid);
        return p == entities.end() ? nullptr : &(p->second);
    }

    // return phone number, either from extended attributes or from legacy attribute
    TString TSyntheticAttributes::AttrPhoneNumber(const TDbProfile* profile, TDbIndex phoneId) {
        if (!profile->Get(phoneId)->Exists) {
            return TStrings::EMPTY;
        }

        // this account was transferred to extended attributes, search there
        const TDbProfile::TAttrs* attrs = GetPhoneAttrs(profile, profile->Get(phoneId)->Value);
        if (!attrs) {
            return TStrings::EMPTY;
        }
        TDbProfile::TAttrs::const_iterator p = attrs->find(TPhoneAttr::NUMBER);
        if (p == attrs->end() || p->second.Value.empty()) {
            return TStrings::EMPTY;
        }

        if (p->second.Value.front() == '+') {
            return p->second.Value;
        }

        return "+" + p->second.Value;
    }

    // return phone confirmation date, either from external attrs or from old attribute
    TString TSyntheticAttributes::AttrPhoneBornDate(const TDbProfile* profile, TDbIndex phoneId) {
        if (!profile->Get(phoneId)->Exists) {
            return TStrings::EMPTY;
        }

        // this account was transferred to extended attributes, search there
        const TDbProfile::TAttrs* attrs = GetPhoneAttrs(profile, profile->Get(phoneId)->Value);
        if (!attrs) {
            return TStrings::EMPTY;
        }
        TDbProfile::TAttrs::const_iterator p = attrs->find(TPhoneAttr::CONFIRMED);
        if (p == attrs->end() || p->second.Value.empty()) {
            return TStrings::EMPTY;
        }

        return p->second.Value;
    }

    static const TString RED = "4";
    static const TString YELLOW = "16";
    static const TString GREEN = "32";
    static const TString GREY = "-1";
    TString TSyntheticAttributes::AttrProtectionLevel(const TDbProfile* profile,
                                                      TDbIndex hintQId,
                                                      TDbIndex hintAId,
                                                      TDbIndex pwdQualityId,
                                                      TDbIndex strongPolicyId,
                                                      TDbIndex pwdHashId,
                                                      TDbIndex totpSecretId) {
        // 2FA
        bool has2FA = Attr2FaOnFunc(profile, totpSecretId) == TStrings::ONE;
        if (has2FA) {
            return GREEN;
        }

        // No pwd
        bool hasPwd = !profile->Get(pwdHashId)->Value.empty();
        if (!hasPwd) {
            return GREY;
        }

        bool hasHint = AttrHaveHintFunc(profile, hintQId, hintAId) == TStrings::ONE;
        bool hasStrongPwd = HasProfileStrongPwd(profile, pwdQualityId, strongPolicyId);
        bool hasSafeRestore = HasProfileRestoreEmail(profile) || HasProfileRestorePhone(profile);

        // Strong pwd
        if (hasStrongPwd) {
            if (hasSafeRestore) {
                return GREEN;
            }
            if (hasHint) {
                return YELLOW;
            }
            return RED;
        }

        // Weak pwd
        if (hasSafeRestore) {
            return YELLOW;
        }

        return RED;
    }

    // check that RFC 2FA is ON (i.e. RFC_Totp secret exists)
    TString TSyntheticAttributes::AttrRfC2FaOnFunc(const TDbProfile* profile, TDbIndex secret) {
        const TDbValue* totpSecret = profile->Get(secret);
        return totpSecret->Value.empty() ? TStrings::EMPTY : TStrings::ONE;
    }

    TString TSyntheticAttributes::AttrHavePlus(const TDbProfile* profile, TDbIndex plusEna) {
        const TDbValue* enaVal = profile->Get(plusEna);
        return enaVal->AsBoolean() ? TStrings::ONE : TStrings::EMPTY;
    }

    TString TSyntheticAttributes::AttrConnectOrganizationIds(const TDbProfile* profile, TDbIndex externalIds) {
        // try to append organizationId to external ids, checking for doubles and sanitizing format
        const TString& internalId = profile->PddDomItem().OrganizationId();
        TString allIds = profile->Get(externalIds)->Value;

        if (allIds.empty() || internalId == allIds) {
            // simple case, no need to split/join the id list
            return NUtils::DigitsOnly(internalId) ? internalId : TStrings::EMPTY;
        }

        // first, append internal id to the list, then split/sanitize/deduplicate/join them all
        NUtils::Append(allIds, ',', internalId);

        TString result;
        result.reserve(allIds.size() + 1);

        for (const TStringBuf& it : NUtils::NormalizeListValue<TStringBuf>(allIds, ",")) {
            if (!NUtils::DigitsOnly(it)) {
                TLog::Error("Bad format of ACCOUNT_EXTERNAL_ORGANIZATION_IDS attribute: '%s' for uid=%s", allIds.c_str(), profile->Uid().c_str());
                return TStrings::EMPTY;
            }
            NUtils::AppendSeparated(result, ',', it);
        }

        return result;
    }

    TString TSyntheticAttributes::AttrPDDOrganizationId(const TDbProfile* profile) {
        const TString& internalId = profile->PddDomItem().OrganizationId();
        return NUtils::DigitsOnly(internalId) ? internalId : TStrings::EMPTY;
    }

    TString TSyntheticAttributes::AttrSuggestPublicName(const TDbProfile* profile,
                                                        TDbIndex firstname,
                                                        TDbIndex lastname,
                                                        TDbIndex displayname,
                                                        TDbIndex dontUseDisplayName,
                                                        TDbIndex showFIO,
                                                        TDbIndex registrationTime) const {
        bool hasDisplayName = !TDisplayNameHelper::DisplayNameEmpty(profile->Get(displayname)->Value);
        bool displayNameDisabled = profile->Get(dontUseDisplayName)->AsBoolean();
        // show full FIO if agreed to show FIO or if registered after given time
        bool fioAllowed = profile->Get(showFIO)->AsBoolean() || profile->Get(registrationTime)->AsTime() > FullFioStartTime_;
        if ((hasDisplayName && !displayNameDisabled) || fioAllowed) {
            return TStrings::EMPTY; // no suggest if has custom displayname or already accepted showing FIO
        }

        return TDisplayNameHelper::SuggestPublicName(profile->Get(firstname)->Value, profile->Get(lastname)->Value);
    }

    TString TSyntheticAttributes::AttrHasCustomPublicId(const TDbProfile* profile, TDbIndex publicId) {
        const TDbValue* id = profile->Get(publicId);
        return id->Value.empty() ? TStrings::EMPTY : TStrings::ONE;
    }

    TString TSyntheticAttributes::AttrPlusMappedSubscriptionLevel(const TDbProfile* profile, TDbIndex plusLevel) {
        const TString& level = profile->Get(plusLevel)->Value;

        if (level.empty()) {
            return TStrings::EMPTY;
        }

        ui32 levelValue;
        if (!TryIntFromString<10>(level, levelValue)) {
            TLog::Warning() << "Bad format of ACCOUNT_PLUS_SUBSCRIPTION_LEVEL attribute: '"
                            << level << "' for uid=" << profile->Uid();

            return TStrings::EMPTY;
        }

        switch (levelValue / 10) {
            case 1:
                return TStrings::ONE;
            case 2:
                return TStrings::TWO;
            case 3:
                return TStrings::THREE;
            case 4:
                return TStrings::FOUR;
            default:
                return TStrings::EMPTY;
        }
    }

    TString TSyntheticAttributes::AttrAccountCompletionRecommended(const TDbProfile* profile,
                                                                   TDbIndex neophonish,
                                                                   TDbIndex portal,
                                                                   TDbIndex firstname,
                                                                   TDbIndex lastname) {
        const auto toBool = [profile](const TDbIndex index) {
            return profile->Get(index)->AsBoolean();
        };
        return toBool(neophonish) && !toBool(portal) && !toBool(firstname) && !toBool(lastname) ? TStrings::ONE : TStrings::EMPTY;
    }

    TString TSyntheticAttributes::AttrAccountFamilyChildrenManagement(const TDbProfile* profile, TDbIndex realAttr) {
        bool isAdmin = profile->FamilyInfo().has_value() && profile->Uid() == profile->FamilyInfo()->AdminUid;
        return profile->Get(realAttr)->AsBoolean() || isAdmin ? TStrings::ONE : TStrings::EMPTY;
    }

    TString TSyntheticAttributes::AttrAccountIsKid(const TDbProfile* profile) {
        return profile->IsKid() ? TStrings::ONE : TStrings::EMPTY;
    }

    TString TSyntheticAttributes::ProxyAttr(const TDbProfile* profile, TDbIndex idx) {
        return profile->Get(idx)->Value;
    }

    TString TSyntheticAttributes::DefaultValueAttr(const TDbProfile* profile, TDbIndex mainAttr, TDbIndex defAttr) {
        const TDbValue* val = profile->Get(mainAttr);
        if (val->Exists) {
            return val->Value;
        }
        return profile->Get(defAttr)->Value;
    }

    bool TSyntheticAttributes::HasProfileStrongPwd(const TDbProfile* profile, TDbIndex pwdQualityId, TDbIndex strongPolicyId) {
        const TString& q = profile->Get(pwdQualityId)->Value;
        if (q.empty()) {
            return false;
        }

        std::size_t colPos = q.find(':');
        if (colPos == TString::npos) {
            return false;
        }

        unsigned long quality = 0;
        try {
            quality = IntFromString<ui32, 10>(q.substr(colPos + 1));
            if (quality > 100) {
                return false;
            }
        } catch (const std::exception&) {
            return false;
        }

        if (profile->Get(strongPolicyId)->AsBoolean()) {
            return quality >= 70;
        }
        return quality >= 15;
    }

    bool TSyntheticAttributes::HasProfileRestoreEmail(const TDbProfile* profile) {
        for (const auto& pairEntity : profile->ExtendedEmailAttrs()) {
            const TDbProfile::TAttrs& emailAttrs = pairEntity.second;

            auto isEmpty = [&emailAttrs](const TString& attr) -> bool {
                auto it = emailAttrs.find(attr);
                return it == emailAttrs.end() || it->second.Value.empty();
            };
            if (isEmpty(TEmailAttr::ADDRESS) || isEmpty(TEmailAttr::CONFIRMED)) {
                continue;
            }

            auto isTrue = [&emailAttrs](const TString& attr) -> bool {
                auto it = emailAttrs.find(attr);
                return it != emailAttrs.end() && it->second.AsBoolean();
            };
            if (isTrue(TEmailAttr::IS_UNSAFE) || isTrue(TEmailAttr::IS_SILENT) || isTrue(TEmailAttr::IS_RPOP)) {
                continue;
            }

            return true;
        }

        return false;
    }

    bool TSyntheticAttributes::HasProfileRestorePhone(const TDbProfile* profile) {
        for (const auto& pairEntity : profile->ExtendedPhoneAttrs()) {
            const TDbProfile::TAttrs& phoneAttrs = pairEntity.second;

            auto isEmpty = [&phoneAttrs](const TString& attr) -> bool {
                auto it = phoneAttrs.find(attr);
                return it == phoneAttrs.end() || it->second.Value.empty();
            };
            if (isEmpty(TPhoneAttr::NUMBER) || isEmpty(TPhoneAttr::CONFIRMED)) {
                continue;
            }

            if (!isEmpty(TPhoneAttr::SECURED)) {
                return true;
            }

            try {
                const time_t MAGIC_MOMENT = 1380191400;
                if (IntFromString<time_t, 10>(phoneAttrs.find(TPhoneAttr::CONFIRMED)->second.Value) < MAGIC_MOMENT) {
                    return true;
                }
            } catch (const std::exception&) {
            }
        }

        return false;
    }

    // given attr for given type, empty otherwise
    TString TSyntheticAttributes::AttrExtGetAttribute(const TDbProfile* profile,
                                                      const TString& entityId,
                                                      const TString& type,
                                                      const TString& attr) {
        const TDbProfile::TExtendedEntities* cont = nullptr;
        if (type == TExtendedAttrType::PHONE) {
            cont = &profile->ExtendedPhones_;
        } else if (type == TExtendedAttrType::EMAIL) {
            cont = &profile->ExtendedEmails_;
        } else if (type == TExtendedAttrType::WEBAUTHN) {
            cont = &profile->ExtendedWebauthn_;
        } else {
            return TStrings::EMPTY;
        }

        TDbProfile::TExtendedEntities::const_iterator itEntity = cont->find(entityId);
        if (itEntity == cont->end()) {
            return TStrings::EMPTY;
        }
        TDbProfile::TAttrs::const_iterator itAttr = itEntity->second.find(attr);
        if (itAttr == itEntity->second.end()) {
            return TStrings::EMPTY;
        }
        const TDbValue& dbValue = itAttr->second;
        if (!dbValue.Exists || dbValue.Value.empty()) {
            return TStrings::EMPTY;
        }

        return dbValue.Value;
    }

    // given attr for given type, empty otherwise
    TString TSyntheticAttributes::AttrExtHasAttribute(const TDbProfile* profile,
                                                      const TString& entityId,
                                                      const TString& type,
                                                      const TString& attr) {
        TString value = AttrExtGetAttribute(profile, entityId, type, attr);
        return value.empty() ? TStrings::EMPTY : TStrings::ONE;
    }

    static const TString DASH_ = "-";
    static const TString FIGURE_DASH_ = "\u2012";
    TString TSyntheticAttributes::AttrExtFormattedNumber(const TDbProfile* profile,
                                                         const TString& entityId) {
        TString value = AttrExtE164Number(profile, entityId);
        if (value.empty()) {
            return value;
        }

        TString formated = NPhoneNumber::TUtils::FormatInternational(value);
        return NUtils::ReplaceAny(formated, DASH_, FIGURE_DASH_);
    }

    TString TSyntheticAttributes::AttrExtE164Number(const TDbProfile* profile,
                                                    const TString& entityId) {
        TString value = AttrExtGetAttribute(profile, entityId, TExtendedAttrType::PHONE, TPhoneAttr::NUMBER);
        if (value.empty()) {
            return value;
        }

        if (value.front() != '+') {
            value.insert(size_t(0), 1, '+');
        }

        return value;
    }

    TString TSyntheticAttributes::AttrExtMaskedFormattedNumber(const TDbProfile* profile,
                                                               const TString& entityId) {
        return TUtils::MaskPhoneNumber(AttrExtFormattedNumber(profile, entityId));
    }

    TString TSyntheticAttributes::AttrExtMaskedE164Number(const TDbProfile* profile,
                                                          const TString& entityId) {
        return TUtils::MaskPhoneNumber(AttrExtE164Number(profile, entityId));
    }

    TString TSyntheticAttributes::AttrExtPhoneIsDefault(const TDbProfile* profile, const TString& entityId) {
        return !entityId.empty() && profile->DefaultPhoneId() == entityId ? TStrings::ONE : TStrings::EMPTY;
    }

    TString TSyntheticAttributes::AttrExtPhoneIsBank(const TDbProfile* profile, const TString& entityId) {
        const TString& bankAlias = profile->GetAlias(TAlias::BANK_PHONE_NUMBER);
        if (bankAlias.empty()) {
            return TStrings::EMPTY;
        }

        TString number = AttrExtGetAttribute(profile, entityId, TExtendedAttrType::PHONE, TPhoneAttr::NUMBER);
        return number == bankAlias ? TStrings::ONE : TStrings::EMPTY;
    }

    TSyntheticAttributes::TAttrFuncType TSyntheticAttributes::GetSyntheticFunc(TDbFetcher& fetcher, const TString& attr) const {
        const TAttrFuncs& funcs = AttrFuncs();

        auto it = funcs.find(attr);
        if (it != funcs.end()) {
            return it->second(fetcher, this);
        }

        auto proxy = ProxyAttributes_.find(attr);
        if (proxy != ProxyAttributes_.end()) {
            return ProxyAttrFunc(fetcher, proxy->second);
        }

        // unknown attribute
        throw TBlackboxError(TBlackboxError::EType::InvalidParams)
            << "invalid synthetic attribute type: " << InvalidValue(attr);
    }

    TSyntheticAttributes::TExtAttrFuncType TSyntheticAttributes::GetSyntheticExtPhoneFunc(TDbFetcher& fetcher, const TString& attr) const {
        const TExtAttrFuncs& funcs = PhoneAttrFuncs();

        auto it = funcs.find(attr);
        if (it == funcs.end()) {
            // unknown attribute
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid synthetic extended phone attribute type: " << InvalidValue(attr);
        }

        return it->second(fetcher, this);
    }

    TSyntheticAttributes::TExtAttrFuncType TSyntheticAttributes::GetSyntheticExtEmailFunc(TDbFetcher& fetcher, const TString& attr) const {
        const TExtAttrFuncs& funcs = EmailAttrFuncs();

        auto it = funcs.find(attr);
        if (it == funcs.end()) {
            // unknown attribute
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid synthetic extended email attribute type: " << InvalidValue(attr);
        }

        return it->second(fetcher, this);
    }

    TSyntheticAttributes::TExtAttrFuncType TSyntheticAttributes::GetSyntheticExtWebauthnFunc(TDbFetcher& fetcher, const TString& attr) const {
        const TExtAttrFuncs& funcs = WebauthnAttrFuncs();

        auto it = funcs.find(attr);
        if (it == funcs.end()) {
            // unknown attribute
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid synthetic extended webauthn attribute type: " << InvalidValue(attr);
        }

        return it->second(fetcher, this);
    }

    TSyntheticAttributes::TAttrFuncType TSyntheticAttributes::ProxyAttrFunc(TDbFetcher& fetcher, const TString& attr) {
        return [attrIdx = fetcher.AddAttr(attr)](
                   const TDbProfile* profile) {
            return TSyntheticAttributes::ProxyAttr(profile, attrIdx);
        };
    }

    const TSyntheticAttributes::TAttrFuncs& TSyntheticAttributes::AttrFuncs() {
        static const TAttrFuncs funcs = {
            {TAttr::PERSON_BIRTH_DAY, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [birthdate = fetcher.AddAttr(TAttr::PERSON_BIRTHDATE)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrBirthDayFunc(profile, birthdate);
                 };
             }},

            {TAttr::PERSON_BIRTH_YEAR, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [birthdate = fetcher.AddAttr(TAttr::PERSON_BIRTHDATE)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrBirthYearFunc(profile, birthdate);
                 };
             }},

            {TAttr::ACCOUNT_2FA_ON, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [totp = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_SECRET)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::Attr2FaOnFunc(profile, totp);
                 };
             }},

            {TAttr::ACCOUNT_2FA_PIN, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [totp = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_SECRET),
                         encryptor = fetcher.TotpEncryptor_](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::Attr2FaPinFunc(profile, totp, encryptor);
                 };
             }},

            {TAttr::ACCOUNT_HAVE_PASSWORD, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [pwd = fetcher.AddAttr(TAttr::PASSWORD_ENCRYPTED),
                         totp = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_SECRET)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrHavePasswordFunc(profile, pwd, totp);
                 };
             }},

            {TAttr::ACCOUNT_HAVE_HINT, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [hintq = fetcher.AddAttr(TAttr::HINT_QUESTION_SERIALIZED),
                         hinta = fetcher.AddAttr(TAttr::HINT_ANSWER_ENCRYPTED)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrHaveHintFunc(profile, hintq, hinta);
                 };
             }},

            {TAttr::PERSON_FIO, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [first = fetcher.AddAttr(TAttr::PERSON_FIRSTNAME),
                         last = fetcher.AddAttr(TAttr::PERSON_LASTNAME)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrFioFunc(profile, first, last);
                 };
             }},

            {TAttr::ACCOUNT_NORMALIZED_LOGIN, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [pddMaster = fetcher.AddAlias(TAlias::PDD_MASTER_LOGIN),
                         portal = fetcher.AddAlias(TAlias::PORTAL_LOGIN),
                         altDomain = fetcher.AddAlias(TAlias::ALT_DOMAIN_LOGIN),
                         social = fetcher.AddAlias(TAlias::SOCIAL_LOGIN),
                         lite = fetcher.AddAlias(TAlias::LITE_LOGIN),
                         phony = fetcher.AddAlias(TAlias::PHONY_LOGIN),
                         malish = fetcher.AddAlias(TAlias::MAILISH_LOGIN),
                         uber = fetcher.AddAlias(TAlias::UBER_ID),
                         yambot = fetcher.AddAlias(TAlias::YAMBOT),
                         colonkish = fetcher.AddAlias(TAlias::COLONKISH),
                         kinopoisk = fetcher.AddAlias(TAlias::KINOPOISK_ID),
                         neophonish = fetcher.AddAlias(TAlias::NEOPHONISH),
                         scholar = fetcher.AddAlias(TAlias::SCHOLAR)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrNormalizedLogin(profile,
                                                                      pddMaster,
                                                                      portal,
                                                                      altDomain,
                                                                      social,
                                                                      lite,
                                                                      phony,
                                                                      malish,
                                                                      uber,
                                                                      yambot,
                                                                      colonkish,
                                                                      kinopoisk,
                                                                      neophonish,
                                                                      scholar);
                 };
             }},

            {TAttr::ACCOUNT_IS_AVAILABLE, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [disabled = fetcher.AddAttr(TAttr::ACCOUNT_DISABLED)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrIsAvailable(profile, disabled);
                 };
             }},

            {TAttr::ACCOUNT_SECRET_IDS, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [totp = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_SECRET),
                         encrypted = fetcher.TotpEncryptor_](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrSecretIds(profile, totp, encrypted);
                 };
             }},

            {TAttr::ACCOUNT_HAVE_ORGANIZATION_NAME, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 Y_UNUSED(fetcher);
                 return [](const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrHaveOrganizationName(profile);
                 };
             }},

            {TAttr::ACCOUNT_PIN_LENGTH, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [totp = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_SECRET),
                         encryptor = fetcher.TotpEncryptor_](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::Attr2FaPinLength(profile, totp, encryptor);
                 };
             }},

            {TAttr::PHONE_NUMBER, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);

                 return [phone = fetcher.AddAttr(TAttr::PHONES_SECURE)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrPhoneNumber(profile, phone);
                 };
             }},

            {TAttr::PHONE_CONFIRMATION_DATETIME, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::CONFIRMED);

                 return [confirmed = fetcher.AddAttr(TAttr::PHONES_SECURE)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrPhoneBornDate(profile, confirmed);
                 };
             }},

            {TAttr::ACCOUNT_PROTECTION_LEVEL, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedEmailAttr(TEmailAttr::ADDRESS);
                 fetcher.AddExtendedEmailAttr(TEmailAttr::CONFIRMED);
                 fetcher.AddExtendedEmailAttr(TEmailAttr::IS_RPOP);
                 fetcher.AddExtendedEmailAttr(TEmailAttr::IS_SILENT);
                 fetcher.AddExtendedEmailAttr(TEmailAttr::IS_UNSAFE);

                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::SECURED);
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::CONFIRMED);

                 return [hintq = fetcher.AddAttr(TAttr::HINT_QUESTION_SERIALIZED),
                         hinta = fetcher.AddAttr(TAttr::HINT_ANSWER_ENCRYPTED),
                         pwdQuality = fetcher.AddAttr(TAttr::PASSWORD_QUALITY),
                         strongPwdPolicy = fetcher.AddAttr(TAttr::ACCOUNT_STRONG_PASSWORD_REQUIRED),
                         pwd = fetcher.AddAttr(TAttr::PASSWORD_ENCRYPTED),
                         totp = fetcher.AddAttr(TAttr::ACCOUNT_TOTP_SECRET)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrProtectionLevel(
                         profile,
                         hintq,
                         hinta,
                         pwdQuality,
                         strongPwdPolicy,
                         pwd,
                         totp);
                 };
             }},

            {TAttr::ACCOUNT_RFC_2FA_ON, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [totp = fetcher.AddAttr(TAttr::ACCOUNT_RFC_TOTP_SECRET)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrRfC2FaOnFunc(profile, totp);
                 };
             }},

            {TAttr::ACCOUNT_HAVE_PLUS, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [plus = fetcher.AddAttr(TAttr::ACCOUNT_PLUS_ENABLED)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrHavePlus(profile, plus);
                 };
             }},

            {TAttr::ACCOUNT_CONNECT_ORGANIZATION_IDS, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [orgIds = fetcher.AddAttr(TAttr::ACCOUNT_EXTERNAL_ORGANIZATION_IDS)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrConnectOrganizationIds(profile, orgIds);
                 };
             }},

            {TAttr::ACCOUNT_PDD_ORGANIZATION_ID, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 Y_UNUSED(fetcher);
                 return [](const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrPDDOrganizationId(profile);
                 };
             }},

            {TAttr::ACCOUNT_SUGGEST_PUBLIC_NAME, [](TDbFetcher& fetcher, const TSyntheticAttributes* attrs) {
                 return [attrs,
                         first = fetcher.AddAttr(TAttr::PERSON_FIRSTNAME),
                         last = fetcher.AddAttr(TAttr::PERSON_LASTNAME),
                         displayName = fetcher.AddAttr(TAttr::ACCOUNT_DISPLAY_NAME),
                         dontUseDisplayName = fetcher.AddAttr(TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME),
                         fioInPublicName = fetcher.AddAttr(TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME),
                         regDate = fetcher.AddAttr(TAttr::ACCOUNT_REGISTRATION_DATETIME)](
                            const TDbProfile* profile) {
                     return attrs->AttrSuggestPublicName(profile,
                                                         first,
                                                         last,
                                                         displayName,
                                                         dontUseDisplayName,
                                                         fioInPublicName,
                                                         regDate);
                 };
             }},

            {TAttr::ACCOUNT_HAS_CUSTOM_PUBLIC_ID, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [publicId = fetcher.AddAlias(TAlias::PUBLICID)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrHasCustomPublicId(profile, publicId);
                 };
             }},

            {TAttr::ACCOUNT_MUSIC_CONTENT_RATING_KLASS, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [musicRating = fetcher.AddAttr(TAttr::ACCOUNT_MUSIC_CONTENT_RATING_CLASS),
                         rating = fetcher.AddAttr(TAttr::ACCOUNT_CONTENT_RATING_CLASS)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::DefaultValueAttr(profile, musicRating, rating);
                 };
             }},

            {TAttr::ACCOUNT_VIDEO_CONTENT_RATING_KLASS, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [videoRating = fetcher.AddAttr(TAttr::ACCOUNT_VIDEO_CONTENT_RATING_CLASS),
                         rating = fetcher.AddAttr(TAttr::ACCOUNT_CONTENT_RATING_CLASS)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::DefaultValueAttr(profile, videoRating, rating);
                 };
             }},

            {TAttr::ACCOUNT_PLUS_MAPPED_SUBSCRIPTION_LEVEL, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [plusLevel = fetcher.AddAttr(TAttr::ACCOUNT_PLUS_SUBSCRIPTION_LEVEL)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrPlusMappedSubscriptionLevel(profile, plusLevel);
                 };
             }},

            {TAttr::ACCOUNT_COMPLETION_RECOMMENDED, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 return [neophonish = fetcher.AddAlias(TAlias::NEOPHONISH),
                         portal = fetcher.AddAlias(TAlias::PORTAL_LOGIN),
                         first = fetcher.AddAttr(TAttr::PERSON_FIRSTNAME),
                         last = fetcher.AddAttr(TAttr::PERSON_LASTNAME)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrAccountCompletionRecommended(profile, neophonish, portal, first, last);
                 };
             }},

            {TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT_SYNT, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddFamilyInfo();
                 return [realAttr = fetcher.AddAttr(TAttr::ACCOUNT_FAMILY_CHILDREN_MANAGEMENT)](
                            const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrAccountFamilyChildrenManagement(profile, realAttr);
                 };
             }},

            {TAttr::ACCOUNT_IS_KID, [](TDbFetcher&, const TSyntheticAttributes*) {
                 return [](const TDbProfile* profile) {
                     return TSyntheticAttributes::AttrAccountIsKid(profile);
                 };
             }},

        };

        return funcs;
    }

    const TSyntheticAttributes::TExtAttrFuncs& TSyntheticAttributes::PhoneAttrFuncs() {
        static const TExtAttrFuncs funcs = {
            {TPhoneAttr::FORMATTED_NUMBER, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtFormattedNumber(profile, entityId);
                 };
             }},

            {TPhoneAttr::E164_NUMBER, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtE164Number(profile, entityId);
                 };
             }},

            {TPhoneAttr::MASKED_FORMATTED_NUMBER, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtMaskedFormattedNumber(profile, entityId);
                 };
             }},

            {TPhoneAttr::MASKED_E164_NUMBER, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtMaskedE164Number(profile, entityId);
                 };
             }},

            {TPhoneAttr::IS_CONFIRMED, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::CONFIRMED);
                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtHasAttribute(
                         profile,
                         entityId,
                         TExtendedAttrType::PHONE,
                         TPhoneAttr::CONFIRMED);
                 };
             }},

            {TPhoneAttr::IS_BOUND, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::BOUND);
                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtHasAttribute(
                         profile,
                         entityId,
                         TExtendedAttrType::PHONE,
                         TPhoneAttr::BOUND);
                 };
             }},

            {TPhoneAttr::IS_DEFAULT, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddAttr(TAttr::PHONES_DEFAULT);
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::BOUND);

                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtPhoneIsDefault(profile, entityId);
                 };
             }},

            {TPhoneAttr::IS_SECURED, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::SECURED);

                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtHasAttribute(profile, entityId, TExtendedAttrType::PHONE, TPhoneAttr::SECURED);
                 };
             }},

            {TPhoneAttr::IS_BANK, [](TDbFetcher& fetcher, const TSyntheticAttributes*) {
                 fetcher.AddAlias(TAlias::BANK_PHONE_NUMBER);
                 fetcher.AddExtendedPhoneAttr(TPhoneAttr::NUMBER);

                 return [](const TDbProfile* profile, const TString& entityId) {
                     return TSyntheticAttributes::AttrExtPhoneIsBank(profile, entityId);
                 };
             }},
        };

        return funcs;
    }

    const TSyntheticAttributes::TExtAttrFuncs& TSyntheticAttributes::EmailAttrFuncs() {
        static const TExtAttrFuncs funcs = {};

        return funcs;
    }

    const TSyntheticAttributes::TExtAttrFuncs& TSyntheticAttributes::WebauthnAttrFuncs() {
        static const TExtAttrFuncs funcs = {};

        return funcs;
    }
}
