#include "utils.h"

#include "attributes.h"
#include "db_fetcher.h"
#include "dbfields_settings.h"
#include "exception.h"
#include "experiment.h"
#include "session_utils.h"
#include "strings.h"
#include "ya_domains.h"

#include <passport/infra/daemons/blackbox/src/loggers/tskvlog.h>

#include <passport/infra/libs/cpp/auth_core/session.h>
#include <passport/infra/libs/cpp/auth_core/sessionutils.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/phonenumber/phonenumber.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/openssl/include/openssl/sha.h>

#include <util/charset/utf8.h>
#include <util/generic/hash.h>

#include <algorithm>
#include <ctime>

namespace NPassport::NBb {
    ESanitizeStatus TUtils::SanitizeLogin(const TStringBuf login, size_t maxLoginSize) {
        if (login.size() > maxLoginSize) {
            TLog::Debug() << "Got too big login '" << login.SubString(0, 20)
                          << "...': " << login.size()
                          << " bytes more than " << maxLoginSize;
            return ESanitizeStatus::TooLong;
        }

        // check for control characters
        auto it = std::find_if(login.cbegin(), login.cend(), [](const char c) {
            return unsigned(c) < ' ';
        });
        return it == login.cend()
                   ? ESanitizeStatus::Ok
                   : ESanitizeStatus::InvalidChars;
    }

    TString TUtils::NormalizeLogin(const TStringBuf login) {
        TString result = TolowerUtf(login);
        std::replace(result.begin(), result.vend(), '.', '-');
        return result;
    }

    TString TUtils::NormalizeTotp(const TStringBuf s) {
        TString result(NUtils::RemoveSpaces(s));
        NUtils::Tolower(result);

        return result;
    }

    void TUtils::DotsToHyphens(TString& s, bool convertEmailsAsWell) {
        TString::size_type atpos = s.find('@');
        if (atpos != TString::npos && !convertEmailsAsWell) {
            return;
        }
        TString::size_type rwall = atpos != TString::npos ? atpos : s.size();
        for (TString::size_type i = 0; i < rwall; ++i) {
            if (s[i] == '.') {
                s[i] = '-';
            }
        }
    }

    template <typename T>
    static T ToIntImpl(TStringBuf strval, TStringBuf name) {
        T value;
        if (!TryIntFromString<10>(strval, value)) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid " << name << " value: " << InvalidValue(strval);
        }
        return value;
    }

    i64 TUtils::ToInt(TStringBuf strval, TStringBuf name) {
        return ToIntImpl<i64>(strval, name);
    }

    ui64 TUtils::ToUInt(TStringBuf strval, TStringBuf name) {
        return ToIntImpl<ui64>(strval, name);
    }

    time_t TUtils::ToTime(TStringBuf strval) {
        time_t value = 0;
        if (strval && !TryIntFromString<10>(strval, value)) {
            throw std::invalid_argument("toTime() failed");
        }
        return value;
    }

    void TUtils::CheckArgValue(const TString& val, const TString& arg) {
        if (val.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Missing " << arg << " argument";
        }
    }

    void TUtils::CheckUserIpArg(const NCommon::TRequest& request) {
        return CheckArgValue(request.GetArg(TStrings::USER_IP), TStrings::USER_IP);
    }

    const TString& TUtils::GetCheckedArg(const NCommon::TRequest& request, const TString& arg) {
        const TString& val = request.GetArg(arg);
        CheckArgValue(val, arg);
        return val;
    }

    TString TUtils::GetCheckedArgSession(const NCommon::TRequest& request, const TString& arg) {
        const TString& val = request.GetArg(arg);
        CheckArgValue(val, arg);
        return NAuth::TSessionUtils::RepairCookie(val);
    }

    TString TUtils::GetUserIpArg(const NCommon::TRequest& request) {
        const TString& ip = request.GetArg(TStrings::USER_IP);
        TString addr = NUtils::TIpAddr::Normalize(ip);
        if (addr.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "invalid userip argument: " << InvalidValue(ip);
        }
        return addr;
    }

    const TString& TUtils::GetUserPortArg(const NCommon::TRequest& request) {
        const TString& port = request.GetArg(TStrings::USER_PORT);

        if (port == "0") {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "param '" << TStrings::USER_PORT << "' should contain valid port: "
                << InvalidValue(port);
        }

        if (port) {
            ToIntImpl<ui16>(port, TStrings::USER_PORT);
        }

        ThrowOnLeadingZeroInNumber(port, TStrings::USER_PORT);

        return port;
    }

    bool TUtils::GetBoolArg(const NCommon::TRequest& request, const TString& arg) {
        if (!request.HasArg(arg)) {
            return false;
        }

        const TString& val = request.GetArg(arg);
        if (val.empty()) {
            return true;
        }

        return NUtils::ToBoolean(val);
    }

    const TString& TUtils::GetIntArg(const NCommon::TRequest& request, const TString& arg, bool mandatory) {
        const TString& val = request.GetArg(arg);

        if (mandatory) {
            CheckArgValue(val, arg);
        }

        if (!val.empty()) {
            ToInt(val, arg);
        }

        ThrowOnLeadingZeroInNumber(val, arg);

        return val;
    }

    const TString& TUtils::GetUIntArg(const NCommon::TRequest& request, const TString& arg, bool mandatory) {
        const TString& val = request.GetArg(arg);

        if (mandatory) {
            CheckArgValue(val, arg);
        }

        if (!val.empty()) {
            ToUInt(val, arg);
        }

        ThrowOnLeadingZeroInNumber(val, arg);

        return val;
    }

    std::vector<TString> TUtils::GetNumbersArgChecked(const TStringBuf argValue, const TStringBuf аrgName) {
        std::vector<TString> res = NUtils::NormalizeListValue(argValue, ",");

        for (const TString& num : res) {
            if (!NUtils::DigitsOnly(num)) {
                throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                    << "invalid " << аrgName << " type: " << InvalidValue(num);
            }
        }

        return res;
    }

    void TUtils::CheckFindByPhoneAliasArg(const NCommon::TRequest& request, TDbFetcher& fetcher) {
        if (!request.HasArg(TStrings::FIND_BY_PHONE_ALIAS)) {
            return; // no such argument
        }

        const TString& arg = request.GetArg(TStrings::FIND_BY_PHONE_ALIAS);

        if (arg == TStrings::FORCE_ON) {
            fetcher.SetPhoneAliasMode(TDbFetcher::EPhoneAliasMode::ForceOn);
        } else if (arg == TStrings::FORCE_OFF) {
            fetcher.SetPhoneAliasMode(TDbFetcher::EPhoneAliasMode::ForceOff);
        } else {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Unknown value of " << TStrings::FIND_BY_PHONE_ALIAS << " argument: " << InvalidValue(arg);
        }
    }

    TString TUtils::SanitizePhoneNumber(const TStringBuf number) {
        TString num = GetDigitsOnly(number);

        if (num.empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "Invalid number format: " << InvalidValue(number);
        }

        return num;
    }

    TString TUtils::NormalizePhone(const TStringBuf phone,
                                   const TString& domain,
                                   const TYandexDomains& yandexDomains,
                                   const TString& cntr) {
        TString login = NormalizePhoneByLegacyWay(GetDigitsOnly(phone), domain);

        if (NPhoneNumber::TUtils::IsValidE164("+" + login)) {
            return login;
        }

        // Override country if domain is provided
        // This is required to choose same user during signing in and email delivery
        TString country = cntr;
        if (!domain.empty()) {
            const TYandexDomains::TDomainData* dom = yandexDomains.Find(domain);
            if (dom) {
                country = dom->GetPhoneCountry();
            } else {
                country.clear();
            }
        }

        return country.empty() ? login : NPhoneNumber::TUtils::FormatOnlyDigits(login, country);
    }

    TString TUtils::GetDigitsOnly(const TStringBuf phone) {
        TString ph;
        ph.reserve(phone.size());

        for (const char p : phone) {
            if ('0' <= p && p <= '9') {
                ph.push_back(p);
            }
        }

        return ph;
    }

    TString TUtils::NormalizePhoneByLegacyWay(const TString& ph,
                                              const TStringBuf domain) {
        // try to guess some common phone abbreviations
        const size_t len = ph.size();
        if (len < 8) { // no rules for such short numbers, skip it
            return ph;
        }

        if (len == 10 && ph[0] == '7') {
            // 7[0-9]{9} - short KZ numbers format
            // 7011234567 => 77011234567
            return TString("7").append(ph);
        }
        if (len == 11 && ph[0] == '8' && ph[1] != '0') {
            // 8[1-9][0-9]{9} - russian legacy format
            // 89161234567 => 79161234567
            TString result(ph);
            result[0] = '7';
            return result;
        }
        if (len == 10 && ph[0] == '0' && ph[1] != '0' && domain == "yandex.ua") {
            // 0[1-9][0-9]{8} - ukraine short format
            // 0391234567 => 380391234567
            return TString("38").append(ph);
        }
        if (len == 11 && ph[0] == '0' && ph[1] == '5' &&
            (domain == "yandex.com" || domain == "yandex.com.tr")) {
            // 05[0-9]{9} @yandex.com/com.tr - turkish phones
            // 05312345678 => 905312345678
            return TString("9").append(ph);
        }
        if (domain == "yandex.by") {
            if (len == 9 && ph[0] != '0') {
                // [1-9][0-9]{8}
                // 291234567 => 375291234567
                return TString("375").append(ph);
            }
            if (len == 10 && ph[0] == '0' && ph[1] != '0') {
                // 0[1-9][0-9]{8}
                // 0291234567 => 375291234567
                return TString("375").append(ph, 1);
            }
            if (len == 11 && ph[0] == '8' && ph[1] == '0' && ph[2] != '0') {
                // 80[1-9][0-9]{8}
                // 80291234567 => 375291234567
                return TString("375").append(ph, 2);
            }
        }

        return ph;
    }

    TString TUtils::MaskPhoneNumber(TString&& phone) {
        // replace: +7 8238435435 926 ***-**-88
        ui8 digits = 0;

        for (auto it = phone.rbegin(); it != phone.rend(); ++it) {
            if (NUtils::NotDigit(*it)) {
                continue;
            }

            ++digits;
            if (digits < 3) {
                continue;
            }
            if (digits > 7) {
                break;
            }

            *it = '*';
        }

        return std::move(phone);
    }

    TString TUtils::FixPythonTimestamp(const TString& ts) {
        TString result(ts);
        const auto pos = result.rfind('.');
        if (pos != TString::npos) {
            result.erase(pos);
        }
        return result;
    }

    std::unique_ptr<NDbPool::TResult> TUtils::WaitResult(NDbPool::TNonBlockingHandle& handle, const char* errorMsg) {
        try {
            std::unique_ptr<NDbPool::TResult> result = handle.WaitResult();
            if (!result) {
                throw TBlackboxError(TBlackboxError::EType::Unknown) << "db result is NULL";
            }
            return result;
        } catch (const NDbPool::TException& e) {
            TLog::Debug("BlackBox: %s : %s", errorMsg, e.what());
            throw TDbpoolError(errorMsg, e.what());
        }
    }

    //
    // By 'greeting' here we assume login name we want to show in a page header
    // For new accounts this is the login name user entered at the registration time.
    // For a few older accounts this is the contents of the accounts.login field.
    //
    // For accounts created after dot ('.') in logins was allowed this is the value of
    // "subscription.login where sid=8". For older accounts this shall be accounts.login.
    // Note, that for mail-for-domains subscription.login.8 contains complete e-mail.
    // So our logic is as follows:
    // - replace all '.' in the subscription.login.8 before the '@' sign with '-'
    // - if this "normalized" part equals (ignoring case) to accounts.login, return subscription.login.8
    // - otherwise, return accounts.login
    //
    // NOTE: sid8Login may be empty (account may not have sid=8 at all or may have sid=800)
    //
    TString TUtils::SelectGreeting(const TString& accountLogin, const TString& sid8Login,
                                   bool mismatchExpected) {
        if (sid8Login.empty()) {
            TLog::Debug("selectGreeting() got empty sid8login. Account: %s",
                        accountLogin.c_str());
            return accountLogin;
        }
        TString::const_iterator l = accountLogin.cbegin();
        TString::const_iterator r = sid8Login.cbegin();
        TString::const_iterator lend = accountLogin.cend();
        TString::const_iterator rend = sid8Login.cend();
        for (; l != lend && r != rend && *l != '@' && *r != '@'; ++l, ++r) {
            if (!((*r == '.' && *l == '-') || std::tolower(*r) == std::tolower(*l))) {
                break;
            }
        }
        if ((l == lend && r == rend) || (l != lend && *l == '@' && r != rend && *r == '@')) {
            return sid8Login;
        }

        if (!mismatchExpected) { // log mismatch
            TLog::Debug("selectGreeting() found mismatch in logins. Account: %s. Sid8: %s",
                        accountLogin.c_str(),
                        sid8Login.c_str());
        }
        return accountLogin;
    }

    bool TUtils::IsSyntheticLogin(const TStringBuf login) {
        return (login.StartsWith(TStrings::SOCIAL_LOGIN_PREFIX) ||
                login.StartsWith(TStrings::PHONISH_LOGIN_PREFIX) ||
                login.StartsWith(TStrings::NEOPHONISH_LOGIN_PREFIX) ||
                login.StartsWith(TStrings::YAMBOT_LOGIN_PREFIX) ||
                login.StartsWith(TStrings::KOLONKISH_LOGIN_PREFIX)) &&
               login.find('@') == TStringBuf::npos;
    }

    TString& TUtils::MakeEmail(TString& buf, const TStringBuf login, const TStringBuf domain) {
        buf.clear();
        NUtils::Append(buf, login, "@", domain);
        return buf;
    }

    void TUtils::CheckUserTicketAllowed(const NCommon::TRequest& req) {
        if (TUtils::GetBoolArg(req, TStrings::GET_USER_TICKET) && req.GetHeader(TStrings::X_YA_SERVICE_TICKET).empty()) {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "param '" << TStrings::GET_USER_TICKET << "' is allowed only with header '"
                << TStrings::X_YA_SERVICE_TICKET << "'";
        }
    }

    void TUtils::ThrowOnLeadingZeroInNumber(TStringBuf value, TStringBuf argName) {
        if (value.StartsWith("0") && value != "0") {
            throw TBlackboxError(TBlackboxError::EType::InvalidParams)
                << "param '" << argName << "' should contain number without leading zero: "
                << InvalidValue(value);
        }
    }

    static THashMap<TString, TString> FillTolowerMap() {
        THashMap<TString, TString> result;

        auto addRange = [&](TStringBuf upper, TStringBuf lower, size_t len) {
            Y_ENSURE(upper.size() == lower.size());
            while (upper) {
                result.emplace(upper.NextTokAt(len), lower.NextTokAt(len));
            }
        };
        TString rusUp = "АБВГДЕЁЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯ";
        TString rusLow = "абвгдеёжзийклмнопрстуфхцчшщъыьэюя";

        addRange(rusUp, rusLow, 2);
        addRange(rusLow, rusLow, 2);

        return result;
    }

    static const THashMap<TString, TString> utfTolowerMap = FillTolowerMap();

    bool TUtils::HasLetter(const TStringBuf str, bool allowUTF) {
        if (!allowUTF) {
            return std::find_if(str.begin(), str.end(), [](char c) {
                       return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z');
                   }) != str.end();
        }

        size_t i = 0;

        while (i < str.size()) {
            unsigned char c = str[i];

            if ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z')) {
                return true; // found ascii letter
            }

            if (c < 0x80) {
                ++i;
                continue; // ascii, non letter
            }

            size_t len = UTF8RuneLen(c);
            if (len == 0) {
                ++i;
                continue; // broken rune
            }

            if (utfTolowerMap.contains(str.SubString(i, len))) {
                return true; // supported utf letter
            }
            i += len;
        }

        return false;
    }

    TString TUtils::TolowerUtf(const TStringBuf str) {
        TString result;
        result.reserve(str.size());
        size_t i = 0;
        while (i < str.size()) {
            unsigned char c = str[i];
            if (c < 0x80) {
                result.append(std::tolower(c));
                ++i;
            } else {
                size_t len = UTF8RuneLen(c);
                if (len == 0) {
                    ++i;
                    continue; // broken rune
                }

                const TStringBuf rune = str.SubString(i, len);
                auto it = utfTolowerMap.find(rune);
                result.append(it != utfTolowerMap.end() ? it->second : rune);
                i += len;
            }
        }
        return result;
    }

}
