#include "validator.h"

#include "anonymiser.h"
#include "db_fetcher.h"
#include "db_profile.h"
#include "db_types.h"
#include "dbfields_converter.h"
#include "exception.h"
#include "experiment.h"
#include "utils.h"
#include "ya_domains.h"

#include <passport/infra/libs/cpp/idn/idn.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/xml_utils.h>

#include <util/string/ascii.h>

#include <cctype>

namespace NPassport::NBb {
    static const TString NAROD_DOMAIN("narod.ru");

    bool TValidatorAddress::Compare(const TString& addr) const {
        if (Native) {
            return TValidator::CompareEmail(Address, addr);
        }
        return AsciiEqualsIgnoreCase(Address, addr);
    }

    const TValidatorAddress TValidatorEmaillist::DUMMY;

    TValidatorEmaillist& TValidatorEmaillist::operator=(const TValidatorAddress& addr) {
        Clear();
        List_.push_back(addr);

        bool isCustomDefault = !DefaultEmailAttr_.empty() && addr.Compare(DefaultEmailAttr_);

        if (addr.Default || isCustomDefault) {
            SetDefault(List_.size() - 1, isCustomDefault);
        }

        return *this;
    }

    void TValidatorEmaillist::PushBackImpl(const TValidatorAddress& a, bool isCustomDefault) {
        List_.push_back(a);

        if (HasCustomDefault_) {
            List_.back().Default = false;
            return;
        }

        if (a.Default || isCustomDefault) {
            SetDefault(List_.size() - 1, isCustomDefault);
        }
    }

    void TValidatorEmaillist::AppendList(TValidatorEmaillist&& l) {
        int oldListSize = List_.size();
        std::move(l.List_.begin(), l.List_.end(), std::back_inserter(List_));

        // if the list being appended has default, override previous default
        if (l.Default_ != -1) {
            SetDefault(oldListSize + l.Default_, !DefaultEmailAttr_.empty() && TValidator::CompareEmail(l.GetDefault().Address, DefaultEmailAttr_));
            l.Default_ = -1;
        }
    }

    inline void TValidatorEmaillist::SetDefault(int def, bool isCustomDefault) {
        if (Default_ == def || HasCustomDefault_) {
            return;
        }

        // dont allow out-of-bounds default index
        if (def < 0 || def >= int(List_.size())) {
            return;
        }

        if (Default_ != -1) {
            List_[Default_].Default = false;
        }
        Default_ = def;
        List_[Default_].Default = true;
        HasCustomDefault_ = isCustomDefault;
    }

    void TValidatorEmaillist::AddNativeAddr(const TString& addr,
                                            const TString& ts,
                                            bool isDefault,
                                            bool isVisible) {
        bool isCustomDefault = !DefaultEmailAttr_.empty() &&
                               TValidator::CompareEmail(addr, DefaultEmailAttr_);
        if (isVisible || isCustomDefault) {
            PushBackImpl(TValidatorAddress(addr, ts, true, true, isDefault),
                         isCustomDefault);
        }
    }

    void TValidatorEmaillist::AddExternalAddr(const TString& addr,
                                              const TString& ts,
                                              bool isConfirmed,
                                              bool isDefault) {
        bool isCustomDefault = !DefaultEmailAttr_.empty() &&
                               isConfirmed &&
                               AsciiEqualsIgnoreCase(addr, DefaultEmailAttr_);
        PushBackImpl(TValidatorAddress(addr, ts, isConfirmed, false, isDefault),
                     isCustomDefault);
    }

    // here we check if address is already in the list (up to 'limit' first elements)
    bool TValidatorEmaillist::HasAddr(const TString& addr, size_t limit) const {
        for (size_t i = 0; i < limit; ++i) {
            if (TValidator::CompareEmail(addr, List_[i].Address)) {
                return true;
            }
        }
        return false;
    }

    // Compares two e-mail addresses ignoring differences in '.' and '-'
    // in the login part
    bool TValidator::CompareEmail(const TString& l, const TString& r) {
        TString::const_iterator lit = l.cbegin();
        TString::const_iterator rit = r.cbegin();
        for (bool seenAt = false; lit != l.cend() && rit != r.cend(); ++lit, ++rit) {
            if (!seenAt) {
                if (*lit == '+') { // skip '+tag'
                    while (lit != l.cend() && *lit != '@') {
                        ++lit;
                    }
                }

                if (*rit == '+') { // skip '+tag'
                    while (rit != r.cend() && *rit != '@') {
                        ++rit;
                    }
                }

                if (lit == l.cend() && rit == r.cend()) {
                    return true;
                }
                if (lit == l.cend() || rit == r.cend()) {
                    return false;
                }
            }

            if (std::tolower(*lit) == std::tolower(*rit)) {
                if (*lit == '@') {
                    seenAt = true;
                }
                continue;
            }
            if (seenAt) {
                return false;
            }
            if ((*lit != '.' && *lit != '-') || (*rit != '.' && *rit != '-')) {
                return false;
            }
        }

        return lit == l.cend() && rit == r.cend();
    }

    //
    // Validator
    //
    TValidator::TValidator(TDbFieldsConverter& conv, const TYandexDomains& yaDomains)
        : YandexDomains_(yaDomains)
    {
        TDbFetcher& fetcher = conv.Fetcher();
        RegDateItem_ = conv.Add(TDbFieldsConverter::REGDATE);
        Sid2LoginItem_ = conv.Add(TDbFieldsConverter::LOGIN2);
        Sid8LoginItem_ = conv.Add(TDbFieldsConverter::LOGIN8);
        Sid16LoginItem_ = conv.Add(TDbFieldsConverter::LOGIN16);
        CountryItem_ = fetcher.AddAttr(TAttr::PERSON_COUNTRY);

        // no need to store indices, check all aliases anyway
        fetcher.AddAlias(TAlias::PDD_MASTER_LOGIN);
        fetcher.AddAlias(TAlias::PDD_ALIAS_LOGIN);

        MailAliasItem_ = fetcher.AddAlias(TAlias::MAIL_LOGIN);
        NarodAliasItem_ = fetcher.AddAlias(TAlias::NAROD_MAIL_LOGIN);
        PhoneAliasItem_ = fetcher.AddAlias(TAlias::PHONE_NUMBER);
        AltDomainLoginItem_ = fetcher.AddAlias(TAlias::ALT_DOMAIN_LOGIN);
        LiteAliasItem_ = fetcher.AddAlias(TAlias::LITE_LOGIN);

        DefaultEmailAttr_ = fetcher.AddAttr(TAttr::ACCOUNT_DEFAULT_EMAIL);
        EnableSearchByPhoneAliasItem_ = fetcher.AddAttr(TAttr::ACCOUNT_ENABLE_SEARCH_BY_PHONE_ALIAS);
        HideYandexDomainsEmails_ = fetcher.AddAttr(TAttr::ACCOUNT_HIDE_YANDEX_DOMAINS_EMAILS);

        // fetch extended email attributes too
        fetcher.AddExtendedEmailAttr(TEmailAttr::ADDRESS);
        fetcher.AddExtendedEmailAttr(TEmailAttr::CREATED);
        fetcher.AddExtendedEmailAttr(TEmailAttr::CONFIRMED);
        fetcher.AddExtendedEmailAttr(TEmailAttr::IS_RPOP);
        fetcher.AddExtendedEmailAttr(TEmailAttr::IS_UNSAFE);
        fetcher.AddExtendedEmailAttr(TEmailAttr::IS_SILENT);
    }

    void TValidator::AppendFromEmailAttrs(TValidatorEmaillist& list, const TDbProfile* profile) const {
        TString login = Sid8LoginItem_->Value(profile);

        // for PDD users we have email in login
        size_t atPos = login.find('@');
        if (atPos != TString::npos) {
            login.resize(atPos);
        }

        const TString& accountTs = RegDateItem_->Value(profile);

        // for lite accounts we need to add their email to the list too
        const TString& liteAlias = profile->Get(LiteAliasItem_)->Value;
        if (!liteAlias.empty()) {
            // add lite alias as confirmed and default (lites do not have native emails)
            list.AddExternalAddr(liteAlias, accountTs, true, !list.HasDefault());
        }

        const TDbProfile::TExtendedEntities& extEntities = profile->ExtendedEmailAttrs();

        size_t listSize = list.size();
        list.Reserve(listSize + extEntities.size());

        unsigned anonymizerNumber = 1;
        for (const auto& [entityId, attrs] : extEntities) {
            if (entityId.empty()) {
                // dummy entity for keeping attrs
                continue;
            }

            const TString& address = attrs.at(TEmailAttr::ADDRESS).Value;
            if (address.empty()) {
                // EAV artifact: there are some attributes for entity,
                // but main attr is deleted already
                continue;
            }

            try {
                TString email = PunyEncodeAddress(address);

                // skip empty/invalid
                if (email.empty()) {
                    TLog::Warning() << "Malformed email in db, uid=" << profile->Uid()
                                    << " email '" << attrs.at(TEmailAttr::ADDRESS).Value << "'";
                    continue;
                }

                // skip if has matching phone alias (including various normalization rules)
                if (MatchesPhoneAlias(email, profile)) {
                    continue;
                }

                // skip if already in the list (checking only native elements)
                // we believe that external emails do not have duplicates
                if (list.HasAddr(email, listSize)) {
                    continue;
                }

                const TString& created = attrs.at(TEmailAttr::CREATED).Value;
                const TString& confirmed = attrs.at(TEmailAttr::CONFIRMED).Value;

                bool validated = !confirmed.empty();
                bool isdefault = !list.HasDefault() && validated;

                if (Anonymizer_) {
                    Anonymizer_->MapEmail(email, login, anonymizerNumber);
                }

                TString ts = validated ? confirmed : created;

                if (ts.empty()) {
                    ts = accountTs;
                } else {
                    ts = NUtils::FormatTimestamp(ts);
                }

                list.AddExternalAddr(email, ts, validated, isdefault);

                list.Rbegin()->Unsafe = attrs.at(TEmailAttr::IS_UNSAFE).AsBoolean();
                list.Rbegin()->Rpop = attrs.at(TEmailAttr::IS_RPOP).AsBoolean();
                list.Rbegin()->Silent = attrs.at(TEmailAttr::IS_SILENT).AsBoolean();

            } catch (const std::out_of_range& e) {
                // should never happen since we get only the attrs we asked for
                throw TBlackboxError(TBlackboxError::EType::Unknown) << "Unable to find requested email attribute";
            }
        }
    }

    void TValidator::LeaveOneIfSpecified(TValidatorEmaillist& ret, const TDbProfile* profile) const {
        TValidatorAddress resultAddr;

        if (JustDefault_) {
            if (ret.HasDefault()) {
                resultAddr = ret.GetDefault();
            }
            ret.Clear();
        } else if (!AddrToTest_.empty()) {
            if (MatchesPhoneAlias(AddrToTest_, profile)) {
                const TString& ts = RegDateItem_->Value(profile);
                resultAddr = TValidatorAddress(AddrToTest_, ts, true, true, false);
            } else {
                for (const TValidatorAddress& addr : ret) {
                    if (addr.Compare(AddrToTest_)) {
                        resultAddr = addr;
                        break;
                    }
                }
            }
            ret.Clear();
        }

        if (!resultAddr.Address.empty()) {
            ret = resultAddr;
        }
    }

    void TValidator::AppendDomainAliases(TValidatorEmaillist& list,
                                         const std::vector<TString>& logins,
                                         const TString& domain,
                                         const TString& ts) {
        TString address;

        for (const TString& login : logins) {
            address.assign(login).push_back('@');
            address.append(domain);
            list.AddNativeAddr(address, ts, !list.HasDefault(), true);
        }
    }

    bool TValidator::GetPddMasterLogin(const TDbProfile* profile, TString& login) const {
        const TDbProfile::TAliases& aliases = profile->Aliases();
        TDbProfile::TAliases::const_iterator it = aliases.find(TAlias::PDD_MASTER_LOGIN);
        if (it == aliases.end()) {
            return false;
        }
        login = it->second.Value;
        if (login.empty()) {
            return false;
        }
        TString::size_type pos = login.find('@');
        if (pos == TString::npos) {
            return false;
        }

        if (AsciiEqualsIgnoreCase(login, Sid8LoginItem_->Value(profile))) {
            login.assign(Sid8LoginItem_->Value(profile));
        }
        login.erase(pos);
        return true;
    }

    void TValidator::AppendPddAliases(TValidatorEmaillist& list, const TDbProfile* profile) const {
        // 1. find logins = { PddMasterLogin, PddAliasLogins }
        const TDbProfile::TMultiAlias& aliasesPdd = profile->AliasesPdd();

        // master login + aliases
        std::vector<TString> logins;
        logins.reserve(1 + aliasesPdd.size());

        // first login is PddMasterLogin, it will become default if no other default specified
        TString login;
        if (!GetPddMasterLogin(profile, login)) {
            return;
        }
        logins.push_back(login);

        for (const TString& alias : aliasesPdd) {
            if (alias.empty()) {
                continue;
            }
            TString login(alias);

            TString::size_type pos = login.find('@');
            if (pos != TString::npos) {
                login.erase(pos);
            }
            logins.push_back(login);
        }

        // 2. Multiply logins X domains

        // logins for master pdd domain + pdd domain slaves
        list.Reserve(logins.size() * (1 + profile->PddDomItem().Slaves().size()));
        AppendDomainAliases(list, logins, profile->PddDomain(), RegDateItem_->Value(profile));

        for (const TString& id : profile->PddDomItem().Slaves()) {
            if (const TDomain* d = profile->DomainList().FindById(id)) {
                AppendDomainAliases(list, logins, d->AsciiName(), RegDateItem_->Value(profile));
            }
        }
    }

    void TValidator::AppendYandexAliases(TValidatorEmaillist& list, const TDbProfile* profile) const {
        TValidatorEmaillist phoneAliases(list.CustomDefaultEmail());
        // Prepare list of "native" e-mails based on the login and the sids.
        // We take the "greeting" and combine it with all domains used for Yandex.Mail.
        //
        // sid=16 login always found, either specific or the same as sid2 login so use <greeting>@narod.ru.

        // check if user has custom mail alias (which differs from portal login)
        const TString& mailAlias = profile->Get(MailAliasItem_)->Value;
        const TString& narodAlias = profile->Get(NarodAliasItem_)->Value;

        const TString& mailLogin = !mailAlias.empty() ? mailAlias : Sid8LoginItem_->Value(profile);

        const TString& country = profile->Get(CountryItem_)->Value;
        const TString& altDomainLogin = profile->Get(AltDomainLoginItem_)->Value;
        const TString& ts = RegDateItem_->Value(profile);
        bool bIgnoreCountry = !AddrToTest_.empty() || JustDefault_;
        bool bShowYandexDomains = !profile->Get(HideYandexDomainsEmails_)->AsBoolean();

        TString phoneAlias;
        if (profile->Get(EnableSearchByPhoneAliasItem_)->AsBoolean()) {
            phoneAlias = profile->Get(PhoneAliasItem_)->Value;
        }

        if (Anonymizer_) {
            TAnonymiser::MapPhone(phoneAlias);
        }

        TString buf;
        if (mailLogin.find('@') != TString::npos) {
            // this is probably alt domain login with no plain yandex login
            if (!profile->AltDomId().empty() && !profile->AltDomain().empty()) {
                const TYandexDomains::TDomainData* data = YandexDomains_.Find(profile->AltDomain());
                if (data) {
                    bool isVisible = bIgnoreCountry || data->Visible(country);
                    list.AddNativeAddr(altDomainLogin, ts, true, isVisible);
                    if (!phoneAlias.empty()) {
                        phoneAliases.AddNativeAddr(TUtils::MakeEmail(buf, phoneAlias, data->Name()), ts, false, isVisible);
                    }
                }
            }
        } else { // this user has plain yandex login that should be multiplied to all yandex domains
            if (phoneAlias.empty()) {
                list.Reserve(YandexDomains_.DomainCount());
            } else {
                list.Reserve(2 * YandexDomains_.DomainCount());
                phoneAliases.Reserve(YandexDomains_.DomainCount());
            }

            for (TYandexDomains::TShowIterator it = YandexDomains_.begin(); it != YandexDomains_.end(); ++it) {
                // filter out list by country, but only if not testing one address
                bool isVisible = bIgnoreCountry || it.Visible(country);

                if (*it == NAROD_DOMAIN && !Sid16LoginItem_->Value(profile).empty()) {
                    // check if user has custom narod or mail alias (which differs from portal login)
                    const TString& narodLogin = !narodAlias.empty() ? narodAlias : mailLogin;

                    if (bShowYandexDomains) {
                        list.AddNativeAddr(TUtils::MakeEmail(buf, narodLogin, *it), ts, false, isVisible);
                        if (!phoneAlias.empty()) {
                            phoneAliases.AddNativeAddr(TUtils::MakeEmail(buf, phoneAlias, *it), ts, false, isVisible);
                        }
                    }
                } else {
                    bool bAltDomain = !it.AltDomId().empty();
                    // check if it is really our alternative domain
                    isVisible &= (!bAltDomain || it.AltDomId() == profile->AltDomId());

                    // If we have at least one alternative address and no default address, alternative becomes the default;
                    // otherwise, choose the default based on the user's native country.
                    if (bAltDomain) {
                        list.AddNativeAddr(altDomainLogin, ts, true, isVisible);
                        if (!phoneAlias.empty()) {
                            phoneAliases.AddNativeAddr(TUtils::MakeEmail(buf, phoneAlias, *it), ts, false, isVisible);
                        }
                    } else if (bShowYandexDomains) {
                        bool isDefault = altDomainLogin.empty() && (*it == YandexDomains_.DefaultDomain(country));
                        list.AddNativeAddr(TUtils::MakeEmail(buf, mailLogin, *it), ts, isDefault, isVisible);
                        if (!phoneAlias.empty()) {
                            phoneAliases.AddNativeAddr(TUtils::MakeEmail(buf, phoneAlias, *it), ts, false, isVisible);
                        }
                    }
                }
            }
        }

        // to show phone aliases list after all other emails
        list.AppendList(std::move(phoneAliases));
    }

    // static const TString yandexRfUtfStr ("яндекс.рф");
    // static const TString yandexRfIdnStr ("xn--d1acpjx3f.xn--p1ai");

    TValidatorEmaillist TValidator::MakeList(const TDbProfile* profile) const {
        if (nullptr == profile) {
            return {};
        }

        const TString& defaultEmailAttr = profile->Get(DefaultEmailAttr_)->Value;
        TValidatorEmaillist ret(defaultEmailAttr);

        if (!Sid2LoginItem_->Value(profile).empty()) { // account has mail subscription
            AppendYandexAliases(ret, profile);
            AppendPddAliases(ret, profile);
        }

        size_t nativeCount = ret.size();

        // Now if we need external emails (not justYandex) OR
        // we have custom default email set up (and need to reset default flag correctly)
        // then add all external emails to the list
        if (!JustYandex_ || (!defaultEmailAttr.empty() && !ret.HasCustomDefault())) {
            AppendFromEmailAttrs(ret, profile);
        }

        // we could have added external emails to find custom default email in the complete email list
        // if so, leave only native (or only default) emails in the list
        if (JustYandex_) {
            ret.Resize(nativeCount);
        } else {
            // Shall we return entire list or check whether specified address
            // belongs to it
            LeaveOneIfSpecified(ret, profile);
        }

        return ret;
    }

    bool TValidator::MatchesPhoneAlias(const TStringBuf email, const TDbProfile* profile) const {
        const TString& phoneAlias = profile->Get(PhoneAliasItem_)->Value;
        if (phoneAlias.empty()) { // no phone alias
            return false;
        }

        if (!profile->Get(EnableSearchByPhoneAliasItem_)->AsBoolean()) { // phone alias disabled
            return false;
        }

        size_t atPos = email.find('@');
        if (atPos == TStringBuf::npos) {
            return false;
        }

        TString domain(email.substr(atPos + 1));
        const TYandexDomains::TDomainData* yadom = YandexDomains_.Find(domain);
        if (!yadom || (!yadom->GetDomid().empty() && yadom->GetDomid() != profile->AltDomId())) {
            return false;
        }

        TStringBuf login = email.substr(0, atPos);
        if (TUtils::HasLetter(login)) {
            return false;
        }

        return phoneAlias == TUtils::NormalizePhone(login, domain, YandexDomains_);
    }

    TString TValidator::PunyEncodeAddress(const TString& addrUTF8) {
        auto atPos = addrUTF8.find('@');
        if (atPos == TString::npos) {
            return {}; // don't show user bad email
        }

        auto skipChars = atPos + 1;
        auto it = addrUTF8.cbegin() + skipChars;
        for (; it != addrUTF8.cend(); ++it) {
            if ((unsigned char)*it > 127) {
                break;
            }
        }
        if (it == addrUTF8.cend()) {
            return addrUTF8;
        }

        TString domain = NIdn::UtfToPunycode(addrUTF8.data() + skipChars);
        if (domain.empty()) {
            return {}; // broken or invalid UTF
        }

        TString res;
        res.reserve(addrUTF8.size() << 1);
        res.append(addrUTF8, 0, skipChars).append(domain);
        return res;
    }

}
