#pragma once

#include <util/generic/noncopyable.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>

#include <memory>
#include <vector>

namespace NPassport::NBb {
    class TAnonymiser;
    class TDbFieldsConverter;
    class TDbFieldsResult;
    class TDbProfile;
    class TYandexDomains;

    struct TValidatorAddress {
        TValidatorAddress() = default;

        TValidatorAddress(const TString& addr,
                          const TString& ts,
                          bool conf,
                          bool native,
                          bool isdef = false)
            : Address(addr)
            , Timestamp(ts)
            , Confirmed(conf)
            , Native(native)
            , Default(isdef)
        {
        }

        bool Compare(const TString& addr) const;

        TString Address;
        TString Timestamp;
        bool Confirmed = false;
        bool Native = false;
        bool Default = false;
        bool Unsafe = false;
        bool Rpop = false;
        bool Silent = false;
    };

    class TValidatorEmaillist: TMoveOnly {
    public:
        TValidatorEmaillist()
            : TValidatorEmaillist(TString())
        {
        }
        explicit TValidatorEmaillist(const TString& defaultEmailAttr)
            : Default_(-1)
            , DefaultEmailAttr_(defaultEmailAttr)
            , HasCustomDefault_(false)
        {
        }

        using TList = std::vector<TValidatorAddress>;
        using TConstIterator = TList::const_iterator;
        using TReverseIterator = TList::reverse_iterator;

        TValidatorEmaillist& operator=(const TValidatorAddress& addr);

        size_t size() const {
            return List_.size();
        }
        void AppendList(TValidatorEmaillist&& l);
        TValidatorAddress& operator[](size_t i) {
            return List_[i];
        }
        const TValidatorAddress& operator[](size_t i) const {
            return List_[i];
        }
        void Clear() {
            Default_ = -1;
            List_.clear();
            HasCustomDefault_ = false;
        }
        TConstIterator begin() const {
            return List_.begin();
        }
        TConstIterator end() const {
            return List_.end();
        }
        TReverseIterator Rbegin() {
            return List_.rbegin();
        }

        bool HasDefault() const {
            return Default_ != -1;
        }
        const TValidatorAddress& GetDefault() const {
            return Default_ != -1 ? List_[Default_] : DUMMY;
        }

        void Reserve(size_t size) {
            List_.reserve(size);
        }
        void Resize(size_t size) {
            List_.resize(size);
        }

        void AddNativeAddr(const TString& addr,
                           const TString& ts,
                           bool isDefault,
                           bool isVisible);
        void AddExternalAddr(const TString& addr,
                             const TString& ts,
                             bool isConfirmed,
                             bool isDefault);

        bool HasCustomDefault() const {
            return HasCustomDefault_;
        }
        const TString& CustomDefaultEmail() const {
            return DefaultEmailAttr_;
        }

        bool HasAddr(const TString& addr, size_t limit) const;

    private:
        inline void SetDefault(int def, bool isCustomDefault);
        inline void PushBackImpl(const TValidatorAddress& a, bool isCustomDefault = false);

        int Default_;
        TList List_;
        TString DefaultEmailAttr_;
        bool HasCustomDefault_;

        static const TValidatorAddress DUMMY;
    };

    class TValidator {
    public:
        TValidator(TDbFieldsConverter& conv, const TYandexDomains& yaDomains);

        TValidatorEmaillist MakeList(const TDbProfile* profile) const;

        void SetAddrToTest(const TString& a) {
            AddrToTest_ = a;
        }
        void SetYandexScope() {
            JustYandex_ = true;
        }
        void SetJustDefaultScope() {
            JustDefault_ = true;
        }
        void SetAnonymiser(const TAnonymiser* anonymizer) {
            Anonymizer_ = anonymizer;
        }

        // Compares two e-mail addresses ignoring differences in '.' and '-'
        // in the login part
        static bool CompareEmail(const TString& l, const TString& r);

    private:
        void LeaveOneIfSpecified(TValidatorEmaillist& ret, const TDbProfile* profile) const;

        static void AppendDomainAliases(TValidatorEmaillist& list,
                                        const std::vector<TString>& logins,
                                        const TString& domain,
                                        const TString& ts);
        void AppendPddAliases(TValidatorEmaillist& list, const TDbProfile* profile) const;
        void AppendYandexAliases(TValidatorEmaillist& list, const TDbProfile* profile) const;
        void AppendFromEmailAttrs(TValidatorEmaillist& list, const TDbProfile* profile) const;

        bool GetPddMasterLogin(const TDbProfile* profile, TString& login) const;
        bool MatchesPhoneAlias(const TStringBuf email, const TDbProfile* profile) const;

        static TString PunyEncodeAddress(const TString& addrUTF8);

    private:
        TString AddrToTest_;
        bool JustYandex_ = false;
        bool JustDefault_ = false;

        const TDbFieldsResult* Sid2LoginItem_;
        const TDbFieldsResult* RegDateItem_;
        const TDbFieldsResult* Sid8LoginItem_;
        const TDbFieldsResult* Sid16LoginItem_;

        // user aliases
        int MailAliasItem_;
        int NarodAliasItem_;
        int PhoneAliasItem_;
        int AltDomainLoginItem_;
        int LiteAliasItem_;

        // user attributes
        int CountryItem_;
        int DefaultEmailAttr_;
        int EnableSearchByPhoneAliasItem_;
        int HideYandexDomainsEmails_;

        const TAnonymiser* Anonymizer_ = nullptr;

        const TYandexDomains& YandexDomains_;
    };

}
