#include "ya_domains.h"

#include "strings.h"

#include <passport/infra/libs/cpp/idn/idn.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

#include <util/generic/yexception.h>

namespace NPassport::NBb {
    TYandexDomains::TDomainData::TDomainData(const TString& name, bool anyCountry)
        : Name_(name)
        , BAnyCountry_(anyCountry)
    {
    }

    const TString& TYandexDomains::TDomainData::Name() const {
        return Name_;
    }

    void TYandexDomains::TDomainData::Add(const TString& country) {
        Countries_.insert(country);
    }

    void TYandexDomains::TDomainData::SetAny(bool bAny) {
        BAnyCountry_ = bAny;
    }

    void TYandexDomains::TDomainData::SetSid(const TString& sid) {
        Sid_ = sid;
    }

    const TString& TYandexDomains::TDomainData::GetSid() const {
        return Sid_;
    }

    void TYandexDomains::TDomainData::SetDomid(const TString& domid) {
        Domid_ = domid;
    }

    const TString& TYandexDomains::TDomainData::GetDomid() const {
        return Domid_;
    }

    void TYandexDomains::TDomainData::SetPhoneCountry(const TString& country) {
        PhoneCountry_ = country;
    }

    const TString& TYandexDomains::TDomainData::GetPhoneCountry() const {
        return PhoneCountry_;
    }

    bool TYandexDomains::TDomainData::Visible(
        const TString& country) const {
        // If this user's country is not among allowed for this domain, skip the domain
        return BAnyCountry_ || Countries_.find(country) != Countries_.end();
    }

    TYandexDomains::TShowIterator::TShowIterator(TYandexDomains::TContIt it)
        : It_(it)
    {
    }

    const TString& TYandexDomains::TShowIterator::operator*() const {
        return It_->second.Name();
    }

    const TString* TYandexDomains::TShowIterator::operator->() const {
        return &It_->second.Name();
    }

    TYandexDomains::TShowIterator& TYandexDomains::TShowIterator::operator++() {
        ++It_;
        return *this;
    }

    TYandexDomains::TShowIterator TYandexDomains::TShowIterator::operator++(int) {
        TShowIterator temp = *this;
        ++It_;
        return temp;
    }

    bool TYandexDomains::TShowIterator::operator==(const TShowIterator& other) const {
        return other.It_ == It_;
    }

    bool TYandexDomains::TShowIterator::operator!=(const TShowIterator& other) const {
        return other.It_ != It_;
    }

    bool TYandexDomains::TShowIterator::Visible(
        const TString& country) const {
        return It_->second.Visible(country);
    }

    const TString& TYandexDomains::TShowIterator::AltDomId() const {
        return It_->second.GetDomid();
    }

    TYandexDomains::TYandexDomains() = default;

    const TString& TYandexDomains::Insert(const TString& orig) {
        TString converted = NIdn::UtfToPunycode(orig);
        Y_ENSURE(!converted.empty(), "Incorrect yandex domain in config: '" << orig << "'");
        NUtils::Tolower(converted);
        Domains_.insert(TCont::value_type(converted, TDomainData(orig, true)));
        return orig;
    }

    const TString& TYandexDomains::Insert(const TDomainData& d) {
        TString converted = NIdn::UtfToPunycode(d.Name());
        Y_ENSURE(!converted.empty(), "Incorrect yandex domain in config: '" << d.Name() << "'");
        NUtils::Tolower(converted);
        Domains_.insert(TCont::value_type(converted, d));
        if (!d.GetDomid().empty()) {
            AltDomainsMap_.insert(TAltDomainsMapType::value_type(d.GetDomid(), d.Name()));
        }
        return d.Name();
    }

    const TYandexDomains::TDomainData* TYandexDomains::Find(const TString& s) const {
        try {
            TString converted = NIdn::UtfToPunycode(s);
            if (!converted) {
                return nullptr;
            }
            NUtils::Tolower(converted);
            TContIt it = Domains_.find(converted);
            return it != Domains_.end() ? &it->second : nullptr;
        } catch (const std::exception& e) {
            TLog::Debug("Couldn't look up domain in the native domains list: %s", e.what());
            return nullptr;
        }
    }

    TYandexDomains::TShowIterator TYandexDomains::begin() const {
        return TShowIterator(Domains_.begin());
    }

    TYandexDomains::TShowIterator TYandexDomains::end() const {
        return TShowIterator(Domains_.end());
    }

    void TYandexDomains::SetOtherCountryDefault(const TString& otherCountry) {
        OtherCountryDefault_ = otherCountry;
    }

    void TYandexDomains::AddDefault(const TString& country, const TString& domain) {
        DefaultMap_.insert(TDefaultMapType::value_type(country, domain));
    }

    const TString& TYandexDomains::DefaultDomain(const TString& country) const {
        TDefaultMapType::const_iterator p = DefaultMap_.find(country);

        if (p != DefaultMap_.end()) {
            return p->second;
        }

        return OtherCountryDefault_;
    }

    const TString& TYandexDomains::AltDomain(const TString& id) const {
        TAltDomainsMapType::const_iterator p = AltDomainsMap_.find(id);

        if (p != AltDomainsMap_.end()) {
            return p->second;
        }

        return TStrings::EMPTY;
    }

    size_t TYandexDomains::DomainCount() const {
        return Domains_.size();
    }
}
