#include "config.h"

#include "scopes.h"
#include "token_info.h"

#include <passport/infra/daemons/blackbox/src/loggers/tskvlog.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>

#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/string/cast.h>

// uncomment for debugging
//#define LOG_QUERIES

namespace NPassport::NBb {
    const TString TOAuthStatboxMsg::ERROR = "error";

    const TString TOAuthStatboxMsg::CLIENT_BLOCKED = "client.blocked";
    const TString TOAuthStatboxMsg::CLIENT_GLOGOUT = "client.glogout";
    const TString TOAuthStatboxMsg::CLIENT_NOT_FOUND = "client.not_found";
    const TString TOAuthStatboxMsg::SCOPES_EMPTY = "scopes.empty";
    const TString TOAuthStatboxMsg::SCOPES_WRONG = "scopes.wrong";
    const TString TOAuthStatboxMsg::TOKEN_EXPIRED = "token.expired";
    const TString TOAuthStatboxMsg::USER_DISABLED = "user.disabled";
    const TString TOAuthStatboxMsg::USER_GLOGOUT = "user.glogout";
    const TString TOAuthStatboxMsg::USER_NOT_FOUND = "user.not_found";
    const TString TOAuthStatboxMsg::USER_PWD_CREATE = "user.password_change_required";
    const TString TOAuthStatboxMsg::USER_PWD_CHANGE = "user.password_create_required";
    const TString TOAuthStatboxMsg::USER_PWD_EXPIRED = "user.password_expired";
    const TString TOAuthStatboxMsg::USER_REVOKED_TOKENS = "user.revoked_tokens";
    const TString TOAuthStatboxMsg::USER_IS_CHILD = "user.child";
    const TString TOAuthStatboxMsg::USER_FEDERAL = "user.federal";
    const TString TOAuthStatboxMsg::USER_ROBOT = "user.robot";

    TOAuthConfig::TOAuthConfig(TTskvLog& oauthLog)
        : OauthLogger_(oauthLog)
    {
    }

    TOAuthConfig::~TOAuthConfig() = default;

    void TOAuthConfig::Init(const NXml::TConfig& config,
                            const TString& xpathPrefix,
                            std::shared_ptr<NDbPool::TDbPoolCtx> ctx) {
        if (!config.Contains(xpathPrefix + "/oauth_db_conf/central")) {
            throw yexception() << "BlackBox: OAuth pool not configured";
        }

        Y_ENSURE(TryFromString(TStringBuf(config.AsString(xpathPrefix + "/oauth_token_refresh_ratio", "0.5")),
                               TokenRefreshRatio_),
                 "oauth_token_refresh_ratio has bad float");

        // init oauth central
        OauthCentral_ = ctx->CreateDb(config,
                                      xpathPrefix + "/oauth_db_conf/central",
                                      xpathPrefix + "/oauth_centraldb_settings");
        OauthCentral_->TryPing();

        // init oauth shards
        OauthShards_ = std::make_unique<TShardsMap>();
        OauthShards_->Init(xpathPrefix, "oauth_", config, ctx);

        // init scope loader
        const TString path = config.AsString(xpathPrefix + "/oauth_scopes");
        const ui32 period = config.AsInt(xpathPrefix + "/oauth_scopes_period", 5);
        Scopes_ = std::make_unique<TOAuthScopesConfig>(path, period);

        AvatarsUrlPrefix_ = config.AsString(xpathPrefix + "/avatars_url");

        TLog::Info("BlackBox: MySQL OAuth configured: centraldb + %d shards",
                   OauthShards_->GetShardCount());
    }

    bool TOAuthConfig::DbOk(TString& msg) const {
        bool centralOk = OauthCentral_->IsOk(&msg);

        TString tmpMsg;
        bool shardsOk = OauthShards_->ShardsOk(tmpMsg);
        NUtils::AddMessage(msg, tmpMsg);

        return centralOk && shardsOk;
    }

    void TOAuthConfig::LogOAuth(const TOAuthTokenInfo& info,
                                const TString& userip,
                                const TString& status,
                                const TString& reason,
                                const TString& consumer,
                                const TString& consumerIp) const {
        OauthLogger_.LogOAuth(
            info.TokenId,
            info.ByAlias,
            status,
            info.Uid,
            info.GetClientAttr(TOAuthClientAttr::DISPLAY_ID),
            info.ScopesComma,
            info.CreateTime,
            info.IssueTime,
            info.ExpireTime,
            userip,
            info.DeviceId,
            info.GetTokenAttr(TOAuthTokenAttr::DEVICE_NAME),
            info.IsRefreshRequired,
            reason,
            consumer,
            consumerIp);
    }

    void TOAuthConfig::LogOAuthErrorByAlias(const TString& uid,
                                            const TString& reason,
                                            const TString& userip,
                                            const TString& consumer,
                                            const TString& consumerIp) const {
        OauthLogger_.LogOAuth(TStrings::EMPTY, true, TOAuthStatboxMsg::ERROR, uid, TStrings::EMPTY, TStrings::EMPTY, TStrings::EMPTY, TStrings::EMPTY, TStrings::EMPTY, userip, TStrings::EMPTY, TStrings::EMPTY, false, reason, consumer, consumerIp);
    }

}
