#include "error.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    const TString TOAuthError::OK = "OK";
    const TString TOAuthError::ACCOUNT_NOT_FOUND = "account not found";
    const TString TOAuthError::ACCOUNT_DISABLED = "account is disabled";
    const TString TOAuthError::EXPIRED_PASSWORD = "account has an expired password";
    const TString TOAuthError::GLOGOUT = "account has been globally logged out";
    const TString TOAuthError::REVOKED = "accounts tokens has been revoked";
    const TString TOAuthError::PASSWORD_CHANGE_REQUIRED = "password change required";
    const TString TOAuthError::PASSWORD_CREATE_REQUIRED = "password create required";
    const TString TOAuthError::EXPIRED_TOKEN = "expired_token";
    const TString TOAuthError::CLIENT_NOT_FOUND = "client_not_found";
    const TString TOAuthError::CLIENT_BLOCKED = "blocked_client";
    const TString TOAuthError::CHILD_ACCOUNT = "child account is not allowed";
    const TString TOAuthError::FEDERAL_ACCOUNT = "federal account is not allowed";
    const TString TOAuthError::EXTERNAL_ROBOT = "robot account from external IP is not allowed";

    TOAuthError::TOAuthError(EError err)
        : Error_(err)
    {
    }

    const TString& TOAuthError::Msg() const {
        switch (Error_) {
            case Ok:
                return OK;
            case AccountNotFound:
                return ACCOUNT_NOT_FOUND;
            case AccountDisabled:
                return ACCOUNT_DISABLED;
            case ExpiredPassword:
                return EXPIRED_PASSWORD;
            case GLogout:
                return GLOGOUT;
            case Revoked:
                return REVOKED;
            case PasswordChangeRequired:
                return PASSWORD_CHANGE_REQUIRED;
            case PasswordCreateRequired:
                return PASSWORD_CREATE_REQUIRED;
            case TokenExpired:
                return EXPIRED_TOKEN;
            case ClientNotFound:
                return CLIENT_NOT_FOUND;
            case ClientBlocked:
                return CLIENT_BLOCKED;
            case ChildAccount:
                return CHILD_ACCOUNT;
            case FederalAccount:
                return FEDERAL_ACCOUNT;
            case ExternalRobot:
                return EXTERNAL_ROBOT;
            case WrongScope:
            case WrongEnvironment:
            case OAuthReject:
                return Msg_;
        }
    }

    TOAuthError::EError TOAuthError::Error() const {
        return Error_;
    }

    namespace {
        TOAuthError::EError ConvertTokenStatus(NAuth::TOAuthToken::EStatus st) {
            switch (st) {
                case NAuth::TOAuthToken::VALID:
                    return TOAuthError::Ok;
                case NAuth::TOAuthToken::EXPIRED:
                    return TOAuthError::TokenExpired;
                case NAuth::TOAuthToken::SIGN_BROKEN:
                    return TOAuthError::OAuthReject;
                case NAuth::TOAuthToken::GLOGOUTED:
                    return TOAuthError::GLogout;
                case NAuth::TOAuthToken::NOT_INITED:
                    return TOAuthError::OAuthReject;
            }
        }
    }

    void TOAuthError::SetMsg(TString&& msg, TOAuthError::EError st) {
        Msg_ = std::move(msg);
        Error_ = st;
    }

    void TOAuthError::SetMsg(TString&& msg, NAuth::TOAuthToken::EStatus st) {
        Msg_ = std::move(msg);
        Error_ = ConvertTokenStatus(st);
    }

    void TOAuthError::SetError(TOAuthError::EError st) {
        Error_ = st;
    }

    void TOAuthError::SetError(NAuth::TOAuthToken::EStatus st) {
        Error_ = ConvertTokenStatus(st);
    }

    TString TOAuthError::AuthLogErrorComment(const TString& tokenId) const {
        TString result = NUtils::CreateStrExt(50, "tokid=", tokenId, ";err=");

        switch (Error_) {
            case Ok:
                result.append("ok");
                break;
            case WrongScope:
                result.append("wrong_scope");
                break;
            case AccountNotFound:
                result.append("account_not_found");
                break;
            case AccountDisabled:
                result.append("account_disabled");
                break;
            case ExpiredPassword:
                result.append("expired_password");
                break;
            case GLogout:
                result.append("glogout");
                break;
            case Revoked:
                result.append("revoked");
                break;
            case PasswordChangeRequired:
                result.append("password_change_req");
                break;
            case OAuthReject:
                result.append("oauth_reject");
                break;
            case TokenExpired:
                result.append("expired_token");
                break;
            case ClientNotFound:
                result.append("client_not_found");
                break;
            case ClientBlocked:
                result.append("blocked_client");
                break;
            case ChildAccount:
                result.append("account_is_child");
                break;
            case FederalAccount:
                result.append("account_is_federal");
                break;
            case ExternalRobot:
                result.append("external_robot");
                break;
            case WrongEnvironment:
                result.append("wrong_environment");
                break;
            case PasswordCreateRequired:
                break;
        }

        return result;
    }

    TOAuthStatus TOAuthError::ConvertToStatus() const {
        switch (Error_) {
            case Ok:
                return TOAuthStatus::Valid;
            case AccountDisabled:
                return TOAuthStatus::Disabled;
            default:
                return TOAuthStatus::Invalid;
        }
    }
}
