#include "fetcher.h"

#include "config.h"
#include "error.h"
#include "scopes.h"
#include "token_embedded_info.h"
#include "token_info.h"

#include <passport/infra/daemons/blackbox/src/loggers/tskvlog.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>
#include <passport/infra/daemons/blackbox/src/misc/utils.h>

#include <passport/infra/libs/cpp/auth_core/oauth_token.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_stats.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/dbpool/value.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>
#include <passport/infra/libs/cpp/xml/config.h>

// uncomment for debugging
//#define LOG_QUERIES

namespace NPassport::NBb {
    // default token and clients attributes to fetch from db
    const TOAuthBaseFetcher::TAttrSet TOAuthBaseFetcher::DEFAULT_CLIENT_ATTRS({"7", "8", "12", "13", "15", "20", "26", "27", "29"});

    const TString TOAuthBaseFetcher::DEFAULT_CLIENT_ATTRS_STR = "7,8,12,13,15,20,26,27,29";

    TOAuthBaseFetcher::TOAuthBaseFetcher(const TOAuthConfig& config)
        : Config_(config)
    {
    }

    void TOAuthBaseFetcher::AddClientAttr(const TString& attr) {
        // add attribute only if it is not synthetic and not fetched by default
        if (DEFAULT_CLIENT_ATTRS.find(attr) != DEFAULT_CLIENT_ATTRS.end() || attr == TStrings::ZERO || TOAuthClientAttr::IsSynthetic(attr)) {
            return;
        }

        AddClientAttrs_.insert(attr);
    }

    void TOAuthBaseFetcher::AddAllClientAttrs() {
        AllClientAttrs_ = true;
    }

    void TOAuthBaseFetcher::ProcessTokenAttribute(const NDbPool::TRow& row, TOAuthTokenInfo& info) const {
        if (info.TokenId.empty()) {
            TString id = row[0].AsString();
            info.TokenId = id;
            info.SetTokenAttr(TOAuthTokenAttr::TOKEN_ID, std::move(id));
        }

        const int type = row[1].AsInt();
        TString value = row[2].AsString();

        const int SCOPE_IDS_ = 2;
        const int UID_ = 3;
        const int DEVICE_ID = 5;
        const int EXPIRES = 7;
        const int IS_REFRESHABLE = 8;
        const int CREATED = 9;
        const int ISSUED = 10;

        const TString& strtype = row[1].AsString();
        if (strtype == TOAuthTokenAttr::UID || strtype == TOAuthTokenAttr::TOKEN_ID || strtype == TOAuthTokenAttr::ALIAS) {
            TExperiment::Get().RunMainAttrsCheck();
        }
        TExperiment::Get().RunFullAttrsCheck();

        switch (type) {
            case SCOPE_IDS_:
                info.SetScopes(Config_.Scopes_->GetScopesFromNums(value));
                break;
            case UID_:
                if (value != TStrings::ZERO) {
                    info.Uid = value;
                }
                break;
            case DEVICE_ID:
                if (value != "-") {
                    info.DeviceId = value;
                }
                break;
            case IS_REFRESHABLE:
                info.IsTtlRefreshable = NUtils::ToBoolean(value);
                break;
            case EXPIRES:
                info.SetExpireTs(TUtils::FixPythonTimestamp(value));
                break;
            case CREATED:
                info.SetCreateTs(TUtils::FixPythonTimestamp(value));
                break;
            case ISSUED:
                info.SetIssueTs(TUtils::FixPythonTimestamp(value));
                break;
        }

        // don't forget to store all values in raw attributes
        info.SetTokenAttr(row[1].AsString(), std::move(value));
    }

    void TOAuthBaseFetcher::PostProcessTokenAttributes(TOAuthTokenInfo& info) const {
        info.NullExpire = info.ExpireTime.empty();
        info.SetRefreshRequired(Config_.TokenRefreshRatio_);
    }

    bool TOAuthBaseFetcher::ProcessClientAttribute(const NDbPool::TRow& row,
                                                   TOAuthTokenInfo& info,
                                                   TOAuthError& error,
                                                   time_t now) const {
        const int type = row[0].AsInt();
        TString value = row[1].AsString();

        const int IS_BLOCKED = 7;
        const int GLOGOUT_TIME = 12;
        const int CREATED_TIME = 13;
        const int IS_YANDEX = 26;
        const int ICON_ID = 27;
        const int DELETE_TIME = 29;

        bool res = true;

        switch (type) {
            case IS_BLOCKED:
                if (NUtils::ToBoolean(value)) {
                    TLog::Debug("OAuthFetcher: token client with id %s blocked",
                                info.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID).c_str());
                    LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::CLIENT_BLOCKED);
                    error.SetError(TOAuthError::ClientBlocked);
                    res = false;
                }
                break;
            case GLOGOUT_TIME:
                if (TUtils::ToTime(TUtils::FixPythonTimestamp(value)) >= info.IssueTimeTs) {
                    TLog::Debug("OAuthFetcher: token client with id %s glogouted",
                                info.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID).c_str());
                    LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::CLIENT_GLOGOUT);
                    error.SetError(TOAuthError::TokenExpired);
                    res = false;
                }
                break;
            case CREATED_TIME:
                info.ClientCreatetime = NUtils::FormatTimestamp(TUtils::FixPythonTimestamp(value));
                break;
            case IS_YANDEX:
                info.ClientIsYandex = NUtils::ToBoolean(value);
                break;
            case ICON_ID:
                info.ClientIcon = NUtils::CreateStr(Config_.AvatarsUrlPrefix_, value, "/normal");
                break;
            case DELETE_TIME:
                if (!value.empty() && TUtils::ToTime(TUtils::FixPythonTimestamp(value)) < now) {
                    TLog::Debug("OAuthFetcher: token client with id %s deleted",
                                info.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID).c_str());
                    LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::CLIENT_NOT_FOUND);
                    error.SetError(TOAuthError::ClientNotFound);
                    res = false;
                }
                break;
        }

        info.SetClientAttr(row[0].AsString(), std::move(value));

        return res;
    }

    void TOAuthBaseFetcher::PostProcessClientAttributes(TOAuthTokenInfo& info) {
        info.SetClientAttr(TOAuthClientAttr::CLIENT_ID,
                           TString(info.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID)));
    }

    bool TOAuthBaseFetcher::CheckTokenInfo(const TOAuthTokenInfo& info, TOAuthError& error) const {
        if (info.GetScopesKeywordList().empty()) {
            TLog::Warning("OAuthFetcher: token has empty scopes. tokid: %s", info.TokenId.c_str());
            LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::SCOPES_EMPTY);
            // TODO: replace with WrongScope
            error.SetError(TOAuthError::TokenExpired);
            return false;
        }

        if (!info.HasTokenAttr(TOAuthTokenAttr::CLIENT_ID)) {
            TLog::Debug("OAuthFetcher: token has empty client_id. tokid: %s", info.TokenId.c_str());
            LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::CLIENT_NOT_FOUND);
            // TODO: replace with ClientNotFound
            error.SetError(TOAuthError::TokenExpired);
            return false;
        }

        if (!info.NullExpire && info.ExpireTimeTs <= time(nullptr)) {
            TLog::Debug("OAuthFetcher: xtoken expired on %s", info.ExpireTime.c_str());
            LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::TOKEN_EXPIRED);
            error.SetError(TOAuthError::TokenExpired);
            return false;
        }

        return true;
    }

    TOAuthSingleFetcher::TOAuthSingleFetcher(const TOAuthConfig& config, const TString& userip, const TString& consumer, const TString& consumerIp)
        : TOAuthBaseFetcher(config)
        , UserIp_(userip)
        , Consumer_(consumer)
        , ConsumerIp_(consumerIp)
    {
    }

    bool TOAuthSingleFetcher::SendNonBlockingRequestByAccess(const TString& token, unsigned shard, const TString& clientId, bool isXtoken) {
        // shard
        if (!token.empty()) {
            NDbPool::TDbPool* shardDb = Config_.OauthShards_->GetPoolUnsafe(shard);
            if (!shardDb) {
                return false;
            }

            ShardSqlh_ = std::make_unique<NDbPool::TNonBlockingHandle>(*shardDb);
            TString query = isXtoken ? BuildXTokenQuery(token) : BuildOAuthShardQuery(ShardSqlh_->EscapeQueryParam(token), TStrings::EMPTY);
#ifdef LOG_QUERIES
            TLog::Error("QUERY (oauth shard nonblocking): '%s'", query.c_str());
#endif
            try {
                ShardSqlh_->SendQuery(query);
            } catch (const NDbPool::TException& e) {
                TLog::Debug("BlackBox: dbpool exception querying oauth shard '%s' : '%s'", query.c_str(), e.what());
                throw TDbpoolError("dbpool exception fetching from oauth shard", e.what());
            }
        }

        // central
        if (!clientId.empty()) {
            TString query = BuildOAuthCentralQuery(clientId);
#ifdef LOG_QUERIES
            TLog::Error("QUERY (oauth central nonblocking): '%s'", query.c_str());
#endif
            CentralSqlh_ = std::make_unique<NDbPool::TNonBlockingHandle>(*Config_.OauthCentral_);
            try {
                CentralSqlh_->SendQuery(query);
            } catch (const NDbPool::TException& e) {
                TLog::Debug("BlackBox: dbpool exception querying oauth central '%s' : '%s'", query.c_str(), e.what());
                throw TDbpoolError("dbpool exception fetching from oauth central", e.what());
            }
        }

        return true;
    }

    std::unique_ptr<TOAuthTokenInfo> TOAuthSingleFetcher::CheckTokenByAlias(const TString& token, const TString& uid) {
        if (!IsValidSyntax(token)) {
            return {};
        }

        std::unique_ptr<NDbPool::TResult> result = FetchTokenAttributesByAlias(token, uid);
        if (!result) {
            TLog::Debug("OAuthFetcher: token %s not found in db (by alias)", token.c_str());
            return {};
        }

        std::unique_ptr<TOAuthTokenInfo> tokenInfo = std::make_unique<TOAuthTokenInfo>(true);
        FillTokenAttributes(*result, *tokenInfo);

        // check expire, etc
        TOAuthError error;
        if (!CheckTokenInfo(*tokenInfo, error)) {
            return {};
        }

        if (!FillClientAttributes(*tokenInfo, error)) {
            return {};
        }

        return tokenInfo;
    }

    // if handles != nullptr we complete non-blocking request,
    // if no handles stored, do the blocking request inside
    std::unique_ptr<TOAuthTokenInfo>
    TOAuthSingleFetcher::CheckTokenByAccess(const TOAuthTokenEmbeddedInfo& embeddedInfo, const TString& token, TOAuthError& error) {
        if (!IsValidSyntax(token)) {
            error.SetError(TOAuthError::TokenExpired);
            return {};
        }

        std::unique_ptr<TOAuthTokenInfo> tokenInfo = std::make_unique<TOAuthTokenInfo>(false);

        // if token has embedded info, we get shard from token, else we'll have shard 0 by default
        std::unique_ptr<NDbPool::TResult> result = GetShardResult(token, TStrings::EMPTY, embeddedInfo.Shard());
        if (result->size() == 0) {
            TLog::Debug() << "OAuthFetcher: token " << TStringBuf(token).Head(token.size() / 2)
                          << "... not found in db";
            error.SetError(TOAuthError::TokenExpired);
            return {};
        }

        FillTokenAttributes(*result, *tokenInfo);

        // check expire and embeddedInfo match if present
        if (!CheckTokenInfo(*tokenInfo, error)) {
            return {};
        }
        if (!CheckEmbeddedInfo(*tokenInfo, embeddedInfo)) {
            error.SetError(TOAuthError::TokenExpired);
            return {};
        }

        if (!FillClientAttributes(*tokenInfo, error)) {
            return {};
        }

        return tokenInfo;
    }

    std::unique_ptr<TOAuthTokenInfo>
    TOAuthSingleFetcher::CheckTokenByAccess(const NAuth::TOAuthToken& token, const TString& strtoken, TOAuthError& error) {
        // fill token info with data from pOAuthToken
        std::unique_ptr<TOAuthTokenInfo> tokenInfo = std::make_unique<TOAuthTokenInfo>(token, strtoken, *Config_.Scopes_);

        if (ShardSqlh_) { // we made an xtoken query before
            std::unique_ptr<TOAuthTokenInfo> xtokenInfo = std::make_unique<TOAuthTokenInfo>(!token.Uid().empty());

            std::unique_ptr<NDbPool::TResult> result =
                TUtils::WaitResult(*ShardSqlh_, "dbpool exception in checkTokenByAccess for oauth shard (xtoken).");
            if (result->size() == 0) {
                TLog::Debug("OAuthFetcher: xtoken with id %s not found in db", token.XtokenId().c_str());
                error.SetError(TOAuthError::TokenExpired);
                return {}; // xtoken not found in db - consider this token expired
            }

            FillTokenAttributes(*result, *xtokenInfo);

            if (xtokenInfo->Uid.empty()) {
                TLog::Warning("OAuthFetcher: xtoken with empty uid! xtoken_id=%s.",
                              xtokenInfo->TokenId.c_str());
                error.SetError(TOAuthError::TokenExpired);
                return {}; // xtoken has empty uid
            }

            if (token.Uid() != xtokenInfo->Uid) {
                TLog::Warning("OAuthFetcher: xtoken uid '%s' does not match token uid '%s'.",
                              xtokenInfo->Uid.c_str(),
                              token.Uid().c_str());
                error.SetError(TOAuthError::TokenExpired);
                return {}; // xtoken has different uid
            }

            if (!xtokenInfo->NullExpire && xtokenInfo->ExpireTimeTs <= time(nullptr)) {
                TLog::Debug("OAuthFetcher: xtoken expired on %s",
                            xtokenInfo->ExpireTime.c_str());
                error.SetError(TOAuthError::TokenExpired);
                return {}; // xtoken expired
            }

            // remember xtoken issue time and device name
            tokenInfo->XtokenIssueTimeTs = xtokenInfo->IssueTimeTs;
            TString deviceName = xtokenInfo->GetTokenAttr(TOAuthTokenAttr::DEVICE_NAME);
            tokenInfo->SetTokenAttr(TOAuthTokenAttr::DEVICE_NAME, std::move(deviceName));
        }

        if (!FillClientAttributes(*tokenInfo, error)) {
            return {};
        }

        // for stateless token without scopes - take scopes from client attributes
        if (token.Scopes().empty()) {
            TString clientScopes = tokenInfo->GetClientAttr(TOAuthClientAttr::SCOPE_IDS);
            if (clientScopes.empty()) {
                TLog::Warning() << "OAuthFetcher: token with no scopes, token_id=" << token.TokenId()
                                << ", client_id=" << tokenInfo->GetClientAttr(TOAuthClientAttr::CLIENT_ID);
                error.SetError(TOAuthError::TokenExpired);
                return {}; // illegal to have token with no scopes
            }

            tokenInfo->SetScopes(Config_.Scopes_->GetScopesFromNums(clientScopes));
            tokenInfo->SetTokenAttr(TOAuthTokenAttr::SCOPE_IDS, std::move(clientScopes));
        }

        return tokenInfo;
    }

    bool TOAuthSingleFetcher::IsValidSyntax(const TString& token) {
        for (const char c : token) {
            if (c < 33 || 126 < c) {
                return false;
            }
        };
        return true;
    }

    void TOAuthSingleFetcher::FillTokenAttributes(NDbPool::TResult& result, TOAuthTokenInfo& info) const {
        for (const NDbPool::TRow& row : result.Table()) {
            ProcessTokenAttribute(row, info);
        }

        PostProcessTokenAttributes(info);
    }

    std::unique_ptr<NDbPool::TResult>
    TOAuthSingleFetcher::GetShardResult(const TString& token, const TString& uid, unsigned shard) const {
        // if it was a non-blocking request, wait for result, it may be ready
        if (ShardSqlh_) {
            return TUtils::WaitResult(*ShardSqlh_, "dbpool exception in waitResult for oauth shard.");
        }

        // else, do the synchronous request
        NDbPool::TDbPool* shardDb = Config_.OauthShards_->GetPoolUnsafe(shard);
        if (!shardDb) {
            return {};
        }

        try {
            NDbPool::TBlockingHandle sqlh(*shardDb);
            TString query = BuildOAuthShardQuery(sqlh.EscapeQueryParam(token), uid);
#ifdef LOG_QUERIES
            TLog::Error("QUERY (oauth shard): '%s'", query.c_str());
#endif
            return sqlh.Query(query);
        } catch (const std::exception& e) {
            throw TDbpoolError("OAuth pool error querying shard", e.what());
        }
    }

    std::unique_ptr<NDbPool::TResult> TOAuthSingleFetcher::FetchTokenAttributesByAlias(const TString& token,
                                                                                       const TString& uid) {
        TString query;

        std::vector<NDbPool::TNonBlockingHandle> queries;
        queries.reserve(Config_.OauthShards_->GetShardCount());

        for (NDbPool::TDbPool& db : Config_.OauthShards_->GetShards()) {
            NDbPool::TNonBlockingHandle handle(db);
            if (query.empty()) {
                query = BuildOAuthShardQuery(handle.EscapeQueryParam(token), uid);
#ifdef LOG_QUERIES
                TLog::Error("QUERY (oauth shard): '%s'", query.c_str());
#endif
            }

            handle.SendQuery(query);
            queries.push_back(std::move(handle));
        }

        for (NDbPool::TNonBlockingHandle& handle : queries) {
            std::unique_ptr<NDbPool::TResult> result =
                TUtils::WaitResult(handle, "dbpool exception in fetchTokenAttributesByAlias for oauth shard.");
            handle = NDbPool::TNonBlockingHandle();

            if (result->size() > 0) { // token found
                return result;
            }
        }

        return {};
    }

    static const TString QUERY_SHARD_SQL_BY_ACCESS_BEGIN =
        "SELECT id,type,value FROM token_attributes JOIN token_by_access_token USING (id) WHERE access_token='";
    static const TString QUERY_SHARD_SQL_BY_ACCESS_END = "'";
    static const TString QUERY_SHARD_SQL_BY_ALIAS_BEGIN =
        "SELECT id,type,value FROM token_attributes JOIN token_by_alias USING (id) WHERE alias='";
    static const TString QUERY_SHARD_AND_UID = "' AND uid=";
    TString TOAuthSingleFetcher::BuildOAuthShardQuery(const TString& escapedToken, const TString& uid) {
        TString tokenHash = NUtils::CreateStr("1:", NUtils::BinToBase64(NUtils::TCrypto::Sha256(escapedToken), false));
        if (uid.empty()) {
            return NUtils::CreateStr(QUERY_SHARD_SQL_BY_ACCESS_BEGIN, tokenHash, QUERY_SHARD_SQL_BY_ACCESS_END);
        }

        if (!NUtils::DigitsOnly(uid)) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "Invalid uid in token: " << uid;
        }
        return NUtils::CreateStr(QUERY_SHARD_SQL_BY_ALIAS_BEGIN, tokenHash, QUERY_SHARD_AND_UID, uid);
    }

    // don't need to fetch all token attributes, we need only uid, device name, expire and issue time to check glogout
    static const TString SQL_XTOKEN_REQUEST = "SELECT id,type,value FROM token_attributes WHERE type IN (3,6,7,10) AND id=";
    TString TOAuthSingleFetcher::BuildXTokenQuery(const TString& tokenId) {
        if (!NUtils::DigitsOnly(tokenId)) {
            throw TBlackboxError(TBlackboxError::EType::Unknown) << "Invalid xtoken_id in token: " << tokenId;
        }
        return NUtils::CreateStr(SQL_XTOKEN_REQUEST, tokenId);
    }

    bool TOAuthSingleFetcher::CheckEmbeddedInfo(const TOAuthTokenInfo& info, const TOAuthTokenEmbeddedInfo& embeddedInfo) {
        if (embeddedInfo.HasInfo()) {
            if (const auto& tokenId = embeddedInfo.TokenId(); tokenId.has_value() && *tokenId != IntFromString<ui64, 10>(info.TokenId)) {
                TLog::Warning() << "OAuthFetcher: token embedded info invalid"
                                << " - token_id (emb=" << *tokenId
                                << " vs attr=" << info.TokenId
                                << "). tokid: " << info.TokenId;
                return false;
            }
            if (embeddedInfo.ClientId() != IntFromString<ui32, 10>(info.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID))) {
                TLog::Warning() << "OAuthFetcher: token embedded info invalid"
                                << " - client_id (emb=" << embeddedInfo.ClientId()
                                << " vs attr=" << info.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID)
                                << "). tokid: " << info.TokenId;
                return false;
            }
            if (embeddedInfo.Uid() != (info.Uid.empty() ? 0 : IntFromString<ui64, 10>(info.Uid))) {
                TLog::Warning() << "OAuthFetcher: token embedded info invalid"
                                << " - uid (emb=" << embeddedInfo.Uid()
                                << " vs attr=" << info.Uid
                                << "). tokid: " << info.TokenId;
                return false;
            }
        }

        return true;
    }

    bool TOAuthSingleFetcher::FillClientAttributes(TOAuthTokenInfo& info, TOAuthError& error) const {
        TString clientId = info.GetTokenAttr(TOAuthTokenAttr::CLIENT_ID);
        if (clientId.empty()) {
            TLog::Debug("OAuthFetcher: token with empty client id, token_id=%s", info.TokenId.c_str());
            LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::CLIENT_NOT_FOUND);
            error.SetError(TOAuthError::ClientNotFound);
            return false;
        }
        std::unique_ptr<NDbPool::TResult> result = GetCentralResult(clientId);

        if (!result->size()) {
            TLog::Debug("OAuthFetcher: token client with id %s not found", clientId.c_str());
            LogOAuth(info, TOAuthStatboxMsg::ERROR, TOAuthStatboxMsg::CLIENT_NOT_FOUND);
            error.SetError(TOAuthError::ClientNotFound);
            return false;
        }

        // Store all info about client for logging
        bool res = true;

        for (const NDbPool::TRow& row : result->Table()) {
            res = ProcessClientAttribute(row, info, error) && res;
        }

        PostProcessClientAttributes(info);

        return res;
    }

    static const TString QUERY_CENTRAL_CENTRAL_QUERY = "SELECT type,value FROM client_attributes WHERE id=";
    static const TString QUERY_CENTRAL_AND_TYPE_IN = " AND type IN (";
    static const TString QUERY_CENTRAL_AND_TYPE_END = ")";
    TString TOAuthSingleFetcher::BuildOAuthCentralQuery(const TString& clientId) const {
        if (!NUtils::DigitsOnly(clientId)) {
            throw TBlackboxError(TBlackboxError::EType::Unknown)
                << "Invalid client_id in token: " << clientId;
        }
        TString query = NUtils::CreateStr(QUERY_CENTRAL_CENTRAL_QUERY, clientId);

        if (AllClientAttrs_) {
            return query;
        }

        TString clientAttrs(DEFAULT_CLIENT_ATTRS_STR);
        for (auto& p : AddClientAttrs_) {
            NUtils::Append(clientAttrs, ',', p);
        }

        NUtils::Append(query, QUERY_CENTRAL_AND_TYPE_IN, clientAttrs, QUERY_CENTRAL_AND_TYPE_END);

        return query;
    }

    std::unique_ptr<NDbPool::TResult> TOAuthSingleFetcher::GetCentralResult(const TString& clientId) const {
        // if it was a non-blocking request, wait for result, it may be ready
        if (CentralSqlh_) {
            return TUtils::WaitResult(*CentralSqlh_, "dbpool exception in waitResult for oauth central");
        }

        // else, do the synchronous request
        try {
            TString query = BuildOAuthCentralQuery(clientId);
#ifdef LOG_QUERIES
            TLog::Error("QUERY (oauth central): '%s'", query.c_str());
#endif
            NDbPool::TBlockingHandle sqlh(*Config_.OauthCentral_);
            return sqlh.Query(query);
        } catch (const std::exception& e) {
            throw TDbpoolError("OAuth pool error querying central", e.what());
        }
    }

    void TOAuthSingleFetcher::LogOAuth(const TOAuthTokenInfo& info,
                                       const TString& status,
                                       const TString& reason) const {
        Config_.LogOAuth(info, UserIp_, status, reason, Consumer_, ConsumerIp_);
    }
}
