#pragma once

#include <passport/infra/libs/cpp/dbpool/handle.h>

#include <util/generic/string.h>

#include <memory>
#include <mutex>
#include <set>

namespace NPassport::NAuth {
    class TOAuthToken;
}

namespace NPassport::NBb {
    class TOAuthConfig;
    class TOAuthError;
    class TOAuthTokenEmbeddedInfo;
    class TOAuthTokenInfo;

    class TOAuthBaseFetcher {
    public:
        using TAttrSet = std::set<TString>;

        TOAuthBaseFetcher(const TOAuthConfig& config);
        virtual ~TOAuthBaseFetcher() = default;

        void AddClientAttr(const TString& attr);
        void AddAllClientAttrs();

    protected:
        void ProcessTokenAttribute(const NDbPool::TRow& row, TOAuthTokenInfo& info) const;
        void PostProcessTokenAttributes(TOAuthTokenInfo& info) const;

        bool ProcessClientAttribute(const NDbPool::TRow& row,
                                    TOAuthTokenInfo& info,
                                    TOAuthError& error,
                                    time_t now = time(nullptr)) const;
        static void PostProcessClientAttributes(TOAuthTokenInfo& info);

        bool CheckTokenInfo(const TOAuthTokenInfo& info, TOAuthError& error) const;

        virtual void LogOAuth(const TOAuthTokenInfo& info,
                              const TString& status,
                              const TString& reason) const = 0;

    protected:
        const TOAuthConfig& Config_;

        bool AllClientAttrs_ = false;

        // default attributes, requested all time
        static const TAttrSet DEFAULT_CLIENT_ATTRS;
        static const TString DEFAULT_CLIENT_ATTRS_STR;

        // additional attributes, if requested besides default
        TAttrSet AddClientAttrs_;
    };

    class TOAuthSingleFetcher: public TOAuthBaseFetcher {
    public:
        TOAuthSingleFetcher(const TOAuthConfig& config,
                            const TString& userip,
                            const TString& consumer,
                            const TString& consumerIp);

        bool SendNonBlockingRequestByAccess(const TString& token, unsigned shard, const TString& clientId, bool isXtoken);
        std::unique_ptr<TOAuthTokenInfo> CheckTokenByAlias(const TString& token,
                                                           const TString& uid);
        std::unique_ptr<TOAuthTokenInfo> CheckTokenByAccess(const TOAuthTokenEmbeddedInfo& embeddedInfo,
                                                            const TString& token,
                                                            TOAuthError& error);
        std::unique_ptr<TOAuthTokenInfo> CheckTokenByAccess(const NAuth::TOAuthToken& token,
                                                            const TString& strtoken,
                                                            TOAuthError& error);

        static bool IsValidSyntax(const TString& token);

    protected:
        static TString BuildOAuthShardQuery(const TString& escapedToken, const TString& uid);
        static TString BuildXTokenQuery(const TString& tokenId);
        TString BuildOAuthCentralQuery(const TString& clientId) const;

        void FillTokenAttributes(NDbPool::TResult& result, TOAuthTokenInfo& info) const;
        std::unique_ptr<NDbPool::TResult> GetShardResult(const TString& token, const TString& uid, unsigned shard) const;

        std::unique_ptr<NDbPool::TResult> FetchTokenAttributesByAlias(const TString& token, const TString& uid);

        static bool CheckEmbeddedInfo(const TOAuthTokenInfo& info, const TOAuthTokenEmbeddedInfo& embeddedInfo);

        bool FillClientAttributes(TOAuthTokenInfo& info, TOAuthError& error) const;
        std::unique_ptr<NDbPool::TResult> GetCentralResult(const TString& clientId) const;

        void LogOAuth(const TOAuthTokenInfo& info,
                      const TString& status,
                      const TString& reason) const override;

        TString UserIp_;
        const TString Consumer_;
        const TString ConsumerIp_;

        std::unique_ptr<NDbPool::TNonBlockingHandle> CentralSqlh_;
        std::unique_ptr<NDbPool::TNonBlockingHandle> ShardSqlh_;
    };
}
