#include "scopes.h"

#include <passport/infra/daemons/blackbox/src/misc/strings.h>

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    TOAuthSingleScope::TOAuthSingleScope(TString&& keyword)
        : Keyword_(std::move(keyword))
    {
        IsDeleted_ = Keyword_.StartsWith(TStrings::DELETED_SCOPES_PREFIX);
    }

    void TOAuthScopes::Reserve(unsigned n) {
        KeywordList_.reserve(TOAuthSingleScope::MAX_KEYWORD_SIZE * n);
        ScopeCollection_.reserve(n);
    }

    void TOAuthScopes::AddScope(const TOAuthSingleScope& scope) {
        if (ScopeCollection_.find(scope.Keyword()) != ScopeCollection_.end()) {
            return; // ignore if already added
        }

        ScopeCollection_.insert(scope.Keyword());

        // make keyword list string, add separator if needed
        NUtils::AppendSeparated(KeywordList_, ' ', scope.Keyword());

        // update ttl to be minimum of non-null scope ttls
        if (scope.Ttl() != 0) {
            if (Ttl_ == 0 || scope.Ttl() < Ttl_) {
                Ttl_ = scope.Ttl();
            }
        }

        if (scope.IsXToken()) {
            IsXToken_ = true;
        }
    }

    TOAuthScopesConfig::TOAuthScopesConfig(const TString& path, ui32 period)
        : Path_(path)
        , DefaultScope_(TStrings::DEFAULT_SCOPE_KEYWORD.copy())
    {
        Loader_ = std::make_unique<NUtils::TFileLoader>(
            path,
            [this](const TStringBuf fileBody, const time_t) { this->Update(fileBody); },
            TDuration::Seconds(period));
    }

    TOAuthScopesConfig::~TOAuthScopesConfig() = default;

    void TOAuthScopesConfig::Update(const TStringBuf fileBody) {
        auto s = Parse(fileBody);
        Scopes_.Set(s);
        TLog::Info("OAuthScopes[OK]: Loaded %lu scopes", s->size());
    }

    std::shared_ptr<TOAuthScopesConfig::TScopes> TOAuthScopesConfig::Parse(const TStringBuf fileBody) {
        rapidjson::Document doc;
        if (!NJson::TReader::DocumentAsObject(fileBody, doc)) {
            throw yexception() << "OAuthScopes[ERROR]: json is invalid";
        }

        std::shared_ptr<TScopes> scopes = std::make_shared<TScopes>();
        scopes->reserve(doc.MemberCount());

        for (auto it = doc.MemberBegin(); it != doc.MemberEnd(); ++it) {
            TString scopeId(it->name.GetString(), it->name.GetStringLength());
            try {
                scopes->insert(std::make_pair(scopeId, LoadScope(it->value)));
            } catch (const std::exception& e) {
                TLog::Warning("OAuthScopes: failed to load scope (id=%s): %s", scopeId.c_str(), e.what());
                continue;
            }
        }

        return scopes;
    }

    TOAuthSingleScope TOAuthScopesConfig::LoadScope(rapidjson::Value& jsonValue) {
        TString keyword;
        if (!NJson::TReader::MemberAsString(jsonValue, "keyword", keyword)) {
            throw yexception() << "keyword is not string";
        }
        if (keyword.empty()) {
            throw yexception() << "keyword is empty";
        }

        TOAuthSingleScope scope(std::move(keyword));
        NJson::TReader::MemberAsUInt(jsonValue, "ttl", scope.Ttl_);
        NJson::TReader::MemberAsBool(jsonValue, "has_xtoken_grant", scope.IsXToken_);
        return scope;
    }

    TOAuthScopes TOAuthScopesConfig::GetScopesFromNums(const TString& numsContainer) const {
        TOAuthScopes result;
        const size_t expectedCount = numsContainer.size() / 2 + 1; // |1|2|3|4|5|...|9| -> 10
        result.Reserve(expectedCount);

        const auto scopes = Scopes_.Get();
        bool hasDeletedScopes = false;

        NUtils::Transform(
            numsContainer,
            '|',
            [&scopes, &result, &hasDeletedScopes](const TStringBuf buf) {
                if (buf.empty()) {
                    return;
                }

                TString id(buf);
                auto it = scopes->find(id);
                if (it == scopes->end()) {
                    TLog::Error("OAuthScopes: scope with id '%s' doesn't exist", id.c_str());
                    return;
                }

                if (it->second.IsDeleted()) {
                    hasDeletedScopes = true;
                    return;
                }

                result.AddScope(it->second);
            });

        if (result.ScopeCollection().empty() && hasDeletedScopes) {
            result.AddScope(DefaultScope_);
        }

        return result;
    }

    TOAuthScopes TOAuthScopesConfig::GetScopesFromIds(const std::unordered_set<ui32>& ids) const {
        TOAuthScopes result;
        result.Reserve(ids.size());

        const auto scopes = Scopes_.Get();
        bool hasDeletedScopes = false;

        for (const ui32 id : ids) {
            TString strId = IntToString<10>(id);
            auto it = scopes->find(strId);
            if (it == scopes->end()) {
                TLog::Error("OAuthScopes: scope with id '%d' doesn't exist", id);
                continue;
            }

            if (it->second.IsDeleted()) {
                hasDeletedScopes = true;
                continue;
            }

            result.AddScope(it->second);
        }

        if (result.ScopeCollection().empty() && hasDeletedScopes) {
            result.AddScope(DefaultScope_);
        }

        return result;
    }

}
