#include "token_embedded_info.h"

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/digest/old_crc/crc.h>

namespace NPassport::NBb {
    static const size_t V2_TOKEN_CONST_SIZE = 39; // base64url(shard, uid, clientid, random)
    static const size_t V3_TOKEN_CONST_SIZE = 58; // y(env)_base64url(shard, uid, clientid, token_id, random, crc32)

    static std::optional<NAuth::EEnvironmentType> ParseEnvironmentType(char byte) {
        switch (byte) {
            case '0':
                return NAuth::EEnvironmentType::Production;
            case '1':
                return NAuth::EEnvironmentType::ProductionYateam;
            case '2':
                return NAuth::EEnvironmentType::Testing;
            case '3':
                return NAuth::EEnvironmentType::TestingYateam;
            case '4':
                return NAuth::EEnvironmentType::Load;
            default:
                return {};
        }
    }

    TOAuthTokenEmbeddedInfo TOAuthTokenEmbeddedInfo::Parse(const TString& token) {
        switch (token.size()) {
            case V2_TOKEN_CONST_SIZE:
                return ParseV2(token);
            case V3_TOKEN_CONST_SIZE:
                return ParseV3(token);
            default:
                // old legacy token without any embedded info. Probably should check for it's size and syntax
                return {};
        }
    }

    TOAuthTokenEmbeddedInfo TOAuthTokenEmbeddedInfo::ParseV2(const TString& token) {
        TOAuthTokenEmbeddedInfo info;

        if (token.size() != V2_TOKEN_CONST_SIZE) {
            info.ErrMsg_ = "invalid token size";
            return info;
        }

        TString decoded = NUtils::Base64url2bin(token);
        if (decoded.empty()) {
            info.ErrMsg_ = "broken base64url";
            return info;
        }

        info.Shard_ = decoded[0] - 1;
        info.Uid_ = NUtils::FromBytesMsb<ui64>(decoded, 1);
        info.ClientId_ = NUtils::FromBytesMsb<ui32>(decoded, 9);

        info.HasInfo_ = true;
        return info;
    }

    TOAuthTokenEmbeddedInfo TOAuthTokenEmbeddedInfo::ParseV3(const TString& token) {
        TOAuthTokenEmbeddedInfo info;

        if (token.size() != V3_TOKEN_CONST_SIZE) {
            info.ErrMsg_ = "invalid token size";
            return info;
        }

        if (token[0] != 'y' || token[2] != '_') {
            info.ErrMsg_ = "malformed prefix";
            return info;
        }

        info.Environment_ = ParseEnvironmentType(token[1]);
        if (!info.Environment_) {
            info.ErrMsg_ = "malformed environment type";
            return info;
        }

        TString decoded = NUtils::Base64url2bin(TStringBuf(token).Skip(3));
        if (decoded.empty()) {
            info.ErrMsg_ = "broken base64url";
            return info;
        }

        info.Shard_ = decoded[0] - 1;
        info.Uid_ = NUtils::FromBytesMsb<ui64>(decoded, 1);
        info.ClientId_ = NUtils::FromBytesMsb<ui32>(decoded, 9);

        info.TokenId_ = NUtils::FromBytesMsb<ui64>(decoded, 13);
        ui32 crc = NUtils::FromBytesMsb<ui32>(decoded, 37);
        if (crc != crc32(decoded.data(), decoded.size() - 4)) {
            info.ErrMsg_ = "broken crc";
            return info;
        }

        info.HasInfo_ = true;
        return info;
    }
}
