#include "token_info.h"

#include "error.h"
#include "scopes.h"

#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <ctime>

namespace NPassport::NBb {
    TOAuthTokenInfo::TOAuthTokenInfo(bool byAlias)
        : ByAlias(byAlias)
    {
    }

    TOAuthTokenInfo::TOAuthTokenInfo(const NAuth::TOAuthToken& token,
                                     const TString& strtoken,
                                     const TOAuthScopesConfig& scopes) {
        TokenAttrs[TOAuthTokenAttr::IS_STATELESS] = TStrings::ONE;
        TokenAttrs[TOAuthTokenAttr::ACCESS_TOKEN] = strtoken;

        TokenId = token.TokenId();
        TokenAttrs[TOAuthTokenAttr::TOKEN_ID] = TokenId;

        Uid = token.Uid();
        TokenAttrs[TOAuthTokenAttr::UID] = Uid.empty() ? "0" : Uid;

        DeviceId = token.DeviceId();
        TokenAttrs[TOAuthTokenAttr::DEVICE_ID] = DeviceId.empty() ? "-" : DeviceId;

        if (!token.Scopes().empty()) {
            TString scopesAttr;
            SetScopes(scopes.GetScopesFromIds(token.Scopes()));
            scopesAttr.reserve(4 * token.Scopes().size()); // each scope is about 3 digits + separator
            scopesAttr.push_back('|');
            for (const auto id : token.Scopes()) {
                scopesAttr.append(IntToString<10>(id)).push_back('|');
            }
            TokenAttrs[TOAuthTokenAttr::SCOPE_IDS] = scopesAttr;
        }

        TString createTimestamp = token.TokenIdTime();
        SetCreateTs(createTimestamp);
        TokenAttrs[TOAuthTokenAttr::CREATED] = createTimestamp;

        SetIssueTs(token.CreateTime());
        TokenAttrs[TOAuthTokenAttr::ISSUED] = IntToString<10>(token.CreateTime());

        SetExpireTs(token.ExpireTime());
        TokenAttrs[TOAuthTokenAttr::EXPIRES] = IntToString<10>(token.ExpireTime());

        TokenAttrs[TOAuthTokenAttr::CLIENT_ID] = token.ClientId();
        ClientAttrs[TOAuthClientAttr::CLIENT_ID] = token.ClientId();
        if (!token.XtokenId().empty()) {
            TokenAttrs[TOAuthTokenAttr::X_TOKEN_ID] = token.XtokenId();
        }

        if (!token.Meta().empty()) {
            TokenAttrs[TOAuthTokenAttr::META] = token.Meta();
        }

        if (!token.LoginId().empty()) {
            TokenAttrs[TOAuthTokenAttr::LOGIN_ID] = token.LoginId();
        }

        if (!token.PaymentAuthContextId().empty()) {
            TokenAttrs[TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID] = token.PaymentAuthContextId();
        }

        if (!token.PaymentAuthScopeAddendum().empty()) {
            TokenAttrs[TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM] = token.PaymentAuthScopeAddendum();
        }
    }

    TString TOAuthTokenInfo::AuthLogComment(const TStringBuf userPort, ui32 consumerTvmId) const {
        TString comment = NUtils::CreateStrExt(200, "clid=", GetClientAttr(TOAuthClientAttr::DISPLAY_ID));

        if (!TokenId.empty()) {
            NUtils::Append(comment,
                           ";tokid=",
                           TokenId);
        }

        if (HasTokenAttr(TOAuthTokenAttr::DEVICE_ID)) {
            NUtils::Append(comment,
                           ";devid=",
                           NUtils::Urlencode(GetTokenAttr(TOAuthTokenAttr::DEVICE_ID)));
        }

        if (HasTokenAttr(TOAuthTokenAttr::DEVICE_NAME)) {
            NUtils::Append(comment,
                           ";devnm=",
                           NUtils::Urlencode(GetTokenAttr(TOAuthTokenAttr::DEVICE_NAME)));
        }

        if (ScopesComma.empty()) {
            TString scopes(OAuthScopes_.KeywordList());
            std::replace(scopes.begin(), scopes.vend(), ' ', ',');
            NUtils::Append(comment,
                           ";scope=",
                           scopes);
        } else {
            NUtils::Append(comment,
                           ";scope=",
                           ScopesComma);
        }

        if (IsRefreshRequired) {
            comment.append(";rf=1");
        }

        if (userPort) {
            NUtils::Append(comment,
                           ";user_port=",
                           userPort);
        }

        if (consumerTvmId) {
            NUtils::Append(comment,
                           ";tvm_id=",
                           consumerTvmId);
        }

        return comment;
    }

    static const TString SCOPE_ERROR_STR =
        "scope check failed: some scopes are not present in the granted scopes list: ";

    bool TOAuthTokenInfo::HasScopes(const TString& scopes, TOAuthError* error) const {
        TString notPresent;
        const std::unordered_set<TString>& scopeCollection = GetScopeCollection();
        notPresent.reserve(TOAuthSingleScope::MAX_KEYWORD_SIZE * scopeCollection.size());
        NUtils::Transform(
            scopes,
            " ,",
            [&](const TStringBuf buf) {
                if (buf.empty()) {
                    return;
                }

                TString keyword(buf);
                if (scopeCollection.find(keyword) != scopeCollection.end()) {
                    return;
                }
                // append keyword to comma-separated list
                NUtils::AddMessage(notPresent, keyword);
            });

        if (notPresent.empty()) {
            return true;
        }

        if (error) {
            error->SetMsg(SCOPE_ERROR_STR + notPresent, TOAuthError::WrongScope);
        }
        return false;
    }

    void TOAuthTokenInfo::SetScopes(TOAuthScopes&& scopes) {
        OAuthScopes_ = std::move(scopes);
        ScopesComma = OAuthScopes_.KeywordList();
        std::replace(ScopesComma.begin(), ScopesComma.vend(), ' ', ',');
    }

    void TOAuthTokenInfo::SetRefreshRequired(double refreshRatio) {
        if (!IsTtlRefreshable) {
            return;
        }

        IsRefreshRequired = (ExpireTimeTs - time(nullptr)) < OAuthScopes_.Ttl() * refreshRatio;
    }

    void TOAuthTokenInfo::SetTokenAttr(const TString& name, TString&& value) {
        // empty values considered as null
        if (value.empty()) {
            return;
        }

        TokenAttrs[name] = std::move(value);
    }

    void TOAuthTokenInfo::SetClientAttr(const TString& name, TString&& value) {
        // empty values considered as null
        if (value.empty()) {
            return;
        }

        ClientAttrs[name] = std::move(value);
    }

    bool TOAuthTokenInfo::HasTokenAttr(const TString& name) const {
        return TokenAttrs.find(name) != TokenAttrs.end();
    }

    bool TOAuthTokenInfo::HasClientAttr(const TString& name) const {
        return ClientAttrs.find(name) != ClientAttrs.end();
    }

    const TString& TOAuthTokenInfo::GetTokenAttr(const TString& name) const {
        TAttrMap::const_iterator it = TokenAttrs.find(name);
        return it != TokenAttrs.end() ? it->second : TStrings::EMPTY;
    }

    const TString& TOAuthTokenInfo::GetClientAttr(const TString& name) const {
        TAttrMap::const_iterator it = ClientAttrs.find(name);
        return it != ClientAttrs.end() ? it->second : TStrings::EMPTY;
    }

    void TOAuthTokenInfo::SetCreateTs(const TString& value) {
        if (value.empty()) {
            CreateTimeTs = 0;
            CreateTime.clear();
            return;
        }
        CreateTimeTs = IntFromString<time_t, 10>(value);
        CreateTime = NUtils::FormatTimestamp(value);
    }

    void TOAuthTokenInfo::SetIssueTs(const TString& value) {
        if (value.empty()) {
            IssueTimeTs = 0;
            IssueTime.clear();
            return;
        }
        IssueTimeTs = IntFromString<time_t, 10>(value);
        IssueTime = NUtils::FormatTimestamp(value);
    }

    void TOAuthTokenInfo::SetExpireTs(const TString& value) {
        if (value.empty()) {
            ExpireTimeTs = 0;
            ExpireTime.clear();
            return;
        }
        ExpireTimeTs = IntFromString<time_t, 10>(value);
        ExpireTime = NUtils::FormatTimestamp(value);
    }

    void TOAuthTokenInfo::SetIssueTs(time_t value) {
        IssueTimeTs = value;
        IssueTime = NUtils::FormatTimestamp(value);
    }

    void TOAuthTokenInfo::SetExpireTs(time_t value) {
        ExpireTimeTs = value;
        ExpireTime = NUtils::FormatTimestamp(value);
    }

    time_t TOAuthTokenInfo::GetMinimumIssueTime() const {
        // if xtoken issue time is known, check minimum of token issue time and xtoken issue time
        // i.e. consider glogouted/revoked if any one of them is glogouted or revoked

        return XtokenIssueTimeTs > 0 ? std::min(XtokenIssueTimeTs, IssueTimeTs)
                                     : IssueTimeTs;
    }

    bool TOAuthTokenInfo::IsXTokenTrusted() const {
        if (!ShowIsXTokenTrusted) {
            return false;
        }

        const bool attrValue = TDbValue::AsBoolean(
            GetTokenAttr(TOAuthTokenAttr::IS_XTOKEN_TRUSTED));
        if (attrValue && !OAuthScopes_.IsXToken()) {
            TLog::Warning() << "OAuthTokenInfo: token=" << TokenId
                            << " has attr IS_XTOKEN_TRUSTED (" << TOAuthTokenAttr::IS_XTOKEN_TRUSTED
                            << ") but it is not xtoken";
        }

        return attrValue && OAuthScopes_.IsXToken();
    }

    TString TOAuthTokenInfo::GetLoginId() const {
        const TString& loginId = GetTokenAttr(TOAuthTokenAttr::LOGIN_ID);
        if (!loginId.empty()) {
            return loginId; // has custom login_id
        }

        const TString& xtokenId = GetTokenAttr(TOAuthTokenAttr::X_TOKEN_ID);
        if (!xtokenId.empty()) {
            return NUtils::CreateStr("t:", xtokenId);
        }

        return NUtils::CreateStr("t:", TokenId);
    }

    const TString& TOAuthTokenInfo::GetScopesKeywordList() const {
        return OAuthScopes_.KeywordList();
    }

    const std::unordered_set<TString>& TOAuthTokenInfo::GetScopeCollection() const {
        return OAuthScopes_.ScopeCollection();
    }

    bool TOAuthTokenInfo::IsXToken() const {
        return OAuthScopes_.IsXToken();
    }

    void TOAuthTokenInfo::AddScope(const TOAuthSingleScope& scope) {
        OAuthScopes_.AddScope(scope);
    }
}
