#include "json_serializer.h"

#include "account_chunk.h"
#include "all_tracks_result.h"
#include "attributes_chunk.h"
#include "auth_chunk.h"
#include "authid_chunk.h"
#include "billing_features_chunk.h"
#include "check_device_signature_result.h"
#include "check_grants_result.h"
#include "check_has_plus_result.h"
#include "check_rfc_totp_result.h"
#include "check_sign_result.h"
#include "create_oauth_token_result.h"
#include "create_session_result.h"
#include "dbfields_chunk.h"
#include "decrease_sessionid_lifetime_result.h"
#include "display_name_chunk.h"
#include "edit_totp_result.h"
#include "emails_chunk.h"
#include "ext_attrs_chunk.h"
#include "family_info_chunk.h"
#include "family_info_result.h"
#include "find_by_phone_numbers_result.h"
#include "get_debug_user_ticket_result.h"
#include "get_device_public_key_result.h"
#include "get_max_uid_result.h"
#include "get_oauth_tokens_result.h"
#include "karma_chunk.h"
#include "lcookie_result.h"
#include "list_result.h"
#include "login_result.h"
#include "loginoccupation_result.h"
#include "new_session_chunk.h"
#include "oauth_result.h"
#include "out_tokens.h"
#include "phone_bindings_chunk.h"
#include "phone_bindings_result.h"
#include "phone_operations_chunk.h"
#include "plus_subscriber_state_chunk.h"
#include "prove_key_diag_result.h"
#include "pwdhistory_result.h"
#include "resigned_cookies.h"
#include "sessguard_chunk.h"
#include "session_result.h"
#include "table_result.h"
#include "test_pwd_hashes_result.h"
#include "track_result.h"
#include "typed_value_result.h"
#include "uid_chunk.h"
#include "user_info_result.h"
#include "user_ticket_result.h"
#include "webauthn_credentials_result.h"

#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/string/cast.h>

namespace NPassport::NBb {
    class TJsonSerializerImpl {
    public:
        static void Serialize(const TAccountChunk* chunk, NJson::TObject& val);
        static void Serialize(const TDbFieldsChunk* dbfields, NJson::TObject& val);
        static void Serialize(const TAttributesChunk* attrChunk, const TString& nodeName, NJson::TObject& val);
        static void Serialize(const TDisplayNameChunk* displayName, NJson::TObject& val);
        static void Serialize(const TKarmaChunk* karma, NJson::TObject& val);
        static void Serialize(const TUidChunk* uid, NJson::TObject& val);
        static void Serialize(const TEmailsChunk* emails, NJson::TObject& val);
        static void Serialize(const TAuthIdChunk* authId, NJson::TObject& val);
        static void Serialize(const TNewSessionChunk* newsession, NJson::TObject& val);
        static void Serialize(const TResignedCookiesChunk* resignedCookies, NJson::TObject& val);
        static void Serialize(const TSessguardChunk* newSessguards, NJson::TObject& val);
        static void Serialize(const TAuthChunk* auth, NJson::TObject& val);
        static void Serialize(const TExtAttrsChunk* extAttrsChunk, const TString& nodeName, NJson::TObject& val);
        static void Serialize(const TPhoneOperationsChunk* phoneOpsChunk, NJson::TObject& val);
        static void Serialize(const TPhoneBindingsChunk* chunk, NJson::TObject& val);
        static void Serialize(const TBillingFeaturesChunk* featuresChunk, NJson::TObject& val);
        static void Serialize(const TPlusSubscriberStateChunk* stateChunk, NJson::TObject& val);
        static void Serialize(const TFamilyInfoChunk* chunk, NJson::TObject& val);

        static void Serialize(const TOAuthTokenInfo& info, NJson::TObject& val);

        static void SerializeStatus(const TSessionResult& result, NJson::TObject& val);
        static void Serialize(const TOAuthChunk& chunk, NJson::TObject& val);

        static void Serialize(const TBaseResult& result, NJson::TObject& val);
        static void Serialize(const TSessionUser* user, NJson::TObject& val);
        static void Serialize(const TUserInfoResult& result, NJson::TObject& val);
        static void Serialize(const TBulkUserInfoResult& result, NJson::TObject& val);
        static void SerializeException(const TExceptionInfo& info, NJson::TObject& val);
        static void Serialize(const NJson::TObject& obj, TString& buf);
    };

    void TJsonSerializerImpl::Serialize(const TAccountChunk* chunk, NJson::TObject& val) {
        if (nullptr == chunk) {
            return;
        }
        val.Add(TOutTokens::LOGIN, chunk->Login);
        if (TExperiment::Get().IsHavePasswordEnabled) {
            val.Add(TOutTokens::HAVE_PASSWORD, chunk->HavePassword);
        }
        if (TExperiment::Get().IsHaveHintEnabled) {
            val.Add(TOutTokens::HAVE_HINT, chunk->HaveHint);
        }

        using TAliases = TAccountChunk::TAliasesType;
        const TAliases& aliases = chunk->Aliases;
        if (chunk->AreAliasesRequired) {
            NJson::TObject aliasesObj(val, TOutTokens::ALIASES);
            using TIter = TAliases::const_iterator;
            for (TIter it = aliases.begin(); it != aliases.end();) {
                std::pair<TIter, TIter> res = aliases.equal_range(it->first);
                if (it->first == TAlias::PDD_ALIAS_LOGIN || it->first == TAlias::OLDPUBLICID) {
                    NJson::TArray aliasesArr(aliasesObj, it->first);
                    for (TIter iter = res.first; iter != res.second; ++iter) {
                        aliasesArr.Add(iter->second);
                    }
                    it = res.second;
                } else {
                    aliasesObj.Add(it->first, it->second);
                    ++it;
                }
            }
        }
    }

    void TJsonSerializerImpl::Serialize(const TDbFieldsChunk* dbfields, NJson::TObject& val) {
        if (nullptr == dbfields || dbfields->Fields.empty()) {
            return;
        }
        NJson::TObject fields(val, TOutTokens::DBFIELDS);
        for (const TDbFieldsChunk::TField& field : dbfields->Fields) {
            fields.Add(field.Name, field.Value);
        }
    }

    void TJsonSerializerImpl::Serialize(const TAttributesChunk* attrChunk, const TString& nodeName, NJson::TObject& val) {
        if (nullptr == attrChunk) {
            return;
        }
        NJson::TObject rootObj(val, nodeName);
        for (const auto& [type, value] : attrChunk->Attrs) {
            rootObj.Add(type, value);
        }
    }

    void TJsonSerializerImpl::Serialize(const TDisplayNameChunk* displayName, NJson::TObject& val) {
        if (nullptr == displayName) {
            return;
        }
        val.Add(TOutTokens::REGNAME, displayName->Regname);

        NJson::TObject dname(val, TOutTokens::DISPLAYNAME);
        dname.Add(TOutTokens::NAME, displayName->DisplayName);

        const std::optional<bool>& isDisplayNameEmpty = displayName->IsDisplayNameEmpty;
        if (isDisplayNameEmpty) {
            dname.Add(TOutTokens::DISPLAY_NAME_EMPTY, *isDisplayNameEmpty);
        }

        const std::optional<TPublicNameData>& publicName = displayName->PublicName;
        if (publicName) {
            dname.Add(TOutTokens::PUBLIC_NAME, publicName->Name);
            dname.Add(TOutTokens::HAS_PUBLIC_PROFILE, publicName->HasPublicProfile);
            dname.Add(TOutTokens::THIRD_PARTY_CAN_USE, publicName->ThirdPartyCanUse);
        }

        {
            NJson::TObject avt(dname, TOutTokens::AVATAR);
            avt.Add(TOutTokens::DEFAULT, displayName->Avatar);
            avt.Add(TOutTokens::EMPTY, displayName->IsAvatarEmpty);
        }

        if (displayName->Verified) {
            dname.Add(TOutTokens::VERIFIED, true);
        }

        if (displayName->Social) {
            NJson::TObject social(dname, TOutTokens::SOCIAL);
            social.Add(TOutTokens::PROFILE_ID, displayName->ProfileId);
            social.Add(TOutTokens::PROVIDER, displayName->Provider);
            social.Add(TOutTokens::REDIRECT_TARGET, displayName->Target);
        }
    }

    void TJsonSerializerImpl::Serialize(const TKarmaChunk* karma, NJson::TObject& val) {
        if (nullptr == karma) {
            return;
        }
        {
            NJson::TObject karmaObj(val, TOutTokens::KARMA);
            karmaObj.Add(TOutTokens::VALUE, karma->Karma);
            if (!karma->Bantime.empty()) {
                try {
                    karmaObj.Add(TOutTokens::ALLOW_UNTIL,
                                 IntFromString<time_t, 10>(karma->Bantime));
                } catch (std::logic_error&) {
                    throw yexception() << "Incorrect format of " << TOutTokens::ALLOW_UNTIL;
                }
            }
        }
        NJson::TObject karmaStatusObj(val, TOutTokens::KARMA_STATUS);
        karmaStatusObj.Add(TOutTokens::VALUE, karma->RawValue);
    }

    void TJsonSerializerImpl::Serialize(const TUidChunk* uid, NJson::TObject& val) {
        if (nullptr == uid) {
            return;
        }
        NJson::TObject uidObj(val, TOutTokens::UID);
        if (!uid->Uid.empty()) {
            uidObj.Add(TOutTokens::VALUE, uid->Uid);
            uidObj.Add(TOutTokens::LITE, uid->Lite);
            uidObj.Add(TOutTokens::HOSTED, uid->Hosted);
            if (uid->Hosted) {
                uidObj.Add(TOutTokens::DOMID, uid->DomainId);
                uidObj.Add(TOutTokens::DOMAIN_, uid->Domain);
                uidObj.Add(TOutTokens::MX, uid->Mx ? TOutTokens::ONE : TOutTokens::ZERO);
                uidObj.Add(TOutTokens::DOMAIN_ENA, uid->DomainEna ? TOutTokens::ONE : TOutTokens::ZERO);
                uidObj.Add(TOutTokens::CATCH_ALL, uid->CatchAll);
            }
        }
    }

    void TJsonSerializerImpl::Serialize(const TEmailsChunk* emails, NJson::TObject& val) {
        if (nullptr == emails) {
            return;
        }

        NJson::TArray addresses(val, TOutTokens::ADDRESS_LIST);
        for (const TValidatorAddress& addr : emails->Emails) {
            NJson::TObject address(addresses);
            address.Add(TOutTokens::ADDRESS, addr.Address);
            address.Add(TOutTokens::VALIDATED, addr.Confirmed);
            address.Add(TOutTokens::DEFAULT, addr.Default);
            address.Add(TOutTokens::RPOP, addr.Rpop);
            address.Add(TOutTokens::SILENT, addr.Silent);
            address.Add(TOutTokens::UNSAFE, addr.Unsafe);
            address.Add(TOutTokens::NATIVE, addr.Native);
            address.Add(TOutTokens::BORN_DATE, addr.Timestamp);
        }
    }

    void TJsonSerializerImpl::Serialize(const TAuthIdChunk* authId, NJson::TObject& val) {
        if (nullptr == authId) {
            return;
        }
        NJson::TObject authidObj(val, TOutTokens::AUTHID);
        authidObj.Add(TOutTokens::ID, authId->Id);
        authidObj.Add(TOutTokens::TIME, authId->Timestamp);
        authidObj.Add(TOutTokens::IP, authId->Ip);
        authidObj.Add(TOutTokens::HOST, authId->Host);
    }

    void TJsonSerializerImpl::Serialize(const TNewSessionChunk* newsession, NJson::TObject& val) {
        if (nullptr == newsession) {
            return;
        }

        {
            NJson::TObject sessionObj(val, TOutTokens::NEW_SESSION);
            sessionObj.Add(TOutTokens::VALUE, newsession->Value);
            sessionObj.Add(TOutTokens::DOMAIN_, newsession->Domain);
            sessionObj.Add(TOutTokens::EXPIRES, newsession->Expires);
            sessionObj.Add(TOutTokens::HTTP_ONLY, true);
            sessionObj.Add(TOutTokens::SECURE, true);
        }

        {
            NJson::TObject sessionObj(val, TOutTokens::NEW_SSLSESSION);
            sessionObj.Add(TOutTokens::VALUE, newsession->LegacyValue);
            sessionObj.Add(TOutTokens::DOMAIN_, newsession->Domain);
            sessionObj.Add(TOutTokens::EXPIRES, newsession->Expires);
            sessionObj.Add(TOutTokens::HTTP_ONLY, true);
            sessionObj.Add(TOutTokens::SECURE, true);
        }
    }

    void TJsonSerializerImpl::Serialize(const TResignedCookiesChunk* resignedCookies, NJson::TObject& val) {
        if (nullptr == resignedCookies) {
            return;
        }

        NJson::TObject obj(val, TOutTokens::RESIGNED_COOKIES);
        for (const auto& [domain, cookie] : resignedCookies->Cookies) {
            NJson::TObject domainObj(obj, domain);
            Serialize(cookie.get(), domainObj);
        }
    }

    void TJsonSerializerImpl::Serialize(const TSessguardChunk* newSessguards, NJson::TObject& val) {
        if (nullptr == newSessguards) {
            return;
        }

        NJson::TObject rootObj(val, TOutTokens::NEW_SESSGUARDS);
        for (const auto& [host, cookie] : newSessguards->Cookies) {
            NJson::TObject obj(rootObj, host);
            NJson::TObject sessguard(obj, TOutTokens::SESSGUARD);
            sessguard.Add(TOutTokens::VALUE, cookie.Value);
            sessguard.Add(TOutTokens::DOMAIN_, cookie.Domain);
            sessguard.Add(TOutTokens::EXPIRES, cookie.Expires);
        }
    }

    void TJsonSerializerImpl::Serialize(const TAuthChunk* auth, NJson::TObject& val) {
        if (nullptr == auth) {
            return;
        }
        NJson::TObject authObj(val, TOutTokens::AUTH);
        authObj.Add(TOutTokens::PASSWORD_VERIFICATION_AGE, auth->PasswordVerificationAge);
        if (TExperiment::Get().IsHavePasswordEnabled) {
            authObj.Add(TOutTokens::HAVE_PASSWORD, auth->HavePassword);
        }
        authObj.Add(TOutTokens::SECURE, true);
        if (TExperiment::Get().IsPddPartnerTokenEnabled) {
            authObj.Add(TOutTokens::PARTNER_PDD_TOKEN, false);
        }
        if (!auth->ProfileId.empty()) {
            NJson::TObject socialObj(authObj, TOutTokens::SOCIAL);
            socialObj.Add(TOutTokens::PROFILE_ID, auth->ProfileId);
        }
        if (auth->Scholar) {
            authObj.Add(TOutTokens::IS_SCHOLAR_SESSION, true);
        }
    }

    void TJsonSerializerImpl::Serialize(const TExtAttrsChunk* extAttrsChunk, const TString& nodeName, NJson::TObject& val) {
        if (nullptr == extAttrsChunk) {
            return;
        }

        NJson::TArray rootArr(val, nodeName);

        for (const auto& [extId, extAttrs] : extAttrsChunk->Attrs) {
            NJson::TObject itemObj(rootArr);
            itemObj.Add(TOutTokens::ID, extId);
            NJson::TObject attrs(itemObj, TOutTokens::ATTRIBUTES);
            for (const auto& [type, value] : extAttrs) {
                attrs.Add(type, value);
            }
        }
    }

    void TJsonSerializerImpl::Serialize(const TPhoneOperationsChunk* phoneOpsChunk, NJson::TObject& val) {
        if (nullptr == phoneOpsChunk) {
            return;
        }

        NJson::TObject opsObj(val, TOutTokens::PHONE_OPERATIONS);
        for (const auto& [id, value] : phoneOpsChunk->Ops) {
            opsObj.Add(id, value);
        }
    }

    void TJsonSerializerImpl::Serialize(const TSessionUser* user, NJson::TObject& val) {
        if (nullptr == user) {
            return;
        }
        if (user->Status) {
            NJson::TObject obj(val, TOutTokens::STATUS);
            obj.Add(TOutTokens::VALUE, user->StatusStr());
            obj.Add(TOutTokens::ID, (int)*user->Status);
        }
        if (user->IsRegCompletionRecommended) {
            val.Add(TOutTokens::IS_REG_COMPLETION_RECOMMENDED, user->IsRegCompletionRecommended.value());
        }

        Serialize((TBaseResult&)*user, val);
        Serialize(user->Auth.get(), val);
    }

    void TJsonSerializerImpl::SerializeStatus(const TSessionResult& result, NJson::TObject& val) {
        if (result.Status) {
            NJson::TObject statusObj(val, TOutTokens::STATUS);
            statusObj.Add(TOutTokens::VALUE, result.StatusStr());
            statusObj.Add(TOutTokens::ID, static_cast<int>(*result.Status));
        }
        val.Add(TOutTokens::ERROR, result.Comment);
    }

    void TJsonSerializerImpl::Serialize(const TOAuthChunk& chunk, NJson::TObject& val) {
        if (chunk.TokenInfo) {
            NJson::TObject oauth(val, TOutTokens::OAUTH);

            TJsonSerializerImpl::Serialize(*chunk.TokenInfo, oauth);

            TJsonSerializerImpl::Serialize(chunk.TokenAttrs.get(), TOutTokens::TOKEN_ATTRIBUTES, oauth);
            TJsonSerializerImpl::Serialize(chunk.ClientAttrs.get(), TOutTokens::CLIENT_ATTRIBUTES, oauth);
        }

        {
            NJson::TObject statusObj(val, TOutTokens::STATUS);
            statusObj.Add(TOutTokens::VALUE, chunk.Status.AsString());
            statusObj.Add(TOutTokens::ID, static_cast<int>(chunk.Status.Status()));
        }

        val.Add(TOutTokens::ERROR, chunk.Comment);
        if (!chunk.LoginId.empty()) {
            val.Add(TOutTokens::LOGIN_ID, chunk.LoginId);
        }
    }

    void TJsonSerializerImpl::Serialize(const TBaseResult& result, NJson::TObject& val) {
        Serialize(result.Uid.get(), val);
        Serialize(result.Account.get(), val);
        Serialize(result.Karma.get(), val);
        Serialize(result.DisplayName.get(), val);
        Serialize(result.Dbfields.get(), val);
        Serialize(result.Attributes.get(), TOutTokens::ATTRIBUTES, val);
        Serialize(result.Emails.get(), val);
        Serialize(result.PhoneAttrs.get(), TOutTokens::PHONES, val);
        Serialize(result.EmailAttrs.get(), TOutTokens::EMAILS, val);
        Serialize(result.WebauthnAttrs.get(), TOutTokens::WEBAUTHN_CREDENTIALS, val);
        Serialize(result.PhoneOperations.get(), val);
        Serialize(result.PhoneBindings.get(), val);
        Serialize(result.BillingFeatures.get(), val);
        Serialize(result.PlusSubscriberState.get(), val);
        Serialize(result.FamilyInfo.get(), val);
        if (!result.UserTicket.empty()) {
            val.Add(TOutTokens::USER_TICKET, result.UserTicket);
        }

        if (result.PublicId) {
            val.Add(TOutTokens::PUBLIC_ID, *result.PublicId);
        }
    }

    void TJsonSerializerImpl::Serialize(const TUserInfoResult& result, NJson::TObject& val) {
        TJsonSerializerImpl::Serialize((TBaseResult&)result, val);
        if (result.PinStatus) {
            val.Add(TOutTokens::PIN_STATUS, *result.PinStatus);
        }
    }

    void TJsonSerializerImpl::Serialize(const TBulkUserInfoResult& result, NJson::TObject& val) {
        NJson::TArray arr(val, TOutTokens::USERS);
        for (const std::pair<TString, TUserInfoResultPtr>& userResult : result.Results) {
            NJson::TObject obj(arr);
            obj.Add(TOutTokens::ID, userResult.first);
            TJsonSerializerImpl::Serialize(*userResult.second, obj);
        }
    }

    void TJsonSerializerImpl::SerializeException(const TExceptionInfo& info, NJson::TObject& val) {
        {
            NJson::TObject obj(val, TOutTokens::EXCEPTION);
            obj.Add(TOutTokens::VALUE, TBlackboxError::StatusStr(info.Status));
            obj.Add(TOutTokens::ID, static_cast<int>(info.Status));
        }
        val.Add(TOutTokens::ERROR, info.ToString());
    }

    void TJsonSerializerImpl::Serialize(const TPhoneBindingsChunk* chunk, NJson::TObject& val) {
        if (nullptr == chunk) {
            return;
        }

        NJson::TArray arr(val, TOutTokens::PHONE_BINDINGS);
        for (const TPhoneBindingsChunk::TPhoneBindingsItem& item : chunk->Bindings) {
            NJson::TObject obj(arr);
            obj.Add(TOutTokens::TYPE, item.Type);
            obj.Add(TOutTokens::NUMBER, item.Number);
            obj.Add(TOutTokens::PHONE_ID, item.PhoneId);
            obj.Add(TOutTokens::UID, item.Uid);
            obj.Add(TOutTokens::BOUND, item.Bound);
            obj.Add(TOutTokens::FLAGS, item.Flags);
        }
    }

    void TJsonSerializerImpl::Serialize(const TBillingFeaturesChunk* featuresChunk, NJson::TObject& val) {
        if (nullptr == featuresChunk) {
            return;
        }

        NJson::TObject rootObj(val, TOutTokens::BILLING_FEATURES);
        for (const auto& [feature, attrs] : featuresChunk->Features) {
            NJson::TObject featureObj(rootObj, feature);
            if (attrs.InTrial) {
                featureObj.Add(TOutTokens::IN_TRIAL, *attrs.InTrial);
            }
            if (attrs.PaidTrial) {
                featureObj.Add(TOutTokens::PAID_TRIAL, *attrs.PaidTrial);
            }
            if (attrs.RegionId) {
                featureObj.Add(TOutTokens::REGION_ID, *attrs.RegionId);
            }
            if (attrs.TrialDuration) {
                featureObj.Add(TOutTokens::TRIAL_DURATION, *attrs.TrialDuration);
            }
            if (attrs.Brand) {
                featureObj.Add(TOutTokens::BRAND, *attrs.Brand);
            }
        }
    }

    void TJsonSerializerImpl::Serialize(const TPlusSubscriberStateChunk* stateChunk, NJson::TObject& val) {
        if (nullptr == stateChunk) {
            return;
        }

        NJson::TObject rootObj(val, TOutTokens::PLUS_SUBSCRIBER_STATE);

        {
            NJson::TArray availableFeatures(rootObj, TOutTokens::AVAILABLE_FEATURES);
            for (const auto& feature : stateChunk->AvailableFeatures) {
                NJson::TObject featureObj(availableFeatures);

                featureObj.Add(TOutTokens::ID, feature.Id);
                featureObj.Add(TOutTokens::END, feature.End.Seconds());
                if (feature.Value) {
                    featureObj.Add(TOutTokens::VALUE, feature.Value);
                }
            }
        }

        {
            NJson::TArray frozenFeatures(rootObj, TOutTokens::FROZEN_FEATURES);
            for (const auto& feature : stateChunk->FrozenFeatures) {
                NJson::TObject featureObj(frozenFeatures);

                featureObj.Add(TOutTokens::ID, feature.Id);
                if (feature.Value) {
                    featureObj.Add(TOutTokens::VALUE, feature.Value);
                }
            }
        }
    }

    void TJsonSerializerImpl::Serialize(const TFamilyInfoChunk* chunk, NJson::TObject& val) {
        if (nullptr == chunk) {
            return;
        }

        NJson::TObject obj(val, TOutTokens::FAMILY_INFO);
        if (chunk->FamilyId.empty()) {
            return;
        }

        obj.Add(TOutTokens::FAMILY_ID, chunk->FamilyId);
        obj.Add(TOutTokens::ADMIN_UID, chunk->AdminUid);
    }

    void TJsonSerializerImpl::Serialize(const TOAuthTokenInfo& info, NJson::TObject& val) {
        if (info.Uid.empty()) {
            val.Add(TOutTokens::UID, nullptr); // 'null' value
        } else {
            val.Add(TOutTokens::UID, info.Uid);
        }

        val.Add(TOutTokens::TOKEN_ID, info.TokenId);
        val.Add(TOutTokens::DEVICE_ID, info.DeviceId);
        val.Add(TOutTokens::DEVICE_NAME, info.GetTokenAttr(TOAuthTokenAttr::DEVICE_NAME));
        val.Add(TOutTokens::SCOPE, info.GetScopesKeywordList());

        val.Add(TOutTokens::CTIME, info.CreateTime);
        val.Add(TOutTokens::ISSUE_TIME, info.IssueTime);
        if (info.NullExpire) {
            val.Add(TOutTokens::EXPIRE_TIME, nullptr); // 'null' value
        } else {
            val.Add(TOutTokens::EXPIRE_TIME, info.ExpireTime);
        }

        val.Add(TOutTokens::IS_TTL_REFRESHABLE, info.IsTtlRefreshable);

        val.Add(TOutTokens::CLIENT_ID, info.GetClientAttr(TOAuthClientAttr::DISPLAY_ID));
        val.Add(TOutTokens::CLIENT_NAME, info.GetClientAttr(TOAuthClientAttr::DEFAULT_TITLE));
        val.Add(TOutTokens::CLIENT_ICON, info.ClientIcon);
        val.Add(TOutTokens::CLIENT_HOMEPAGE, info.GetClientAttr(TOAuthClientAttr::HOMEPAGE));
        val.Add(TOutTokens::CLIENT_CTIME, info.ClientCreatetime);
        val.Add(TOutTokens::CLIENT_IS_YANDEX, info.ClientIsYandex);
        val.Add(TOutTokens::XTOKEN_ID, info.GetTokenAttr(TOAuthTokenAttr::X_TOKEN_ID));

        if (info.IsXTokenTrusted()) {
            val.Add(TOutTokens::IS_XTOKEN_TRUSTED, true);
        }

        val.Add(TOutTokens::META, info.GetTokenAttr(TOAuthTokenAttr::META));

        if (info.HasTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID)) {
            val.Add(TOutTokens::PAYMENT_AUTH_CONTEXT_ID, info.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID));
        }

        if (info.HasTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM)) {
            val.Add(TOutTokens::PAYMENT_AUTH_SCOPE_ADDENDUM, info.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM));
        }
    }

    TString TJsonSerializer::Serialize(const TSessionResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        bool multisession = result.IsMultisession;

        if (multisession || result.Users.empty()) { // global status always present for multisession or if no users parsed (some error)
            TJsonSerializerImpl::SerializeStatus(result, root);
        }

        if (result.Age >= 0) {
            root.Add(TOutTokens::AGE, result.Age);
            long int expiresIn = result.Expires - result.Age;
            if (expiresIn >= 0) {
                root.Add(TOutTokens::EXPIRES_IN, expiresIn);
            }
        }
        if (!result.Ttl.empty()) {
            root.Add(TOutTokens::TTL, result.Ttl);
        }

        if (!result.Users.empty()) { // serialize users if have some
            unsigned defId = result.DefaultId;

            if (multisession) {
                root.Add(TOutTokens::DEFAULT_UID, result.Users.at(defId).first);
                {
                    NJson::TArray arr(root, TOutTokens::USERS);
                    for (const TSessionResult::TUidUserPair& pair : result.Users) {
                        NJson::TObject obj(arr);
                        obj.Add(TOutTokens::ID, pair.first);
                        TJsonSerializerImpl::Serialize(pair.second.get(), obj);
                    }
                }
                root.Add(TOutTokens::ALLOW_MORE_USERS, result.AllowMoreUsers);
            } else { // show only default user in old format
                const TSessionResult::TUidUserPair& pair = result.Users.at(defId);
                root.Add(TOutTokens::ERROR, pair.second->Comment);
                TJsonSerializerImpl::Serialize(pair.second.get(), root);
            }
        }

        TJsonSerializerImpl::Serialize(result.AuthId.get(), root);
        TJsonSerializerImpl::Serialize(result.NewSession.get(), root);
        TJsonSerializerImpl::Serialize(result.ResignedCookies.get(), root);
        TJsonSerializerImpl::Serialize(result.NewSessguards.get(), root);

        if (result.SpecialKind != TSessionResult::None) {
            NJson::TObject kindObj(root, TOutTokens::SPECIAL);
            kindObj.Add(TOutTokens::VALUE, result.SpecialKindName());
            kindObj.Add(TOutTokens::ID, result.SpecialKind);
        }

        if (!result.ConnectionId.empty()) {
            root.Add(TOutTokens::CONNECTION_ID, result.ConnectionId);
        }

        if (!result.LoginId.empty()) {
            root.Add(TOutTokens::LOGIN_ID, result.LoginId);
        }

        if (!result.UserTicket.empty()) {
            root.Add(TOutTokens::USER_TICKET, result.UserTicket);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TBulkUserInfoResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        TJsonSerializerImpl::Serialize(result, root);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TLoginResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        {
            NJson::TObject statusObj(root,
                                     result.ApiVersion > 1 ? TOutTokens::LOGIN_STATUS
                                                           : TOutTokens::STATUS);
            statusObj.Add(TOutTokens::VALUE, result.LoginStatusStr);
            statusObj.Add(TOutTokens::ID, result.LoginStatus);
        }

        if (result.ApiVersion > 1) {
            {
                NJson::TObject passwdStatusObj(root, TOutTokens::PASSWORD_STATUS);
                passwdStatusObj.Add(TOutTokens::VALUE, result.PasswordStatusStr);
                passwdStatusObj.Add(TOutTokens::ID, result.PasswordStatus);
            }

            TLoginResult::EResistPolicy policy = result.ResistPolicy;
            if (policy != TLoginResult::None) {
                NJson::TObject policyObj(root, TOutTokens::BRUTEFORCE_POLICY);
                policyObj.Add(TOutTokens::VALUE, result.ResistPolicyName());
                if (policy == TLoginResult::Delay) {
                    policyObj.Add(TOutTokens::DELAY, result.ResistDelay);
                }
            }
        }

        root.Add(result.ApiVersion > 1 ? TOutTokens::COMMENT : TOutTokens::ERROR, result.Comment);

        TJsonSerializerImpl::Serialize((TBaseResult&)result, root);

        if (result.RestrictSession) {
            root.Add(TOutTokens::RESTRICTED_SESSION, true);
        }

        if (result.TotpCheckTime) {
            root.Add(TOutTokens::TOTP_CHECK_TIME, result.TotpCheckTime);
        }

        if (!result.ConnectionId.empty()) {
            root.Add(TOutTokens::CONNECTION_ID, result.ConnectionId);
        }

        if (result.AllowedSecondSteps) {
            root.Add(TOutTokens::ALLOWED_SECOND_STEPS, *result.AllowedSecondSteps);
        }

        if (result.BadauthCounts) {
            NJson::TObject badauthCountsObj(root, TOutTokens::BADAUTH_COUNTS);

            for (const auto& [name, value, limit] : result.BadauthCounts->Counters) {
                NJson::TObject counterObj(badauthCountsObj, name);
                counterObj.Add(TOutTokens::VALUE, value);
                counterObj.Add(TOutTokens::LIMIT, limit);
            }
        }

        if (result.ScholarSession) {
            root.Add(TOutTokens::IS_SCHOLAR_SESSION, true);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TOAuthResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        TJsonSerializerImpl::Serialize(result.OauthChunk, root);
        TJsonSerializerImpl::Serialize((TBaseResult&)result, root);
        if (!result.ConnectionId.empty()) {
            root.Add(TOutTokens::CONNECTION_ID, result.ConnectionId);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TTypedValueResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        if (TTypedValueResult::STRING == result.Type) {
            root.Add(result.Name, result.Value);
        } else if (TTypedValueResult::NUMBER == result.Type) {
            root.Add(result.Name, IntFromString<i32, 10>(result.Value));
        } else if (TTypedValueResult::BOOL == result.Type) {
            root.Add(result.Name, NUtils::ToBoolean(result.Value));
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TListResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::COUNT, IntToString<10>(result.Data.size()));
        root.Add(TOutTokens::TOTAL_COUNT, IntToString<10>(result.TotalCount));

        NJson::TArray arr(root, result.Name);
        for (const TString& s : result.Data) {
            arr.Add(s);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TTableResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        NJson::TArray arr(root, result.Name);

        const TTableResult::TStringVector& cols = result.Columns;
        const unsigned ncols = cols.size();

        for (const TTableResult::TStringVector& row : result.Rows) {
            NJson::TObject rowObj(arr);
            for (unsigned i = 0; i < ncols; ++i) {
                if (i >= row.size()) {
                    break;
                }
                rowObj.Add(cols[i], row[i]);
            }
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TCreateSessionResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::DEFAULT_UID, result.DefaultUid);
        TJsonSerializerImpl::Serialize(result.NewSession.get(), root);
        TJsonSerializerImpl::Serialize(result.NewSessguards.get(), root);
        TJsonSerializerImpl::Serialize(result.AuthId.get(), root);
        if (!result.LoginId.empty()) {
            root.Add(TOutTokens::LOGIN_ID, result.LoginId);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TLoginOccupationResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        NJson::TObject obj(root, TOutTokens::LOGINS);

        for (const auto& [login, status] : result.Statuses) {
            NJson::TObject loginObj(obj, login);
            loginObj.Add(TOutTokens::STATUS, status.StatusName());
            if (status.Uid) {
                loginObj.Add(TOutTokens::UID, status.Uid);
            }
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TLCookieResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::UID, result.Uid);
        root.Add(TOutTokens::TIMESTAMP, result.Timestamp);
        root.Add(TOutTokens::LOGIN, result.Login);
        root.Add(TOutTokens::DISPLAY_LOGIN, result.DisplayLogin);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TTestPwdHashesResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        NJson::TObject obj(root, TOutTokens::HASHES);

        for (const auto& [hash, status] : result.Data) {
            obj.Add(hash, status);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TTrackResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::UID, result.Uid);
        root.Add(TOutTokens::TRACK_ID, result.TrackId);

        if (result.ContentRapidObj) {
            root.Add(TOutTokens::CONTENT, *result.ContentRapidObj);
            root.Add(TOutTokens::CREATED, result.Created);
            root.Add(TOutTokens::EXPIRED, result.Expired);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TProveKeyDiagResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        {
            bool isTotpCorrect = result.IsTotpCorrect.first;
            const TString secretId = IntToString<10>(isTotpCorrect ? result.IsTotpCorrect.second
                                                                   : result.CorrectTimestamp.second);

            NJson::TObject obj(root, TOutTokens::TOTP_ID);
            obj.Add(TOutTokens::IS_CORRECT, isTotpCorrect);
            obj.Add(TOutTokens::CORRECT_TIMESTAMP,
                    IntToString<10>(result.CorrectTimestamp.first));
            obj.Add(TOutTokens::ID, secretId);
        }
        {
            NJson::TObject obj(root, TOutTokens::PIN_SECRET_ID);
            obj.Add(TOutTokens::IS_CORRECT, result.IsPinSecretCorrect.first);
            obj.Add(TOutTokens::ID, IntToString<10>(result.IsPinSecretCorrect.second));
        }
        {
            NJson::TObject obj(root, TOutTokens::SECRET_ID);
            obj.Add(TOutTokens::IS_CORRECT, result.IsSecretIdCorrect.first);
            obj.Add(TOutTokens::ID, IntToString<10>(result.IsSecretIdCorrect.second));
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TEditTotpResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::ERROR, result.Error());
        root.Add(TOutTokens::SECRET_VALUE, result.SecretValue());
        if (!result.CheckTime().empty()) {
            root.Add(TOutTokens::CHECK_TIME, result.CheckTime());
        }
        if (!result.JunkSecretValue().empty()) {
            root.Add(TOutTokens::JUNK_SECRET_VALUE, result.JunkSecretValue());
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TPwdHistoryResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::PASSWORD_HISTORY_RESULT,
                 result.IsFound() ? TOutTokens::FOUND : TOutTokens::NOT_FOUND);

        if (result.IsFound()) {
            root.Add(TOutTokens::REASON, result.Reason);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TPhoneBindingsResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        TJsonSerializerImpl::Serialize(result.PhoneBindingsChunk.get(), root);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TAllTracksResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        NJson::TArray arr(root, TOutTokens::TRACK);

        for (const std::unique_ptr<TTrackResult>& track : result.Tracks) {
            NJson::TObject trackObj(arr);

            trackObj.Add(TOutTokens::UID, track->Uid);
            trackObj.Add(TOutTokens::TRACK_ID, track->TrackId);

            trackObj.Add(TOutTokens::CONTENT, *track->ContentRapidObj);
            trackObj.Add(TOutTokens::CREATED, track->Created);
            trackObj.Add(TOutTokens::EXPIRED, track->Expired);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TCreateOAuthTokenResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::OAUTH_TOKEN, result.Token);
        root.Add(TOutTokens::TOKEN_ID, result.TokenId);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TCheckRfcTotpResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::STATUS, result.Status);
        if (result.Time) {
            root.Add(TOutTokens::TIME, result.Time);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TCheckSignResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::STATUS, result.Status);
        root.Add(TOutTokens::VALUE, result.Value);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TCheckDeviceSignatureResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::STATUS, result.Status);
        root.Add(TOutTokens::ERROR, result.Error);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TGetDevicePublicKeyResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::STATUS, result.Status);
        root.Add(TOutTokens::VALUE, result.Value);
        root.Add(TOutTokens::VERSION, result.Version);
        root.Add(TOutTokens::OWNER_ID, result.OwnerId);
        root.Add(TOutTokens::ERROR, result.Error);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TCheckHasPlusResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::HAS_PLUS, result.HasPlus);

        {
            NJson::TArray uids(root, TOutTokens::UIDS);
            if (result.Uid) {
                uids.Add(result.Uid);
            }
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TFamilyInfoResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        {
            NJson::TObject status(root, TOutTokens::STATUS);
            status.Add(TOutTokens::VALUE, result.Status.Value);
            if (result.Status.Description) {
                status.Add(TOutTokens::DESCRIPTION, result.Status.Description);
            }
        }

        NJson::TObject family(root, TOutTokens::FAMILY);

        NJson::TObject node(family, result.FamilyId);

        if (result.AdminUid.empty()) {
            return buf;
        }

        node.Add(TOutTokens::ADMIN_UID, result.AdminUid);

        if (result.AllowMoreKids) {
            node.Add(TOutTokens::ALLOW_MORE_KIDS, *result.AllowMoreKids);
        }

        NJson::TArray uids(node, TOutTokens::USERS);
        for (const TFamilyInfoResult::TUser& user : result.Users) {
            NJson::TObject u(uids);
            u.Add(TOutTokens::UID, user.Uid);
            if (user.Place) {
                u.Add(TOutTokens::PLACE, *user.Place);
            }
            if (user.IsChild) {
                u.Add(TOutTokens::IS_CHILD, true);
            }
            if (user.IsKid) {
                u.Add(TOutTokens::IS_KID, true);
            }

            if (user.BaseResult) {
                NJson::TObject info(u, TOutTokens::INFO);
                TJsonSerializerImpl::Serialize(*user.BaseResult, info);
            }
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TFindByPhoneNumbersResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        for (const auto& [number, uids] : result.Data) {
            NJson::TObject numberObj(root, number);
            NJson::TArray uidsArr(numberObj, TOutTokens::UIDS);

            for (const TString& uid : uids) {
                uidsArr.Add(uid);
            }
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TUserTicketResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        TJsonSerializerImpl::Serialize(result, root);

        if (result.ForKid) {
            const TUserTicketResult::TForKid& forKid = *result.ForKid;

            NJson::TObject forKidObj(root, TOutTokens::FOR_KID);
            if (forKid.Error) {
                forKidObj.Add(TOutTokens::ERROR, forKid.Error);
            } else {
                forKidObj.Add(TOutTokens::USER_TICKET, forKid.UserTicket);
            }
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TWebauthnCredentialsResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        NJson::TObject credObj(root, result.CredentialId);
        credObj.Add(TOutTokens::UID, result.Uid);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TGetMaxUidResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::MAX_UID, result.Uid);
        if (result.Pdduid) {
            root.Add(TOutTokens::MAX_PDD_UID, *result.Pdduid);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TGetDebugUserTicketResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::STATUS, result.Status);
        root.Add(TOutTokens::COMMENT, result.Comment);
        root.Add(TOutTokens::USER_TICKET, result.UsetTicket);

        return buf;
    }

    TString TJsonSerializer::Serialize(const TGetOAuthTokensResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        NJson::TArray tokens(root, TOutTokens::TOKENS);
        for (const TOAuthChunk& t : result.Tokens) {
            NJson::TObject token(tokens);
            TJsonSerializerImpl::Serialize(t, token);
        }

        return buf;
    }

    TString TJsonSerializer::Serialize(const TDecreaseSessionidLifetimeResult& result) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        root.Add(TOutTokens::STATUS, result.Status);
        root.Add(TOutTokens::COMMENT, result.Comment);
        root.Add(TOutTokens::NEW_SESSION, result.Session);

        return buf;
    }

    TString TJsonSerializer::SerializeCheckGrants(const TCheckGrantsResult& result,
                                                  const TExceptionInfo* info) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        if (!info) {
            root.Add(TOutTokens::GRANTS_STATUS, TOutTokens::OK);
            return buf;
        }

        root.Add(TOutTokens::GRANTS_STATUS, TBlackboxError::StatusStr(info->Status));
        root.Add(TOutTokens::DESCRIPTION, info->ToString());

        NJson::TArray arr(root, TOutTokens::CHECK_GRANTS_ERRORS);
        for (const TString& s : result.Errors) {
            arr.Add(s);
        }

        return buf;
    }

    TString TJsonSerializer::SerializeException(const TExceptionInfo& info) {
        TString buf;
        NJson::TWriter wr(buf);
        NJson::TObject root(wr);

        TJsonSerializerImpl::SerializeException(info, root);

        return buf;
    }

}
