#include "xml_serializer.h"

#include "account_chunk.h"
#include "all_tracks_result.h"
#include "attributes_chunk.h"
#include "auth_chunk.h"
#include "authid_chunk.h"
#include "billing_features_chunk.h"
#include "check_device_signature_result.h"
#include "check_grants_result.h"
#include "check_has_plus_result.h"
#include "check_rfc_totp_result.h"
#include "check_sign_result.h"
#include "create_oauth_token_result.h"
#include "create_session_result.h"
#include "dbfields_chunk.h"
#include "decrease_sessionid_lifetime_result.h"
#include "display_name_chunk.h"
#include "edit_totp_result.h"
#include "emails_chunk.h"
#include "ext_attrs_chunk.h"
#include "family_info_result.h"
#include "find_by_phone_numbers_result.h"
#include "get_debug_user_ticket_result.h"
#include "get_device_public_key_result.h"
#include "get_max_uid_result.h"
#include "get_oauth_tokens_result.h"
#include "karma_chunk.h"
#include "lcookie_result.h"
#include "list_result.h"
#include "login_result.h"
#include "loginoccupation_result.h"
#include "new_session_chunk.h"
#include "oauth_result.h"
#include "out_tokens.h"
#include "phone_bindings_chunk.h"
#include "phone_bindings_result.h"
#include "phone_operations_chunk.h"
#include "plus_subscriber_state_chunk.h"
#include "prove_key_diag_result.h"
#include "pwdhistory_result.h"
#include "resigned_cookies.h"
#include "sessguard_chunk.h"
#include "session_result.h"
#include "table_result.h"
#include "test_pwd_hashes_result.h"
#include "track_result.h"
#include "typed_value_result.h"
#include "uid_chunk.h"
#include "user_info_result.h"
#include "user_ticket_result.h"
#include "webauthn_credentials_result.h"

#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/experiment.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>

#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/xml/writer.h>
#include <passport/infra/libs/cpp/xml/xml_utils.h>

#include <util/string/cast.h>

using namespace NPassport::NCommon;

namespace NPassport::NBb {
    class TXmlSerializerImpl {
    public:
        static void Serialize(const TAccountChunk* chunk, TString& buf);
        static void Serialize(const TDbFieldsChunk* dbfields, TString& buf);
        static void Serialize(const TAttributesChunk* attrChunk, const TString& nodeName, TString& buf);
        static void Serialize(const TDisplayNameChunk* displayName, TString& buf);
        static void Serialize(const TKarmaChunk* karma, TString& buf);
        static void Serialize(const TUidChunk* uid, TString& buf);
        static void Serialize(const TEmailsChunk* emails, TString& buf);
        static void Serialize(const TAuthIdChunk* authId, TString& buf);
        static void Serialize(const TNewSessionChunk* newsession, TString& buf);
        static void Serialize(const TResignedCookiesChunk* resignedCookies, TString& buf);
        static void Serialize(const TSessguardChunk* newSessguards, TString& buf);
        static void Serialize(const TAuthChunk* auth, TString& buf);
        static void Serialize(const TExtAttrsChunk* extAttrsChunk,
                              const TString& nodeName,
                              const TString& itemName,
                              TString& buf);
        static void Serialize(const TPhoneOperationsChunk* phoneOpsChunk, TString& buf);
        static void Serialize(const TPhoneBindingsChunk* chunk, TString& buf);
        static void Serialize(const TBillingFeaturesChunk* featuresChunk, TString& buf);
        static void Serialize(const TPlusSubscriberStateChunk* stateChunk, TString& buf);
        static void Serialize(const TFamilyInfoChunk* chunk, TString& buf);

        static void Serialize(const TOAuthTokenInfo& info, TString& buf);

        static void Serialize(const TBaseResult& result, TString& buf);
        static void Serialize(const TSessionUser* user, TString& buf);

        static void SerializeStatus(const TSessionResult& result, TString& buf);
        static void Serialize(const TOAuthChunk& chunk, TString& buf);

        static void Serialize(const TUserInfoResult& result, TString& buf);
        static void Serialize(const TBulkUserInfoResult& result, TString& buf);

        static void SerializeException(const TExceptionInfo& info, TString& buf);
    };

    void TXmlSerializerImpl::Serialize(const TAccountChunk* chunk, TString& buf) {
        if (nullptr == chunk) {
            return;
        }
        NXml::NWriter::AddChildNode(TOutTokens::LOGIN, chunk->Login, buf);
        if (TExperiment::Get().IsHavePasswordEnabled) {
            NXml::NWriter::AddChildNode(TOutTokens::HAVE_PASSWORD, chunk->HavePassword, buf);
        }
        if (TExperiment::Get().IsHaveHintEnabled) {
            NXml::NWriter::AddChildNode(TOutTokens::HAVE_HINT, chunk->HaveHint, buf);
        }
        if (chunk->AreAliasesRequired) {
            NXml::NWriter::TNode aliasesNode(TOutTokens::ALIASES, buf);
            for (const auto& [type, value] : chunk->Aliases) {
                NXml::NWriter::TNode alias(TOutTokens::ALIAS, buf);
                alias.AddProp(TOutTokens::TYPE, type);
                alias.SetContent(value);
            }
        }
    }

    void TXmlSerializerImpl::Serialize(const TDbFieldsChunk* dbfields, TString& buf) {
        if (nullptr == dbfields) {
            return;
        }

        for (const TDbFieldsChunk::TField& field : dbfields->Fields) {
            NXml::NWriter::TNode fieldNode(TOutTokens::DBFIELD, buf);
            fieldNode.AddProp(TOutTokens::ID, field.Name);
            fieldNode.SetContent(field.Value);
        }
    }

    void TXmlSerializerImpl::Serialize(const TAttributesChunk* attrChunk, const TString& nodeName, TString& buf) {
        if (nullptr == attrChunk) {
            return;
        }
        NXml::NWriter::TNode rootNode(nodeName, buf);
        for (const auto& [type, value] : attrChunk->Attrs) {
            NXml::NWriter::TNode attrNode(TOutTokens::ATTRIBUTE, buf);
            attrNode.AddProp(TOutTokens::TYPE, type);
            attrNode.SetContent(value);
        }
    }

    void TXmlSerializerImpl::Serialize(const TDisplayNameChunk* displayName, TString& buf) {
        if (nullptr == displayName) {
            return;
        }

        NXml::NWriter::AddChildNode(TOutTokens::REGNAME, displayName->Regname, buf);

        NXml::NWriter::TNode displayNode(TOutTokens::DISPLAYNAME, buf);

        NXml::NWriter::AddChildNode(TOutTokens::NAME, displayName->DisplayName, buf);
        const std::optional<bool>& isDisplayNameEmpty = displayName->IsDisplayNameEmpty;
        if (isDisplayNameEmpty) {
            NXml::NWriter::AddChildNode(TOutTokens::DISPLAY_NAME_EMPTY, *isDisplayNameEmpty, buf);
        }

        const std::optional<TPublicNameData>& publicName = displayName->PublicName;
        if (publicName) {
            NXml::NWriter::AddChildNode(TOutTokens::PUBLIC_NAME, publicName->Name, buf);
            NXml::NWriter::AddChildNode(TOutTokens::HAS_PUBLIC_PROFILE, publicName->HasPublicProfile, buf);
            NXml::NWriter::AddChildNode(TOutTokens::THIRD_PARTY_CAN_USE, publicName->ThirdPartyCanUse, buf);
        }

        {
            NXml::NWriter::TNode displayNode(TOutTokens::AVATAR, buf);
            NXml::NWriter::AddChildNode(TOutTokens::DEFAULT, displayName->Avatar, buf);
            NXml::NWriter::AddChildNode(TOutTokens::EMPTY, displayName->IsAvatarEmpty, buf);
        }

        if (displayName->Verified) {
            NXml::NWriter::AddChildNode(TOutTokens::VERIFIED, TOutTokens::ONE, buf);
        }

        if (displayName->Social) {
            NXml::NWriter::TNode social(TOutTokens::SOCIAL, buf);
            NXml::NWriter::AddChildNode(TOutTokens::PROFILE_ID, displayName->ProfileId, buf);
            NXml::NWriter::AddChildNode(TOutTokens::REDIRECT_TARGET, displayName->Target, buf);
            NXml::NWriter::AddChildNode(TOutTokens::PROVIDER, displayName->Provider, buf);
        }
    }

    void TXmlSerializerImpl::Serialize(const TKarmaChunk* karma, TString& buf) {
        if (nullptr == karma) {
            return;
        }
        {
            NXml::NWriter::TNode karmaNode(TOutTokens::KARMA, buf);
            karmaNode.AddProp(TOutTokens::CONFIRMED, TOutTokens::ZERO);
            if (!karma->Bantime.empty()) {
                karmaNode.AddProp(TOutTokens::ALLOW_UNTIL, karma->Bantime);
            }
            karmaNode.SetContent(karma->Karma);
        }
        {
            NXml::NWriter::TNode karmaStatusNode(TOutTokens::KARMA_STATUS, buf);
            karmaStatusNode.SetContent(karma->RawValue);
        }
    }

    void TXmlSerializerImpl::Serialize(const TUidChunk* uid, TString& buf) {
        if (nullptr == uid) {
            return;
        }
        NXml::NWriter::TNode uidNode(uid->Lite ? TOutTokens::LITEUID : TOutTokens::UID, buf);
        uidNode.AddProp(TOutTokens::HOSTED, uid->Hosted);
        if (uid->Hosted) {
            uidNode.AddProp(TOutTokens::DOMID, uid->DomainId);
            uidNode.AddProp(TOutTokens::DOMAIN_, uid->Domain);
            uidNode.AddProp(TOutTokens::MX, uid->Mx);
            uidNode.AddProp(TOutTokens::DOMAIN_ENA, uid->DomainEna);
            uidNode.AddProp(TOutTokens::CATCH_ALL, uid->CatchAll);
        }
        uidNode.SetContent(uid->Uid);
    }

    void TXmlSerializerImpl::Serialize(const TEmailsChunk* emails, TString& buf) {
        if (nullptr == emails) {
            return;
        }

        NXml::NWriter::TNode addresses(TOutTokens::ADDRESS_LIST, buf);
        for (const TValidatorAddress& addr : emails->Emails) {
            NXml::NWriter::TNode address(TOutTokens::ADDRESS, buf);
            address.AddProp(TOutTokens::VALIDATED, addr.Confirmed);
            address.AddProp(TOutTokens::DEFAULT, addr.Default);
            address.AddProp(TOutTokens::RPOP, addr.Rpop);
            address.AddProp(TOutTokens::SILENT, addr.Silent);
            address.AddProp(TOutTokens::UNSAFE, addr.Unsafe);
            address.AddProp(TOutTokens::NATIVE, addr.Native);
            address.AddProp(TOutTokens::BORN_DATE, addr.Timestamp);
            address.SetContent(addr.Address);
        }
    }

    void TXmlSerializerImpl::Serialize(const TAuthIdChunk* authId, TString& buf) {
        if (nullptr == authId) {
            return;
        }
        NXml::NWriter::TNode authidNode(TOutTokens::AUTHID, buf);
        authidNode.AddProp(TOutTokens::TIME, authId->Timestamp);
        authidNode.AddProp(TOutTokens::IP, authId->Ip);
        authidNode.AddProp(TOutTokens::HOST, authId->Host);
        authidNode.SetContent(authId->Id);
    }

    void TXmlSerializerImpl::Serialize(const TNewSessionChunk* newsession, TString& buf) {
        if (nullptr == newsession) {
            return;
        }

        {
            NXml::NWriter::TNode session(TOutTokens::NEW_SESSION, buf);
            session.AddProp(TOutTokens::DOMAIN_, newsession->Domain);
            session.AddProp(TOutTokens::EXPIRES, newsession->Expires);
            session.AddProp(TOutTokens::HTTP_ONLY, TOutTokens::ONE);
            session.AddProp(TOutTokens::SECURE, TOutTokens::ONE);

            session.SetContent(newsession->Value);
        }

        {
            NXml::NWriter::TNode sslsession(TOutTokens::NEW_SSLSESSION, buf);
            sslsession.AddProp(TOutTokens::DOMAIN_, newsession->Domain);
            sslsession.AddProp(TOutTokens::EXPIRES, newsession->Expires);
            sslsession.AddProp(TOutTokens::HTTP_ONLY, TOutTokens::ONE);
            sslsession.AddProp(TOutTokens::SECURE, TOutTokens::ONE);

            sslsession.SetContent(newsession->LegacyValue);
        }
    }

    void TXmlSerializerImpl::Serialize(const TResignedCookiesChunk* resignedCookies, TString& buf) {
        if (nullptr == resignedCookies) {
            return;
        }

        NXml::NWriter::TNode obj(TOutTokens::RESIGNED_COOKIES, buf);
        for (const auto& [domain, cookie] : resignedCookies->Cookies) {
            NXml::NWriter::TNode domainObj(TOutTokens::DOMAIN_, buf);
            domainObj.AddProp(TOutTokens::ID, domain);
            Serialize(cookie.get(), buf);
        }
    }

    void TXmlSerializerImpl::Serialize(const TSessguardChunk* newSessguards, TString& buf) {
        if (nullptr == newSessguards) {
            return;
        }

        NXml::NWriter::TNode rootObj(TOutTokens::NEW_SESSGUARDS, buf);
        for (const auto& [host, cookie] : newSessguards->Cookies) {
            NXml::NWriter::TNode obj(TOutTokens::HOST, buf);
            obj.AddProp(TOutTokens::ID, host);
            NXml::NWriter::TNode sessguard(TOutTokens::SESSGUARD, buf);
            sessguard.AddProp(TOutTokens::DOMAIN_, cookie.Domain);
            sessguard.AddProp(TOutTokens::EXPIRES, cookie.Expires);
            sessguard.SetContent(cookie.Value);
        }
    }

    void TXmlSerializerImpl::Serialize(const TAuthChunk* auth, TString& buf) {
        if (nullptr == auth) {
            return;
        }

        NXml::NWriter::TNode authNode(TOutTokens::AUTH, buf);
        NXml::NWriter::AddChildNode(TOutTokens::PASSWORD_VERIFICATION_AGE, auth->PasswordVerificationAge, buf);
        NXml::NWriter::AddChildNode(TOutTokens::SECURE, TOutTokens::ONE, buf);
        if (TExperiment::Get().IsPddPartnerTokenEnabled) {
            NXml::NWriter::AddChildNode(TOutTokens::PARTNER_PDD_TOKEN, TOutTokens::ZERO, buf);
        }
        if (!auth->ProfileId.empty()) {
            NXml::NWriter::TNode social(TOutTokens::SOCIAL, buf);
            NXml::NWriter::AddChildNode(TOutTokens::PROFILE_ID, auth->ProfileId, buf);
        }
        if (auth->Scholar) {
            NXml::NWriter::AddChildNode(TOutTokens::IS_SCHOLAR_SESSION, TOutTokens::ONE, buf);
        }
    }

    void TXmlSerializerImpl::Serialize(const TExtAttrsChunk* extAttrsChunk,
                                       const TString& nodeName,
                                       const TString& itemName,
                                       TString& buf) {
        if (nullptr == extAttrsChunk) {
            return;
        }

        NXml::NWriter::TNode rootNode(nodeName, buf);
        for (const auto& [id, attrs] : extAttrsChunk->Attrs) {
            NXml::NWriter::TNode itemNode(itemName, buf);
            itemNode.AddProp(TOutTokens::ID, id);

            for (const auto& [type, value] : attrs) {
                NXml::NWriter::TNode attr(TOutTokens::ATTRIBUTE, buf);
                attr.AddProp(TOutTokens::TYPE, type);
                attr.SetContent(value);
            }
        }
    }

    void TXmlSerializerImpl::Serialize(const TPhoneOperationsChunk* phoneOpsChunk, TString& buf) {
        if (nullptr == phoneOpsChunk) {
            return;
        }

        NXml::NWriter::TNode rootNode(TOutTokens::PHONE_OPERATIONS, buf);
        for (const auto& [id, value] : phoneOpsChunk->Ops) {
            NXml::NWriter::TNode opNode(TOutTokens::OPERATION, buf);
            opNode.AddProp(TOutTokens::ID, id);
            opNode.SetContent(value);
        }
    }

    void TXmlSerializerImpl::Serialize(const TPhoneBindingsChunk* chunk, TString& buf) {
        if (nullptr == chunk) {
            return;
        }

        NXml::NWriter::TNode root(TOutTokens::PHONE_BINDINGS, buf);
        for (const TPhoneBindingsChunk::TPhoneBindingsItem& item : chunk->Bindings) {
            NXml::NWriter::TNode node(TOutTokens::ITEM, buf);
            node.AddProp(TOutTokens::TYPE, item.Type);
            node.AddProp(TOutTokens::NUMBER, item.Number);
            node.AddProp(TOutTokens::PHONE_ID, item.PhoneId);
            node.AddProp(TOutTokens::UID, item.Uid);
            node.AddProp(TOutTokens::BOUND, item.Bound);
            node.AddProp(TOutTokens::FLAGS, item.Flags);
        }
    }

    void TXmlSerializerImpl::Serialize(const TBillingFeaturesChunk* featuresChunk, TString& buf) {
        if (nullptr == featuresChunk) {
            return;
        }

        NXml::NWriter::TNode rootNode(TOutTokens::BILLING_FEATURES, buf);
        for (const auto& [feature, attrs] : featuresChunk->Features) {
            NXml::NWriter::TNode itemNode(TOutTokens::FEATURE, buf);
            itemNode.AddProp(TOutTokens::NAME, feature);

            if (attrs.InTrial) {
                itemNode.AddProp(TOutTokens::IN_TRIAL, *attrs.InTrial);
            }
            if (attrs.PaidTrial) {
                itemNode.AddProp(TOutTokens::PAID_TRIAL, *attrs.PaidTrial);
            }
            if (attrs.RegionId) {
                itemNode.AddProp(TOutTokens::REGION_ID, *attrs.RegionId);
            }
            if (attrs.TrialDuration) {
                itemNode.AddProp(TOutTokens::TRIAL_DURATION, *attrs.TrialDuration);
            }
            if (attrs.Brand) {
                itemNode.AddProp(TOutTokens::BRAND, *attrs.Brand);
            }
        }
    }

    void TXmlSerializerImpl::Serialize(const TPlusSubscriberStateChunk* stateChunk, TString& buf) {
        if (nullptr == stateChunk) {
            return;
        }

        NXml::NWriter::TNode rootNode(TOutTokens::PLUS_SUBSCRIBER_STATE, buf);

        {
            NXml::NWriter::TNode availableFeaturesNode(TOutTokens::AVAILABLE_FEATURES, buf);
            for (const auto& feature : stateChunk->AvailableFeatures) {
                NXml::NWriter::TNode featureNode(TOutTokens::FEATURE, buf);

                featureNode.AddProp(TOutTokens::ID, feature.Id);
                featureNode.AddProp(TOutTokens::END, feature.End.Seconds());
                if (feature.Value) {
                    featureNode.AddProp(TOutTokens::VALUE, feature.Value);
                }
            }
        }

        {
            NXml::NWriter::TNode frozenFeaturesNode(TOutTokens::FROZEN_FEATURES, buf);
            for (const auto& feature : stateChunk->FrozenFeatures) {
                NXml::NWriter::TNode featureNode(TOutTokens::FEATURE, buf);

                featureNode.AddProp(TOutTokens::ID, feature.Id);
                if (feature.Value) {
                    featureNode.AddProp(TOutTokens::VALUE, feature.Value);
                }
            }
        }
    }

    void TXmlSerializerImpl::Serialize(const TFamilyInfoChunk* chunk, TString& buf) {
        if (nullptr == chunk) {
            return;
        }

        NXml::NWriter::TNode node(TOutTokens::FAMILY_INFO, buf);
        if (chunk->FamilyId.empty()) {
            return;
        }

        NXml::NWriter::AddChildNode(TOutTokens::FAMILY_ID, chunk->FamilyId, buf);
        NXml::NWriter::AddChildNode(TOutTokens::ADMIN_UID, chunk->AdminUid, buf);
    }

    void TXmlSerializerImpl::Serialize(const TOAuthTokenInfo& info, TString& buf) {
        NXml::NWriter::AddChildNode(TOutTokens::UID, info.Uid, buf);

        NXml::NWriter::AddChildNode(TOutTokens::TOKEN_ID, info.TokenId, buf);
        NXml::NWriter::AddChildNode(TOutTokens::DEVICE_ID, info.DeviceId, buf);
        NXml::NWriter::AddChildNode(TOutTokens::DEVICE_NAME, info.GetTokenAttr(TOAuthTokenAttr::DEVICE_NAME), buf);
        NXml::NWriter::AddChildNode(TOutTokens::SCOPE, info.GetScopesKeywordList(), buf);

        NXml::NWriter::AddChildNode(TOutTokens::CTIME, info.CreateTime, buf);
        NXml::NWriter::AddChildNode(TOutTokens::ISSUE_TIME, info.IssueTime, buf);
        NXml::NWriter::AddChildNode(TOutTokens::EXPIRE_TIME, info.ExpireTime, buf);

        NXml::NWriter::AddChildNode(TOutTokens::IS_TTL_REFRESHABLE, info.IsTtlRefreshable, buf);

        NXml::NWriter::AddChildNode(TOutTokens::CLIENT_ID, info.GetClientAttr(TOAuthClientAttr::DISPLAY_ID), buf);
        NXml::NWriter::AddChildNode(TOutTokens::CLIENT_NAME, info.GetClientAttr(TOAuthClientAttr::DEFAULT_TITLE), buf);
        NXml::NWriter::AddChildNode(TOutTokens::CLIENT_ICON, info.ClientIcon, buf);
        NXml::NWriter::AddChildNode(TOutTokens::CLIENT_HOMEPAGE, info.GetClientAttr(TOAuthClientAttr::HOMEPAGE), buf);
        NXml::NWriter::AddChildNode(TOutTokens::CLIENT_CTIME, info.ClientCreatetime, buf);
        NXml::NWriter::AddChildNode(TOutTokens::CLIENT_IS_YANDEX, info.ClientIsYandex, buf);
        NXml::NWriter::AddChildNode(TOutTokens::XTOKEN_ID, info.GetTokenAttr(TOAuthTokenAttr::X_TOKEN_ID), buf);

        if (info.IsXTokenTrusted()) {
            NXml::NWriter::AddChildNode(TOutTokens::IS_XTOKEN_TRUSTED, TOutTokens::ONE, buf);
        }

        NXml::NWriter::AddChildNode(TOutTokens::META, info.GetTokenAttr(TOAuthTokenAttr::META), buf);

        if (info.HasTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID)) {
            NXml::NWriter::AddChildNode(TOutTokens::PAYMENT_AUTH_CONTEXT_ID, info.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_CONTEXT_ID), buf);
        }

        if (info.HasTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM)) {
            NXml::NWriter::AddChildNode(TOutTokens::PAYMENT_AUTH_SCOPE_ADDENDUM, info.GetTokenAttr(TOAuthTokenAttr::PAYMENT_AUTH_SCOPE_ADDENDUM), buf);
        }
    }

    void TXmlSerializerImpl::Serialize(const TSessionUser* user, TString& buf) {
        if (nullptr == user) {
            return;
        }
        if (user->Status) {
            NXml::NWriter::TNode node(TOutTokens::STATUS, buf);
            node.AddProp(TOutTokens::ID, (int)*user->Status);
            node.SetContent(user->StatusStr());
        }
        if (user->IsRegCompletionRecommended) {
            NXml::NWriter::AddChildNode(TOutTokens::IS_REG_COMPLETION_RECOMMENDED, user->IsRegCompletionRecommended.value(), buf);
        }
        Serialize((TBaseResult&)*user, buf);
        Serialize(user->Auth.get(), buf);
    }

    void TXmlSerializerImpl::SerializeStatus(const TSessionResult& result, TString& buf) {
        if (result.Status) {
            NXml::NWriter::TNode node(TOutTokens::STATUS, buf);
            node.AddProp(TOutTokens::ID, static_cast<int>(*result.Status));
            node.SetContent(result.StatusStr());
        }
        NXml::NWriter::AddChildNode(TOutTokens::ERROR, result.Comment, buf);
    }

    void TXmlSerializerImpl::Serialize(const TOAuthChunk& chunk, TString& buf) {
        if (chunk.TokenInfo) {
            NXml::NWriter::TNode oauth(TOutTokens::OAUTH_CAPITAL, buf);

            TXmlSerializerImpl::Serialize(*chunk.TokenInfo, buf);

            TXmlSerializerImpl::Serialize(chunk.TokenAttrs.get(), TOutTokens::TOKEN_ATTRIBUTES, buf);
            TXmlSerializerImpl::Serialize(chunk.ClientAttrs.get(), TOutTokens::CLIENT_ATTRIBUTES, buf);
        }

        {
            NXml::NWriter::TNode node(TOutTokens::STATUS, buf);
            node.AddProp(TOutTokens::ID, static_cast<int>(chunk.Status.Status()));
            node.SetContent(chunk.Status.AsString());
        }

        NXml::NWriter::AddChildNode(TOutTokens::ERROR, chunk.Comment, buf);
        if (!chunk.LoginId.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::LOGIN_ID, chunk.LoginId, buf);
        }
    }

    void TXmlSerializerImpl::Serialize(const TBaseResult& result, TString& buf) {
        Serialize(result.Uid.get(), buf);
        Serialize(result.Account.get(), buf);
        Serialize(result.Karma.get(), buf);
        Serialize(result.DisplayName.get(), buf);
        Serialize(result.Dbfields.get(), buf);
        Serialize(result.Attributes.get(), TOutTokens::ATTRIBUTES, buf);
        Serialize(result.Emails.get(), buf);
        Serialize(result.PhoneAttrs.get(), TOutTokens::PHONES, TOutTokens::PHONE, buf);
        Serialize(result.EmailAttrs.get(), TOutTokens::EMAILS, TOutTokens::EMAIL, buf);
        Serialize(result.WebauthnAttrs.get(), TOutTokens::WEBAUTHN_CREDENTIALS, TOutTokens::ITEM, buf);
        Serialize(result.PhoneOperations.get(), buf);
        Serialize(result.PhoneBindings.get(), buf);
        Serialize(result.BillingFeatures.get(), buf);
        Serialize(result.PlusSubscriberState.get(), buf);
        Serialize(result.FamilyInfo.get(), buf);
        if (!result.UserTicket.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::USER_TICKET, result.UserTicket, buf);
        }

        if (result.PublicId) {
            NXml::NWriter::AddChildNode(TOutTokens::PUBLIC_ID, *result.PublicId, buf);
        }
    }

    void TXmlSerializerImpl::Serialize(const TUserInfoResult& result, TString& buf) {
        TXmlSerializerImpl::Serialize((TBaseResult&)result, buf);
        if (result.PinStatus) {
            NXml::NWriter::AddChildNode(TOutTokens::PIN_STATUS, *result.PinStatus, buf);
        }
    }

    void TXmlSerializerImpl::Serialize(const TBulkUserInfoResult& result, TString& buf) {
        if (1 == result.Results.size()) {
            TXmlSerializerImpl::Serialize(*result.Results[0].second, buf);
            return;
        }

        for (const std::pair<TString, TUserInfoResultPtr>& userResult : result.Results) {
            NXml::NWriter::TNode user(TOutTokens::USER, buf);
            user.AddProp(TOutTokens::ID, userResult.first);
            TXmlSerializerImpl::Serialize(*userResult.second, buf);
        }
    }

    void TXmlSerializerImpl::SerializeException(const TExceptionInfo& info, TString& buf) {
        {
            NXml::NWriter::TNode exception(TOutTokens::EXCEPTION, buf);
            exception.AddProp(TOutTokens::ID, (int)info.Status);
            exception.SetContent(TBlackboxError::StatusStr(info.Status));
        }
        NXml::NWriter::AddChildNode(TOutTokens::ERROR, info.ToString(), buf);
    }

    TString TXmlSerializer::Serialize(const TOAuthResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        TXmlSerializerImpl::Serialize(result.OauthChunk, buf);
        TXmlSerializerImpl::Serialize((TBaseResult&)result, buf);
        if (!result.ConnectionId.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::CONNECTION_ID, result.ConnectionId, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TSessionResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        bool multisession = result.IsMultisession;

        if (multisession || result.Users.empty()) { // global status always present for multisession or if no users parsed (some error)
            TXmlSerializerImpl::SerializeStatus(result, buf);
        }

        if (result.Age >= 0) {
            NXml::NWriter::AddChildNode(TOutTokens::AGE, result.Age, buf);
            long int expiresIn = result.Expires - result.Age;
            if (expiresIn >= 0) {
                NXml::NWriter::AddChildNode(TOutTokens::EXPIRES_IN, expiresIn, buf);
            }
        }
        if (!result.Ttl.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::TTL, result.Ttl, buf);
        }

        if (!result.Users.empty()) { // serialize users if have some
            unsigned defId = result.DefaultId;

            if (multisession) {
                NXml::NWriter::AddChildNode(TOutTokens::DEFAULT_UID, result.Users.at(defId).first, buf);
                for (const TSessionResult::TUidUserPair& pair : result.Users) {
                    NXml::NWriter::TNode userNode(TOutTokens::USER, buf);
                    userNode.AddProp(TOutTokens::ID, pair.first);
                    TXmlSerializerImpl::Serialize(pair.second.get(), buf);
                }
                NXml::NWriter::AddChildNode(TOutTokens::ALLOW_MORE_USERS, result.AllowMoreUsers, buf);
            } else { // show only default user in old format
                const TSessionResult::TUidUserPair& pair = result.Users.at(defId);
                NXml::NWriter::AddChildNode(TOutTokens::ERROR, pair.second->Comment, buf);
                TXmlSerializerImpl::Serialize(pair.second.get(), buf);
            }
        }

        TXmlSerializerImpl::Serialize(result.AuthId.get(), buf);
        TXmlSerializerImpl::Serialize(result.NewSession.get(), buf);
        TXmlSerializerImpl::Serialize(result.ResignedCookies.get(), buf);
        TXmlSerializerImpl::Serialize(result.NewSessguards.get(), buf);

        if (result.SpecialKind != TSessionResult::None) {
            NXml::NWriter::TNode node(TOutTokens::SPECIAL, buf);
            node.AddProp(TOutTokens::ID, (int)result.SpecialKind);
            node.SetContent(result.SpecialKindName());
        }
        if (!result.ConnectionId.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::CONNECTION_ID, result.ConnectionId, buf);
        }
        if (!result.LoginId.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::LOGIN_ID, result.LoginId, buf);
        }
        if (!result.UserTicket.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::USER_TICKET, result.UserTicket, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TBulkUserInfoResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        TXmlSerializerImpl::Serialize(result, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TLoginResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        {
            NXml::NWriter::TNode status(result.ApiVersion > 1 ? TOutTokens::LOGIN_STATUS : TOutTokens::STATUS, buf);
            status.AddProp(TOutTokens::ID, result.LoginStatus);
            status.SetContent(result.LoginStatusStr);
        }

        if (result.ApiVersion > 1) {
            {
                NXml::NWriter::TNode status(TOutTokens::PASSWORD_STATUS, buf);
                status.AddProp(TOutTokens::ID, result.PasswordStatus);
                status.SetContent(result.PasswordStatusStr);
            }

            TLoginResult::EResistPolicy policy = result.ResistPolicy;
            if (policy != TLoginResult::None) {
                NXml::NWriter::TNode policyNode(TOutTokens::BRUTEFORCE_POLICY, buf);
                const TString& nodeName = result.ResistPolicyName();
                if (policy == TLoginResult::Delay) {
                    NXml::NWriter::AddChildNode(nodeName, result.ResistDelay, buf);
                } else {
                    NXml::NWriter::AddChildNode(nodeName, buf);
                }
                NXml::NWriter::AddChildNode(TOutTokens::LEVEL, TOutTokens::ZERO, buf);
            }
        }
        NXml::NWriter::AddChildNode(result.ApiVersion > 1 ? TOutTokens::COMMENT : TOutTokens::ERROR,
                                    result.Comment,
                                    buf);
        TXmlSerializerImpl::Serialize((TBaseResult&)result, buf);
        if (result.RestrictSession) {
            NXml::NWriter::AddChildNode(TOutTokens::RESTRICTED_SESSION, TOutTokens::ONE, buf);
        }
        if (result.TotpCheckTime) {
            NXml::NWriter::AddChildNode(TOutTokens::TOTP_CHECK_TIME, result.TotpCheckTime, buf);
        }
        if (!result.ConnectionId.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::CONNECTION_ID, result.ConnectionId, buf);
        }

        if (result.AllowedSecondSteps) {
            NXml::NWriter::TNode steps(TOutTokens::ALLOWED_SECOND_STEPS, buf);
            steps.SetContent(*result.AllowedSecondSteps);
        }

        if (result.BadauthCounts) {
            NXml::NWriter::TNode steps(TOutTokens::BADAUTH_COUNTS, buf);

            for (const auto& [Name, Value, Limit] : result.BadauthCounts->Counters) {
                NXml::NWriter::TNode elemNode(TOutTokens::ITEM, buf);
                elemNode.AddProp(TOutTokens::NAME, Name);
                elemNode.AddProp(TOutTokens::VALUE, Value);
                elemNode.AddProp(TOutTokens::LIMIT, Limit);
            }
        }

        if (result.ScholarSession) {
            NXml::NWriter::AddChildNode(TOutTokens::IS_SCHOLAR_SESSION, TOutTokens::ONE, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TTypedValueResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(result.Name, result.Value, buf);

        return buf;
    }

    TString TXmlSerializer::SerializeException(const TExceptionInfo& info) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        TXmlSerializerImpl::SerializeException(info, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TListResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::ELEM_COUNT, result.Data.size(), buf);
        NXml::NWriter::AddChildNode(TOutTokens::TOTAL_COUNT, result.TotalCount, buf);
        NXml::NWriter::TNode node(result.Name, buf);

        for (const TString& str : result.Data) {
            NXml::NWriter::TNode item(TOutTokens::ITEM, buf);
            item.AddProp(result.PropertyName, str);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TTableResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::TNode node(result.Name, buf);

        const TTableResult::TStringVector& cols = result.Columns;
        const unsigned ncols = cols.size();

        for (const TTableResult::TStringVector& vec : result.Rows) {
            NXml::NWriter::TNode item(TOutTokens::ITEM, buf);
            for (unsigned i = 0; i < ncols; ++i) {
                if (i >= vec.size()) {
                    break;
                }
                node.AddProp(cols[i], vec[i]);
            }
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TCreateSessionResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::DEFAULT_UID, result.DefaultUid, buf);
        TXmlSerializerImpl::Serialize(result.NewSession.get(), buf);
        TXmlSerializerImpl::Serialize(result.NewSessguards.get(), buf);
        TXmlSerializerImpl::Serialize(result.AuthId.get(), buf);
        if (!result.LoginId.empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::LOGIN_ID, result.LoginId, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TLoginOccupationResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        for (const auto& [login, status] : result.Statuses) {
            NXml::NWriter::TNode elemNode(TOutTokens::LOGIN, buf);
            elemNode.AddProp(TOutTokens::NAME, login);
            if (status.Uid) {
                elemNode.AddProp(TOutTokens::UID, status.Uid);
            }
            elemNode.SetContent(status.StatusName());
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TLCookieResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::UID, result.Uid, buf);
        NXml::NWriter::AddChildNode(TOutTokens::TIMESTAMP, result.Timestamp, buf);
        NXml::NWriter::AddChildNode(TOutTokens::LOGIN, result.Login, buf);
        NXml::NWriter::AddChildNode(TOutTokens::DISPLAY_LOGIN, result.DisplayLogin, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TTestPwdHashesResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        for (const auto& [hash, status] : result.Data) {
            NXml::NWriter::TNode elemNode(TOutTokens::HASH, buf);
            elemNode.AddProp(TOutTokens::VALUE, hash);
            elemNode.SetContent(status);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TTrackResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::UID, result.Uid, buf);
        NXml::NWriter::AddChildNode(TOutTokens::TRACK_ID, result.TrackId, buf);

        if (result.ContentRapidObj) {
            NXml::NWriter::AddChildNode(TOutTokens::CREATED, result.Created, buf);
            NXml::NWriter::AddChildNode(TOutTokens::EXPIRED, result.Expired, buf);
            NXml::NWriter::AddChildNode(TOutTokens::CONTENT, result.Content, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TProveKeyDiagResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        {
            bool isTotpCorrect = result.IsTotpCorrect.first;
            const ui64 secretId = isTotpCorrect ? result.IsTotpCorrect.second
                                                : result.CorrectTimestamp.second;

            NXml::NWriter::TNode totpId(TOutTokens::TOTP_ID, buf);
            NXml::NWriter::AddChildNode(TOutTokens::IS_CORRECT, isTotpCorrect, buf);
            NXml::NWriter::AddChildNode(TOutTokens::CORRECT_TIMESTAMP, result.CorrectTimestamp.first, buf);
            NXml::NWriter::AddChildNode(TOutTokens::ID, secretId, buf);
        }
        {
            NXml::NWriter::TNode pinSecretId(TOutTokens::PIN_SECRET_ID, buf);
            NXml::NWriter::AddChildNode(TOutTokens::IS_CORRECT, result.IsPinSecretCorrect.first, buf);
            NXml::NWriter::AddChildNode(TOutTokens::ID, result.IsPinSecretCorrect.second, buf);
        }
        {
            NXml::NWriter::TNode secretId(TOutTokens::SECRET_ID, buf);
            NXml::NWriter::AddChildNode(TOutTokens::IS_CORRECT, result.IsSecretIdCorrect.first, buf);
            NXml::NWriter::AddChildNode(TOutTokens::ID, result.IsSecretIdCorrect.second, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TEditTotpResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::ERROR, result.Error(), buf);
        NXml::NWriter::AddChildNode(TOutTokens::SECRET_VALUE, result.SecretValue(), buf);
        if (!result.CheckTime().empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::CHECK_TIME, result.CheckTime(), buf);
        }
        if (!result.JunkSecretValue().empty()) {
            NXml::NWriter::AddChildNode(TOutTokens::JUNK_SECRET_VALUE, result.JunkSecretValue(), buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TPwdHistoryResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::PASSWORD_HISTORY_RESULT,
                                    result.IsFound() ? TOutTokens::FOUND
                                                     : TOutTokens::NOT_FOUND,
                                    buf);

        if (result.IsFound()) {
            NXml::NWriter::AddChildNode(TOutTokens::REASON, result.Reason, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TPhoneBindingsResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        TXmlSerializerImpl::Serialize(result.PhoneBindingsChunk.get(), buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TAllTracksResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        for (const std::unique_ptr<TTrackResult>& track : result.Tracks) {
            NXml::NWriter::TNode trackHolder(TOutTokens::TRACK, buf);

            NXml::NWriter::AddChildNode(TOutTokens::UID, track->Uid, buf);
            NXml::NWriter::AddChildNode(TOutTokens::TRACK_ID, track->TrackId, buf);
            NXml::NWriter::AddChildNode(TOutTokens::CREATED, track->Created, buf);
            NXml::NWriter::AddChildNode(TOutTokens::EXPIRED, track->Expired, buf);
            NXml::NWriter::AddChildNode(TOutTokens::CONTENT, track->Content, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TCreateOAuthTokenResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::OAUTH_TOKEN, result.Token, buf);
        NXml::NWriter::AddChildNode(TOutTokens::TOKEN_ID, result.TokenId, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TCheckRfcTotpResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::STATUS, result.Status, buf);
        if (result.Time) {
            NXml::NWriter::AddChildNode(TOutTokens::TIME, result.Time, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TCheckSignResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::STATUS, result.Status, buf);
        NXml::NWriter::AddChildNode(TOutTokens::VALUE, result.Value, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TCheckDeviceSignatureResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::STATUS, result.Status, buf);
        NXml::NWriter::AddChildNode(TOutTokens::ERROR, result.Error, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TGetDevicePublicKeyResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::STATUS, result.Status, buf);
        NXml::NWriter::AddChildNode(TOutTokens::VALUE, result.Value, buf);
        NXml::NWriter::AddChildNode(TOutTokens::VERSION, result.Version, buf);
        NXml::NWriter::AddChildNode(TOutTokens::OWNER_ID, result.OwnerId, buf);
        NXml::NWriter::AddChildNode(TOutTokens::ERROR, result.Error, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TCheckHasPlusResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::HAS_PLUS, result.HasPlus, buf);
        if (result.Uid) {
            NXml::NWriter::AddChildNode(TOutTokens::UID, result.Uid, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TFamilyInfoResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        {
            NXml::NWriter::TNode status(TOutTokens::STATUS, buf);
            NXml::NWriter::AddChildNode(TOutTokens::VALUE, result.Status.Value, buf);
            if (result.Status.Description) {
                NXml::NWriter::AddChildNode(TOutTokens::DESCRIPTION, result.Status.Description, buf);
            }
        }

        NXml::NWriter::TNode family(TOutTokens::FAMILY, buf);
        family.AddProp(TOutTokens::ID, result.FamilyId);

        if (result.AdminUid.empty()) {
            return buf;
        }

        NXml::NWriter::AddChildNode(TOutTokens::ADMIN_UID, result.AdminUid, buf);

        if (result.AllowMoreKids) {
            NXml::NWriter::AddChildNode(TOutTokens::ALLOW_MORE_KIDS, *result.AllowMoreKids, buf);
        }

        for (const TFamilyInfoResult::TUser& user : result.Users) {
            NXml::NWriter::TNode node(TOutTokens::USER, buf);
            node.AddProp(TOutTokens::UID, user.Uid);
            if (user.Place) {
                node.AddProp(TOutTokens::PLACE, *user.Place);
            }
            if (user.IsChild) {
                node.AddProp(TOutTokens::IS_CHILD, true);
            }
            if (user.IsKid) {
                node.AddProp(TOutTokens::IS_KID, true);
            }

            if (user.BaseResult) {
                TXmlSerializerImpl::Serialize(*user.BaseResult, buf);
            }
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TFindByPhoneNumbersResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        for (const auto& [number, uids] : result.Data) {
            NXml::NWriter::TNode numberNode(TOutTokens::NUMBER, buf);
            numberNode.AddProp(TOutTokens::ID, number);

            NXml::NWriter::TNode node(TOutTokens::UIDS, buf);

            for (const TString& uid : uids) {
                NXml::NWriter::TNode item(TOutTokens::ITEM, buf);
                item.AddProp(TOutTokens::ID, uid);
            }
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TUserTicketResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        TXmlSerializerImpl::Serialize(result, buf);

        if (result.ForKid) {
            const TUserTicketResult::TForKid& forKid = *result.ForKid;

            NXml::NWriter::TNode forKidObj(TOutTokens::FOR_KID, buf);
            if (forKid.Error) {
                NXml::NWriter::AddChildNode(TOutTokens::ERROR, forKid.Error, buf);
            } else {
                NXml::NWriter::AddChildNode(TOutTokens::USER_TICKET, forKid.UserTicket, buf);
            }
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TWebauthnCredentialsResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::TNode credNode(TOutTokens::ITEM, buf);
        credNode.AddProp(TOutTokens::ID, result.CredentialId);
        credNode.AddProp(TOutTokens::UID, result.Uid);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TGetMaxUidResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::MAX_UID, result.Uid, buf);
        if (result.Pdduid) {
            NXml::NWriter::AddChildNode(TOutTokens::MAX_PDD_UID, *result.Pdduid, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TGetDebugUserTicketResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::STATUS, result.Status, buf);
        NXml::NWriter::AddChildNode(TOutTokens::COMMENT, result.Comment, buf);
        NXml::NWriter::AddChildNode(TOutTokens::USER_TICKET, result.UsetTicket, buf);

        return buf;
    }

    TString TXmlSerializer::Serialize(const TGetOAuthTokensResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::TNode tokens(TOutTokens::TOKENS, buf);
        for (const TOAuthChunk& t : result.Tokens) {
            NXml::NWriter::TNode token(TOutTokens::ITEM, buf);
            TXmlSerializerImpl::Serialize(t, buf);
        }

        return buf;
    }

    TString TXmlSerializer::Serialize(const TDecreaseSessionidLifetimeResult& result) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        NXml::NWriter::AddChildNode(TOutTokens::STATUS, result.Status, buf);
        NXml::NWriter::AddChildNode(TOutTokens::COMMENT, result.Comment, buf);
        NXml::NWriter::AddChildNode(TOutTokens::NEW_SESSION, result.Session, buf);

        return buf;
    }

    TString TXmlSerializer::SerializeCheckGrants(const TCheckGrantsResult& result,
                                                 const TExceptionInfo* info) {
        TString buf = NXml::NWriter::SerializeHeader();
        NXml::NWriter::TNode doc(TOutTokens::DOC, buf);

        if (!info) {
            NXml::NWriter::AddChildNode(TOutTokens::GRANTS_STATUS, TOutTokens::OK, buf);
            return buf;
        }

        NXml::NWriter::AddChildNode(TOutTokens::GRANTS_STATUS,
                                    TBlackboxError::StatusStr(info->Status),
                                    buf);
        NXml::NWriter::AddChildNode(TOutTokens::DESCRIPTION, info->ToString(), buf);

        NXml::NWriter::TNode node(TOutTokens::CHECK_GRANTS_ERRORS, buf);
        for (const TString& s : result.Errors) {
            NXml::NWriter::AddChildNode(TOutTokens::ERROR, s, buf);
        }

        return buf;
    }
}
