#include "staff_client.h"

#include <passport/infra/daemons/blackbox/src/misc/strings.h>

#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    TStaffClient::TStaffClient(const TSettings& settings)
        : Settings_(settings)
        , HTTPClient_(settings.Host,
                      settings.Port,
                      settings.QueryTimeout,
                      settings.ConnectionTimeout)
    {
    }

    TStaffClient::TResponse TStaffClient::GetRobotsPage(ui64 lastEntityId, NUnistat::TTimeStat& unistatResponseTime) {
        const TKeepAliveHttpClient::THeaders headers = {{
            TStrings::X_YA_SERVICE_TICKET,
            Settings_.TvmClient ? Settings_.TvmClient->GetServiceTicketFor(NTvmCommon::TServiceTickets::STAFFAPI_) : "",
        }};

        for (ui32 retries = 0; retries < Settings_.Retries; ++retries) {
            TStringStream out;
            TInstant start = TInstant::Now();

            TString request = MakeRobotsRequest(lastEntityId);
            try {
                auto code = HTTPClient_.DoGet(request, &out, headers);

                TDuration responseTime = TInstant::Now() - start;
                unistatResponseTime.Insert(responseTime);

                if (code == 200) {
                    return {
                        .Success = true,
                        .Retries = retries,
                        .Body = out.Str(),
                    };
                }

                TLog::Info() << "StaffClient: Failed request: " << request
                             << ". Took: " << responseTime
                             << ". Got: code " << code << ": " << NUtils::EscapeEol(out.Str());
            } catch (const std::exception& e) {
                TDuration responseTime = TInstant::Now() - start;
                unistatResponseTime.Insert(responseTime);

                TLog::Info() << "StaffClient: Failed request: " << request
                             << ". Took: " << responseTime
                             << ". Got: " << e.what() << " : " << NUtils::EscapeEol(out.Str());
            }
        }

        return {
            .Success = false,
            .Retries = Settings_.Retries,
        };
    }

    TString TStaffClient::MakeRobotsRequest(ui64 lastEntityId) const {
        return NUtils::CreateStr(
            "/v3/persons",
            "?_sort=id"
            "&_query=",
            NUtils::Urlencode("id>"),
            lastEntityId,
            "&official.is_robot=true",
            "&_limit=",
            Settings_.Limit,
            "&_nopage=1",
            "&_fields=uid,id");
    }
}
