#include "staff_fetcher.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/string/cast.h>

namespace NPassport::NBb {
    static const char TIME_DELIM('.');

    static const size_t EXPECTED_ROBOTS_COUNT = 12000;

    TStaffFetcher::TStaffFetcher(TSettings&& settings)
        : Settings_(std::move(settings))
        , UnistatResponseTime_("staff.response_time", NUnistat::TTimeStat::CreateBoundsFromMaxValue(settings.ClientSettings.QueryTimeout))
    {
        try {
            ReadFromDisk();
        } catch (const std::exception& e) {
            TLog::Warning() << "StaffFetcher: failed to get cache from disk: " << e.what();
        }

        if (!RobotsList_.Get()) {
            try {
                UpdateFromStaff();
            } catch (const std::exception& e) {
                ythrow yexception() << "StaffFetcher: no valid disk cache, failed to update from staff: " << e.what();
            }
        }
    }

    TStaffFetcher::~TStaffFetcher() = default;

    void TStaffFetcher::AddUnistat(NUnistat::TBuilder& builder) {
        builder.Add(UnistatQueryErrors_);
        builder.Add(UnistatParsingErrors_);
        UnistatResponseTime_.AddUnistat(builder);
    }

    NJuggler::TStatus TStaffFetcher::GetJugglerStatus() const {
        if (LastUpdateTime_ + Settings_.UpdatePeriod * 4 <= TInstant::Now()) {
            return NJuggler::TStatus(
                NJuggler::ECode::Warning,
                "staff fetcher cache is outdated for ",
                (TInstant::Now() - LastUpdateTime_).Minutes(),
                "m");
        }
        return {};
    }

    std::shared_ptr<TStaffFetcher::TRobotsList> TStaffFetcher::GetRobots() const {
        return RobotsList_.Get();
    }

    void TStaffFetcher::Run() {
        if (LastUpdateTime_ + Settings_.UpdatePeriod <= TInstant::Now()) {
            try {
                UpdateFromStaff();
            } catch (const std::exception& e) {
                TLog::Warning() << "StaffFetcher: failed to update from staff: " << e.what();
            }
        }
    }

    void TStaffFetcher::ReadFromDisk() {
        TLog::Info() << "StaffFetcher: trying to read disk cache from " << Settings_.CachePath;

        TString cache = NUtils::ReadFile(Settings_.CachePath);
        TStringBuf buf = cache;

        ui64 timestamp;
        Y_ENSURE(TryIntFromString<10>(buf.NextTok(TIME_DELIM), timestamp),
                 "invalid cache: can't find update timestamp");

        rapidjson::Document doc;
        Y_ENSURE(NJson::TReader::DocumentAsArray(buf, doc),
                 "invalid cache: incorrect json format");

        auto robotsList = std::make_shared<TRobotsList>();
        for (size_t i = 0; i < doc.Size(); ++i) {
            ParseRobotsPage(doc[i], *robotsList);
        }

        LastUpdateTime_ = TInstant::Seconds(timestamp);

        TLog::Info() << "StaffFetcher: fetched " << robotsList->size() << " robots UIDs"
                     << " from " << Settings_.CachePath
                     << "; last updated at " << LastUpdateTime_;

        RobotsList_.Set(std::move(robotsList));
    }

    void TStaffFetcher::UpdateFromStaff() {
        TLog::Info() << "StaffFetcher: trying to update from staff-api";

        TInstant start = TInstant::Now();

        std::vector<TString> pages;
        pages.reserve(EXPECTED_ROBOTS_COUNT / Settings_.ClientSettings.Limit + 1);

        auto client = GetStaffClient();

        ui64 lastEntityId = 0;
        auto robotsList = std::make_shared<TRobotsList>();
        while (true) {
            auto response = client->GetRobotsPage(lastEntityId, UnistatResponseTime_);

            UnistatQueryErrors_ += response.Retries;

            Y_ENSURE(response.Success,
                     "failed request with " << response.Retries << " retries");

            try {
                rapidjson::Document doc;
                Y_ENSURE(NJson::TReader::DocumentAsObject(response.Body, doc),
                         "invalid response: incorrect json format: not an object");

                lastEntityId = ParseRobotsPage(doc, *robotsList);
            } catch (const std::exception&) {
                ++UnistatParsingErrors_;
                throw;
            }

            if (lastEntityId == 0) {
                break;
            }

            pages.emplace_back(std::move(response.Body));
        };

        LastUpdateTime_ = start;

        TLog::Info() << "StaffFetcher: updated " << robotsList->size() << " robots UIDs from staff-api";

        RobotsList_.Set(std::move(robotsList));

        WriteCache(pages, start);
    }

    static const TString INCORRECT_PAGE_FORMAT = "incorrect page format: ";
    ui64 TStaffFetcher::ParseRobotsPage(rapidjson::Value& page, TStaffFetcher::TRobotsList& cache) {
        const rapidjson::Value* result;
        Y_ENSURE(NJson::TReader::MemberAsArray(page, "result", result),
                 INCORRECT_PAGE_FORMAT << "result");

        ui64 lastEntityId = 0;

        for (size_t i = 0; i < result->Size(); ++i) {
            const rapidjson::Value& fields = (*result)[i];

            ui64 id;
            Y_ENSURE(NJson::TReader::MemberAsUInt64(fields, "id", id),
                     INCORRECT_PAGE_FORMAT << "result/id");
            lastEntityId = std::max(lastEntityId, id);

            TString uidStr;
            Y_ENSURE(NJson::TReader::MemberAsString(fields, "uid", uidStr),
                     INCORRECT_PAGE_FORMAT << "result/uid");

            ui64 uid;
            Y_ENSURE(TryIntFromString<10>(uidStr, uid),
                     INCORRECT_PAGE_FORMAT << "should be uint64");

            cache.insert(uid);
        }

        return lastEntityId;
    }

    void TStaffFetcher::WriteCache(const std::vector<TString>& pages, TInstant timestamp) const {
        TString cache;
        size_t size = 12 + pages.size();
        for (const auto& page : pages) {
            size += page.size();
        }
        cache.reserve(size);

        cache.append(IntToString<10>(timestamp.Seconds())).append(TIME_DELIM).append('[');
        for (const TString& page : pages) {
            cache.append(page).append(',');
        }

        if (cache.back() == ',') {
            cache.back() = ']';
        } else {
            cache.append(']');
        }

        try {
            NUtils::WriteFileViaTmp(Settings_.CachePath, cache);
        } catch (const std::exception& e) {
            ythrow yexception() << "failed to update cache file: " << e.what();
        }
    }

    std::unique_ptr<IStaffClient> TStaffFetcher::GetStaffClient() {
        return std::make_unique<TStaffClient>(Settings_.ClientSettings);
    }
}
