#pragma once

#include "totp.h"

#include <passport/infra/daemons/blackbox/src/misc/utils.h>

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <algorithm>
#include <vector>

namespace NPassport::NBb::NTotp {
    struct TTotpResult {
        enum ECode {
            Valid,
            Invalid,
            AlreadyUsed,
        };

        TTotpResult(ECode code = Invalid, std::time_t totpTime = 0)
            : Code_(code)
            , TotpTime_(totpTime)
        {
        }

        bool Succeeded() const {
            return Code() == Valid;
        }

        ECode Code() const {
            return Code_;
        }

        std::time_t TotpTime() const {
            return TotpTime_;
        }

    private:
        ECode Code_;
        std::time_t TotpTime_;
    };

    using namespace NRfc4226;
    using namespace NRfc6238;

    class TTotpInstance {
    public:
        TTotpInstance(std::size_t totpLen, std::size_t totpWindow, THotpFactory::EHotpType hotpType, THotpFactory::EHashFunction hash, std::time_t stepTime = TTotpFactory::defaultStepTime, std::time_t initTime = TTotpFactory::defaultInitTime)
            : HotpFactory_(totpLen, hotpType, hash)
            , TotpFactory_(HotpFactory_, stepTime, initTime)
            , Window_(totpWindow)
        {
        }

        TTotpResult Check(const TString& secret,
                          const TString& totp,
                          std::time_t lastLoginTime,
                          std::time_t now) const {
            TTotpFactory::TIterator it = TotpFactory_.GetIterator(now, Window_, secret);
            do {
                if (it.Value() == totp) { // password matches, check lastLoginTime
                    if (lastLoginTime < it.Timestamp()) {
                        return TTotpResult(TTotpResult::Valid, it.Timestamp()); // ok, did't check this password yet
                    }
                    return TTotpResult(TTotpResult::AlreadyUsed, 0); // already checked password from this time window
                }
            } while (it.Next());

            return TTotpResult(TTotpResult::Invalid, 0);
        }

    private:
        friend class TService;
        friend class TServiceRfc;
        const THotpFactory HotpFactory_;
        const TTotpFactory TotpFactory_;
        const std::size_t Window_;
    };

    class TService {
    public:
        TService(std::size_t digitsLen, std::size_t lettersLen, std::size_t totpWindow)
            : Digits_(digitsLen, totpWindow, THotpFactory::Digits, THotpFactory::Sha256)
            , Letters_(lettersLen, totpWindow, THotpFactory::Letters, THotpFactory::Sha256)
        {
        }

        TTotpResult Check(const TString& secret, const TString& totp, std::time_t lastLoginTime, std::time_t now = std::time(nullptr)) const {
            TString password = TUtils::NormalizeTotp(totp);
            const TTotpInstance* instance = GetInstance(password);

            if (!instance) {
                TLog::Debug("Password does not look like valid TOTP password");
                return TTotpResult(TTotpResult::Invalid);
            }

            return instance->Check(secret, password, lastLoginTime, now);
        }

        const TTotpFactory& LettersFactory() const {
            return Letters_.TotpFactory_;
        }

        // check if password looks like digit TOTP
        bool ValidDigitPassword(const TString& pwd) const {
            return NUtils::DigitsOnly(pwd) &&
                   pwd.length() == Digits_.HotpFactory_.GetHotpLength();
        }

        // check if password looks like letter TOTP
        bool ValidLetterPassword(const TString& pwd) const {
            return NUtils::LowercaseLettersOnly(pwd) &&
                   pwd.length() == Letters_.HotpFactory_.GetHotpLength();
        }

    private:
        const TTotpInstance* GetInstance(const TString& password) const {
            if (ValidDigitPassword(password)) {
                return &Digits_;
            }

            if (ValidLetterPassword(password)) {
                return &Letters_;
            }

            return nullptr;
        }

        const TTotpInstance Digits_;
        const TTotpInstance Letters_;
    };

    class TServiceRfc {
    public:
        static const size_t TOTP_LENGTH = 6;

        TServiceRfc(size_t totpWindow)
            : Digits_(TOTP_LENGTH, totpWindow, THotpFactory::Digits, THotpFactory::Sha1)
        {
        }

        NTotp::TTotpResult Check(const TString& secret,
                                 const TString& password,
                                 std::time_t lastLoginTime,
                                 std::time_t now = std::time(nullptr)) const {
            // Totp is sanitized already
            return Digits_.Check(secret, password, lastLoginTime, now);
        }

    private:
        const TTotpInstance Digits_;
    };

}
