#include "hotp.h"

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <util/generic/yexception.h>

#include <iomanip>
#include <sstream>

namespace NPassport::NBb::NRfc4226 {
    using TDynamicTruncationResult = THotpFactory::TDynamicTruncationResult;

    const TDynamicTruncationResult THotpFactory::DIGITS_TOTP_POWERS_[THotpFactory::powersDigitsSize] = {
        1L, // 10 ^ 0
        10L,
        100L,
        1000L,
        10000L,
        100000L,
        1000000L,
        10000000L,
        100000000L,  // 8 - this is maximum for the standard HOTP truncation
        1000000000L, // 9 - here the non standard i64 HOTP truncation begins
        10000000000L,
        100000000000L,
        1000000000000L // 10 ^ 12
    };

    const TDynamicTruncationResult THotpFactory::LETTERS_TOTP_POWERS_[THotpFactory::powersLettersSize] = {
        1L, // 26 ^ 0
        26L,
        676L,
        17576L,
        456976L,
        11881376L,
        308915776L,
        8031810176L,
        208827064576L // 26 ^ 8
    };

    THotp THotpFactory::Produce(const TKey& key, TCounter counter) const {
        return Truncate(Hmac(key, counter));
    }

    THotpFactory::THotpFactory(std::size_t length, EHotpType type, EHashFunction hash)
        : Hash_(hash)
        , HotpType_(type)
    {
        SetHotpLength(length);
    }

    void THotpFactory::SetHotpLength(std::size_t v) {
        if (!ValidHotpLength(v, HotpType_)) {
            throw std::invalid_argument("HotpFactory:setHotpLength(): unsupported length");
        }
        HotpLength_ = v;
    }

    bool THotpFactory::ValidHotpLength(std::size_t v, EHotpType type) {
        if (type == Digits) {
            return v >= minHotpDigitsLength && v <= maxHotpDigitsLength;
        }
        return v == defaultHotpLettersLength;
    }

    TString THotpFactory::Hmac(const TKey& key, TCounter counter) const {
        TString out;
        switch (Hash_) {
            case Sha1:
                out = NPassport::NUtils::TCrypto::HmacSha1(key, ToString(counter));
                break;
            case Sha256:
                out = NPassport::NUtils::TCrypto::HmacSha256(key, ToString(counter));
                break;
        }

        if (out.empty()) {
            throw yexception() << "Failed to generate HMAC";
        }

        return out;
    }

    THotp THotpFactory::Truncate(const TString& hs) const {
        if (HotpType_ == Digits) {
            const TDynamicTruncationResult binary = DynamicTruncationDigits(hs) % Pow(GetHotpLength());
            return MakeHotpDigits(binary);
        }
        const TDynamicTruncationResult binary = DynamicTruncationT<i64>(hs) % LETTERS_TOTP_POWERS_[GetHotpLength()];
        return MakeHotpLetters(binary);
    }

    TDynamicTruncationResult THotpFactory::DynamicTruncationDigits(const TString& hs) const {
        if (GetHotpLength() > maxHotpStandardLength) {
            return DynamicTruncationT<i64>(hs);
        }
        return DynamicTruncationT<i32>(hs);
    }

    TString THotpFactory::MakeHotpDigits(TDynamicTruncationResult binary) const {
        std::ostringstream s;
        s << std::setw(GetHotpLength()) << std::setfill('0') << binary;
        return TString(s.str());
    }

    TString THotpFactory::MakeHotpLetters(TDynamicTruncationResult binary) const {
        TString s(GetHotpLength(), 'a');

        for (TString::reverse_iterator p = s.rbegin(); p != s.rend(); ++p) {
            *p = static_cast<char>(binary % 26) + 'a';
            binary /= 26;
        }
        return s;
    }

    TString THotpFactory::ToString(TCounter counter) {
        TString retval(sizeof(TCounter), 0);
        NUtils::ToBytesMsb(counter, retval);
        return retval;
    }
}
