#pragma once

#include <util/generic/string.h>
#include <util/system/types.h>

namespace NPassport::NBb::NRfc4226 {
    using THotp = TString;

    class THotpFactory {
    public:
        using TKey = TString;
        using TCounter = ui64;

        enum {
            minHotpDigitsLength = 6,
            maxHotpStandardLength = 8,
            maxHotpDigitsLength = 12,
            defaultHotpDigitsLength = minHotpDigitsLength,
            defaultHotpLettersLength = 8
        };

        enum EHotpType {
            Digits,
            Letters
        };

        enum EHashFunction {
            Sha1,
            Sha256
        };

        THotp Produce(const TKey& key, TCounter counter) const;

        THotpFactory(std::size_t length = defaultHotpDigitsLength, EHotpType type = Digits, EHashFunction hash = Sha256);

        std::size_t GetHotpLength() const {
            return HotpLength_;
        }

        void SetHotpLength(std::size_t v);

        static bool ValidHotpLength(std::size_t v, EHotpType type);

        using TDynamicTruncationResult = ui64;

    private:
        TString Hmac(const TKey& key, TCounter counter) const;

        THotp Truncate(const TString& hs) const;

        TDynamicTruncationResult DynamicTruncationDigits(const TString& hs) const;

        template <typename T>
        T DynamicTruncationT(const TString& hs) const {
            if (hs.length() < 16 + sizeof(T)) {
                throw std::logic_error("HotpFactory::dynamicTruncationT(): hash string length is too small!");
            }

            const std::size_t offset = hs.back() & 0xf;

            T binary = T(hs[offset + 0] & 0x7f) << (sizeof(T) - 1) * 8;
            for (std::size_t i = 1; i != sizeof(T); ++i) {
                binary |= T(hs[offset + i] & 0xff) << (sizeof(T) - i - 1) * 8;
            }
            return binary;
        }

        static TDynamicTruncationResult Pow(std::size_t pow) {
            return DIGITS_TOTP_POWERS_[pow];
        }

        TString MakeHotpDigits(TDynamicTruncationResult binary) const;

        TString MakeHotpLetters(TDynamicTruncationResult binary) const;

        static TString ToString(TCounter counter);

        std::size_t HotpLength_ = defaultHotpDigitsLength;
        const EHashFunction Hash_;
        const EHotpType HotpType_;

        enum { powersDigitsSize = maxHotpDigitsLength + 1,
               powersLettersSize = defaultHotpLettersLength + 1 };
        static const TDynamicTruncationResult DIGITS_TOTP_POWERS_[powersDigitsSize];
        static const TDynamicTruncationResult LETTERS_TOTP_POWERS_[powersLettersSize];
    };

}
