#pragma once

#include "hotp.h"

#include <ctime>

namespace NPassport::NBb::NRfc6238 {
    using TTotp = NRfc4226::THotp;

    class TTotpFactory {
    public:
        static const std::time_t defaultStepTime = 30;
        static const std::time_t defaultInitTime = 0;

        using THotpFactory = NRfc4226::THotpFactory;
        using TKey = THotpFactory::TKey;
        using TTime = THotpFactory::TCounter;

        static TString GenTotp(size_t len,
                               NRfc4226::THotpFactory::EHotpType type,
                               const TString& secret,
                               time_t keyTime) {
            THotpFactory hotpFactory(len, type);
            TTotpFactory factory(hotpFactory);
            return factory.Produce(secret, keyTime);
        }

        static TString GenRfcTotp(const TString& secret, time_t keyTime) {
            THotpFactory hotpFactory(6, NRfc4226::THotpFactory::Digits, NRfc4226::THotpFactory::Sha1);
            TTotpFactory factory(hotpFactory);
            return factory.Produce(secret, keyTime);
        }

        class TIterator {
            enum EWindowSide {
                Left,
                Right,
                Center
            };

            const TTotpFactory& Parent_;
            const TTotpFactory::TTime Now_;
            const TTotpFactory::TTime From_;
            const TTotpFactory::TTime To_;
            TTotpFactory::TTime Current_;
            const TString Key_;
            EWindowSide S_ = Center;
            TTotpFactory::TTime LeftIdx_;
            TTotpFactory::TTime RightIdx_;

            friend class TTotpFactory;

            TIterator(const TTotpFactory& parent, std::time_t now, std::size_t window, const TString& key)
                : Parent_(parent)
                , Now_(parent.TimeSlotNumber(now))
                , From_(Now_ - window)
                , To_(Now_ + window)
                , Current_(Now_)
                , Key_(key)
                , LeftIdx_(Current_)
                , RightIdx_(Current_)
            {
            }

        public:
            bool Next() {
                switch (S_) {
                    case Center:
                    case Right:
                        Current_ = --LeftIdx_;
                        S_ = Left;
                        break;
                    case Left:
                        Current_ = ++RightIdx_;
                        S_ = Right;
                        break;
                }

                return From_ <= Current_ && Current_ <= To_;
            }

            TString Value() const {
                return Parent_.Hotp_.Produce(Key_, Current_);
            }

            std::time_t Timestamp() const {
                return Parent_.Timestamp(Current_);
            }

            std::time_t Shift() const {
                return Current_ - From_;
            }
        };

        TIterator GetIterator(std::time_t now, std::size_t window, const TString& key) const {
            return TIterator(*this, now, window, key);
        }

        /**
         * Function produces TOTP for specified secret key and time, according
         * to RFC-6238
         */
        TTotp Produce(const TKey& key, std::time_t now) const {
            return Hotp_.Produce(key, TimeSlotNumber(now));
        }

        /**
         * Function produces a number of TOTP for specified secret key, time and
         * window of counter. It places TOTPs into the inserter ins. Window must
         * be less than time counts for now.
         */
        template <typename Ins>
        Ins Produce(const TKey& key, std::time_t now, std::size_t window, Ins ins) const {
            const TTime center = TimeSlotNumber(now);
            if (window > center) {
                throw std::invalid_argument("TotpFactory::produce() window too large - underflow!");
            }

            const TTime from = center - window;
            const TTime to = center + window;
            if (to < from) {
                throw std::invalid_argument("TotpFactory::produce() window too large - overflow!");
            }

            for (TTime t = from; t <= to; ++t) {
                *(ins++) = Hotp_.Produce(key, t);
            }
            return ins;
        }

        std::time_t GetStepTime() const {
            return StepTime_;
        }

        std::time_t SkewToWindow(std::time_t skew) const {
            return skew / StepTime_;
        }

        TTotpFactory(const THotpFactory& hotp,
                     std::time_t stepTime = defaultStepTime,
                     std::time_t initTime = defaultInitTime)
            : Hotp_(hotp)
            , StepTime_(stepTime)
            , InitTime_(initTime)
        {
        }

    private:
        TTime TimeSlotNumber(std::time_t now) const {
            return TTime(now - InitTime_) / TTime(StepTime_);
        }

        std::time_t Timestamp(TTime now) const {
            return now * StepTime_ + InitTime_;
        }

        const THotpFactory& Hotp_;
        const std::time_t StepTime_;
        const std::time_t InitTime_;

        friend class TIterator;
    };

}
