#include "totp_profile.h"

#include <passport/infra/daemons/blackbox/src/misc/utils.h>

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NBb {
    TTotpProfile::TTotpProfile(ui64 uid, const TString& pin)
        : Uid_(uid)
        , Pin_(pin)
    {
    }

    bool TTotpProfile::IsInited() const {
        return Uid_ > 0 && !Pin_.empty();
    }

    NTotp::TTotpResult TTotpProfile::Check(const NTotp::TService& service,
                                           const TString& pwd,
                                           std::time_t lastValidTime) const {
        for (const TSecretData& secret : Data_) {
            NTotp::TTotpResult r = service.Check(secret.TotpKey(Pin_), pwd, lastValidTime);
            if (r.Code() != NTotp::TTotpResult::Invalid) {
                return r;
            }
        }
        return NTotp::TTotpResult(NTotp::TTotpResult::Invalid);
    }

    NTotp::TTotpResult TTotpProfile::Check(const NTotp::TService& service,
                                           const TString& pwd,
                                           time_t lastValidTime,
                                           ui64 secretId) const {
        for (const TSecretData& secret : Data_) {
            if (secretId == secret.SecretId()) {
                return service.Check(secret.TotpKey(Pin_), pwd, lastValidTime);
            }
        }
        return NTotp::TTotpResult(NTotp::TTotpResult::Invalid);
    }

    static const TString TOTP_IS_NOT_INITED = "TotpProfile is not inited: can't add secret";

    TTotpProfile::TSecretData* TTotpProfile::AddSecret() {
        if (!IsInited()) {
            LastError_ = TOTP_IS_NOT_INITED;
            return nullptr;
        }
        LastError_.clear();
        Data_.push_back(TSecretData());
        return &Data_.back();
    }

    TTotpProfile::TSecretData* TTotpProfile::AddSecret(const TString& secret, ui64 secretId, time_t created) {
        if (!IsInited()) {
            LastError_ = TOTP_IS_NOT_INITED;
            return nullptr;
        }

        for (const TSecretData& secret : Data_) {
            if (secretId == secret.SecretId()) {
                LastError_.clear();
                NUtils::Append(LastError_,
                               "Secret id=",
                               secretId,
                               " is duplicated. It was first added at: ",
                               secret.Created());
                return nullptr;
            }
        }

        LastError_.clear();
        TSecretData sd;
        sd.Secret_ = secret;
        sd.SecretId_ = secretId;
        sd.Created_ = created;
        Data_.push_back(std::move(sd));

        return &Data_.back();
    }

    static const TString NO_SUCH_SECRET_ID = "No such secret id";

    bool TTotpProfile::DropSecret(ui64 secretId) {
        auto it = std::find_if(Data_.begin(), Data_.end(), [&secretId](const TSecretData& d) { return d.SecretId() == secretId; });
        if (it != Data_.end()) {
            LastError_.clear();
            Data_.erase(it);
            return true;
        }

        LastError_ = NO_SUCH_SECRET_ID;
        return false;
    }

    std::pair<bool, ui64> TTotpProfile::DiagSecret(
        const TString& secretToMatch,
        const TString& salt) const {
        for (const TSecretData& secret : Data_) {
            TString expectedSecret = NUtils::TCrypto::Sha256(salt + secret.Secret());
            if (expectedSecret == secretToMatch) {
                return {true, secret.SecretId()};
            }
        }
        return {false, 0};
    }

    std::pair<bool, ui64> TTotpProfile::DiagPinSecret(
        const TString& pairToMatch,
        const TString& salt) const {
        for (const TSecretData& secret : Data_) {
            TString expectedPinAndSecret = NUtils::TCrypto::Sha256(
                TString(salt).append(Pin_).append(secret.Secret()));
            if (expectedPinAndSecret == pairToMatch) {
                return {true, secret.SecretId()};
            }
        }
        return {false, 0};
    }

    std::pair<bool, ui64> TTotpProfile::DiagTotp(
        const NTotp::TService& service,
        const TString& totpToMatch,
        time_t timestamp,
        const TString& salt) const {
        for (const TSecretData& secret : Data_) {
            TString expectedPinAndSecret = NUtils::TCrypto::Sha256(
                salt + service.LettersFactory().Produce(secret.TotpKey(Pin_), timestamp));
            if (expectedPinAndSecret == totpToMatch) {
                return {true, secret.SecretId()};
            }
        }
        return {false, 0};
    }

    std::pair<time_t, ui64> TTotpProfile::DiagTime(const NTotp::TService& service,
                                                   const TString& totpToMatch,
                                                   const TString& skew,
                                                   time_t timestamp,
                                                   const TString& salt) const {
        unsigned window = service.LettersFactory().SkewToWindow(TUtils::ToUInt(skew, "skew"));

        for (const TSecretData& secret : Data_) {
            // iterator moves from center to 'start' and 'finish' at the same time
            NRfc6238::TTotpFactory::TIterator it =
                service.LettersFactory().GetIterator(timestamp, window, secret.TotpKey(Pin_));

            do {
                if (NUtils::TCrypto::Sha256(salt + it.Value()) == totpToMatch) {
                    return {it.Timestamp(), secret.SecretId()};
                }
            } while (it.Next());
        }

        return {0, 0};
    }

    ui64 TTotpProfile::OldestSecretId() const {
        ui64 id = 0;
        std::time_t minTime = std::time(nullptr); // use now as starting point

        for (const TSecretData& secret : Data_) {
            if (secret.Created() < minTime) {
                minTime = secret.Created();
                id = secret.SecretId();
            }
        }

        return id;
    }

    TString TTotpProfile::GetSecretInfo() const {
        TString result;
        result.reserve(Data_.size() * 20);
        for (const TSecretData& secret : Data_) {
            if (!result.empty()) {
                result.push_back(',');
            }
            NUtils::Append(result,
                           secret.SecretId(),
                           ":",
                           secret.Created());
        }
        return result;
    }

    const TString& TTotpProfile::TSecretData::TotpKey(const TString& pin) const {
        if (TotpKey_.empty() || Pin_ != pin) { // works only for V3 - previous versions already filled in TotpEncryptor
            TotpKey_ = NUtils::TCrypto::Sha256(pin + Secret_);
            Pin_ = pin;
        }
        return TotpKey_;
    }

}
