#include "facade.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/log/registry.h>

namespace NPassport::NA2h {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    static TPipe* PIPE = nullptr;

    void SignalsHandlers(int signo) {
        if (SIGHUP == signo) {
            TLog::LogGotSignal(TLog::ESignal::Sighup);
            NUtils::TLogRegistry::GetInstance().RotateLogs();
        } else if (SIGUSR1 == signo) {
            TLog::LogGotSignal(TLog::ESignal::Sigusr1);
            if (PIPE) {
                PIPE->Stop();
                PIPE = nullptr;
            }
        }
    }

    TFacade::TFacade(const TString& configPath) {
        const NXml::TConfig cfg = NXml::TConfig::ReadFromFile(configPath);
        const TString xpath = "/config";

        InitLogs(cfg, xpath);

        TParser parser = CreateParser(cfg, xpath + "/parser");
        TSamplerStorage storage = CreateStorage(cfg, xpath + "/storage");
        TSignalerPtr signaler = CreateSignaler(cfg, xpath + "/signaler", storage.GetLimit());

        Pipe_ = std::make_unique<TPipe>(std::move(parser), std::move(storage), std::move(signaler));
        PIPE = Pipe_.get();
        Y_ENSURE(signal(SIGUSR1, SignalsHandlers) != SIG_ERR, "Cannot set up SIGUSR1 handler");
    }

    void TFacade::Run() {
        TLog::Info() << "Facade: Started";
        Pipe_->Run(Cin, Cout);
        TLog::Info() << "Facade: Stopping";
    }

    void TFacade::InitLogs(const NXml::TConfig& cfg, const TString& xpath) {
        cfg.InitCommonLog(xpath + "/logger");
        TLog::Info() << "Facade: Starting";

        // log rotation
        Y_ENSURE(signal(SIGHUP, SignalsHandlers) != SIG_ERR, "Cannot set up SIGHUP handler");
    }

    TParser TFacade::CreateParser(const NXml::TConfig& cfg, const TString& xpath) {
        const TDuration period = TDuration::Seconds(cfg.AsNum<size_t>(xpath + "/sampling_period", 3600));

        TLog::Info() << "Facade: parser/sampling_period=" << period;

        return TParser(period);
    }

    TSamplerStorage TFacade::CreateStorage(const NXml::TConfig& cfg, const TString& xpath) {
        const size_t limit = cfg.AsNum<size_t>(xpath + "/limit", 10000000);

        TLog::Info() << "Facade: storage/limit=" << limit;

        return TSamplerStorage(limit);
    }

    TSignalerPtr TFacade::CreateSignaler(const NXml::TConfig& cfg, const TString& xpath, const size_t limit) {
        const TString host = cfg.AsString(xpath + "/host", "localhost");
        const ui16 port = cfg.AsNum<ui16>(xpath + "/port");
        const TString path = cfg.AsString(xpath + "/xunistater_path");
        const bool isRequired = cfg.AsBool(xpath + "/is_xunistater_required", true);
        const TDuration pushPeriod = TDuration::MilliSeconds(cfg.AsInt(xpath + "/push_period", 1000));

        TLog::Info() << "Facade: signaler/host=" << host;
        TLog::Info() << "Facade: signaler/port=" << port;
        TLog::Info() << "Facade: signaler/xunistater_path=" << path;
        TLog::Info() << "Facade: signaler/is_xunistater_required=" << isRequired;
        TLog::Info() << "Facade: signaler/push_period=" << pushPeriod;

        TKeepAliveHttpClient client(host, port);

        return NUtils::TRegularTaskDecorator<TSignaler>::CreateUnique(
            {
                pushPeriod,
                "signaler",
            },
            std::move(client),
            path,
            isRequired ? TSignaler::EXunistaterRequired::True
                       : TSignaler::EXunistaterRequired::False,
            limit);
    }
}
