#include "parser.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NA2h {
    static const size_t ENTRY_TIME = 1;
    static const size_t ENTRY_TYPE = 7;
    static const size_t ENTRY_STATUS = 8;
    static const size_t COMMENT = 9;
    static const size_t USERIP = 10;

    TParser::TParser(TDuration samplingPeriod)
        : SamplingPeriod_(samplingPeriod.Seconds())
    {
        Y_ENSURE(SamplingPeriod_ > 0, "Sampling period cannot be < 0sec");

        AllocatedVector_.reserve(16);
    }

    TParser::EStatus TParser::ParseOAuthSuccessful(const TStringBuf inBuf, TString& entryKey) {
        entryKey.clear();

        // skip push client magic prefix
        TStringBuf in = inBuf;
        const char PUCHCLIENT_PREFIX_DELIM = ';';
        in.NextTok(PUCHCLIENT_PREFIX_DELIM);
        in.NextTok(PUCHCLIENT_PREFIX_DELIM);
        in.NextTok(PUCHCLIENT_PREFIX_DELIM);

        // authlog entry
        SplitToVector(in);
        if (AllocatedVector_.size() < 16) {
            TLog::Warning() << "Parser: expected column count at least==16, actual==" << AllocatedVector_.size()
                            << ". Full entry: '" << inBuf << "'";
            return EStatus::Error;
        }

        if ("oauthcheck" != AllocatedVector_[ENTRY_TYPE] || "successful" != AllocatedVector_[ENTRY_STATUS]) {
            return EStatus::NotSamplable; // it is not our target
        }

        const TStringBuf userip = AllocatedVector_[USERIP];
        if (!userip || "-" == userip) {
            TLog::Warning() << "Parser: bad userip: '" << userip << "'. Full entry: '" << inBuf << "'";
            return EStatus::Error;
        }

        const TStringBuf tokenid = GetTokenidFromComment(AllocatedVector_[COMMENT]);
        if (!tokenid) {
            TLog::Warning() << "Parser: empty tokenid. Full entry: '" << inBuf << "'";
            return EStatus::Error;
        }

        TInstant entryTime;
        if (!TInstant::TryParseIso8601(AllocatedVector_[ENTRY_TIME], entryTime)) {
            TLog::Warning() << "Parser: invalid entry time: " << AllocatedVector_[ENTRY_TIME]
                            << ". Full entry: '" << inBuf << "'";
            return EStatus::Error;
        }

        entryTime -= TDuration::Seconds(entryTime.Seconds() % SamplingPeriod_);

        NUtils::Append(entryKey, tokenid, "|", userip, "|", entryTime.Seconds());
        return EStatus::Samplable;
    }

    void TParser::SplitToVector(TStringBuf in) {
        AllocatedVector_.clear();
        while (in) {
            AllocatedVector_.push_back(in.NextTok(' '));
        }
    }

    TStringBuf TParser::GetTokenidFromComment(TStringBuf comment) {
        while (comment) {
            TStringBuf kv = comment.NextTok(';');
            const TStringBuf key = kv.NextTok('=');
            if ("tokid" == key) {
                return kv;
            }
        }

        return {};
    }
}
