#include "pipe.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/stream/input.h>
#include <util/stream/output.h>

namespace NPassport::NA2h {
    // TODO: make 3 threads:
    // 1. reading from stdin
    // 2. processing line
    // 3. writing to stdout
    TPipe::TPipe(TParser&& parser, TSamplerStorage&& storage, TSignalerPtr signaler)
        : Parser_(std::move(parser))
        , Storage_(std::move(storage))
        , Signaler_(std::move(signaler))
    {
    }

    static const TDuration PRINT_PERIOD = TDuration::Minutes(10);

    void TPipe::Run(IInputStream& in, IOutputStream& out) {
        TInstant nextPrint;
        size_t count = 0;
        TString line;

        while (in.ReadLine(line) > 0 && !Exit_.load(std::memory_order_relaxed)) {
            ++count;

            try {
                const TInstant now = TInstant::Now();
                if (nextPrint < now) {
                    nextPrint = now + PRINT_PERIOD;
                    TLog::Debug() << "Pipe: heartbeat: proceed " << count << " lines";
                }

                Signaler_->SetStorageSize(Storage_.size());

                if (ProcessLine(line)) {
                    continue;
                }
            } catch (const std::exception& e) {
                TLog::Error() << "Pipe: exception while processing line: " << e.what();
            }

            out << line << Endl;
        }

        TLog::Info() << "Pipe: exiting, got " << count << " lines";
    }

    void TPipe::Stop() {
        Exit_ = true;
    }

    bool TPipe::ProcessLine(TStringBuf line) {
        Signaler_->Add(TSignaler::ESignal::AllLines);

        switch (Parser_.ParseOAuthSuccessful(line, EntryKey_)) {
            case TParser::EStatus::NotSamplable:
                return false;
            case TParser::EStatus::Error:
                Signaler_->Add(TSignaler::ESignal::ParsingErrors);
                return false;
            case TParser::EStatus::Samplable:
                break;
        }

        Signaler_->Add(TSignaler::ESignal::OAuthSuccessAll);

        switch (Storage_.TryAdd(EntryKey_)) {
            case TSamplerStorage::EStatus::Added:
                return false;
            case TSamplerStorage::EStatus::Exists:
                Signaler_->Add(TSignaler::ESignal::OAuthSuccessSkipped);
                return true;
        }
    }
}
