#pragma once

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/split.h>

#include <unordered_set>

namespace NTskvFormat {
    class TLogBuilder;
}

namespace NPassport::NBbAccess {
    using TCgiParams = THashMap<TStringBuf, TStringBuf>;
    using TRespSize = ui64;
    using TCountedValues = THashMap<TString, ui64>;

    class TCumulativeValue {
        using TType = ui64;

    public:
        void Map(TType v);
        void Reduce(TCumulativeValue& to) const;

        TType GetMin() const {
            return Min_;
        }
        TType GetMax() const {
            return Max_;
        }
        double GetAvg() const {
            return 1. * Sumvalue_ / Count_;
        }

    private:
        TType Sumvalue_ = 0;
        TType Min_ = std::numeric_limits<TType>::max();
        TType Max_ = std::numeric_limits<TType>::min();
        TType Count_ = 0;
    };

    class TConsumer { // NOLINT(bugprone-exception-escape)
    public:
        using TSet = THashSet<TString>;

        struct TData {
            bool IsHttps = false;
            bool IsCacheHit = false;
            TCgiParams* Params = nullptr;
            TCgiParams* GetParams = nullptr;
            TDuration RespTime;
            TRespSize RespSize = 0;
            TStringBuf Ip;
            std::optional<ui32> RequestNumberThroughOneConnection;
            TStringBuf Line;

            void EnsureIsOk() const;
        };

        void Reserve();

        void Map(TData& data);
        void Reduce(TConsumer& to) const;

        void PrintRaw(TStringBuf grantType, TStringBuf method, TStringBuf consumer, IOutputStream& stream) const;
        void PrintPretty(TStringBuf consumer, ui64 totalMethod, IOutputStream& stream) const;

        ui64 TotalRequests() const {
            return Total_;
        }

    protected:
        static TString SerializeSet(const TSet& set);
        static TString SerializeCountedValues(const TCountedValues& values, size_t topCount);
        static std::optional<TStringBuf> GetHostFromRequest(const TCgiParams& params);

    private:
        static void ProcessListedArg(TCgiParams& params, TSet& set, const TString& arg);
        static void PrintRawListedArg(const TSet& set, const TString& arg, NTskvFormat::TLogBuilder& tskv);

        static void ProcArgCount(TCumulativeValue& val, const TString& arg, const TCgiParams& params);

        static void ReduceSet(const TSet& from, TSet& to);

    private:
        ui64 Total_ = 0;
        ui64 Https_ = 0;
        ui64 CacheHit_ = 0;
        ui64 ReqsWithUserPort_ = 0;
        TDuration RespTime_;
        TRespSize RespSize_ = 0;
        TSet Dbfields_;
        TSet Attributes_;
        TSet PhoneAttributes_;
        TSet Args_;
        TSet GetArgs_;
        TSet Ip_;
        TSet Examples_;
        TSet UserIpNoSessguard_;
        TCountedValues Hosts_;

        TCumulativeValue Uids_;
        TCumulativeValue RequestNumberThroughOneConnection_;
    };

}
