#include "sessguard.h"

#include <passport/infra/libs/cpp/re2/regex.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/string_utils/tskv_format/builder.h>

namespace NPassport::NBbAccess {
    static const size_t EXAMPLES_COUNT = 3;

    TSessguardStats::TSessguardStats()
        : Config_(std::make_unique<TSessguardConfigReader>())
    {
    }

    void TSessguardStats::Map(EGrantType gt, TStringBuf consumer, TStringBuf method, const TConsumer::TData& data) {
        const TStringBuf host = GetHost(data);
        if (host.empty()) {
            return;
        }

        const TStringBuf guard = DetectGuardSpace(host);
        if (guard.empty()) {
            return;
        }

        TCounters& counters = GetCounters(gt, consumer, method, host, guard);
        ++counters.Total;

        auto addExample = [&](std::vector<TString>& examples) {
            if (examples.size() < EXAMPLES_COUNT) {
                examples.emplace_back(data.Line);
            }
        };

        auto it = data.Params->find("sessguard");
        if (it == data.Params->end()) {
            addExample(counters.ExamplesWithoutSessguard);
            return;
        }
        ++counters.WithAnySessguard;

        if (it->second) {
            ++counters.WithNonEmptySessgaurd;
        } else {
            addExample(counters.ExamplesWithEmptySessguard);
        }
    }

    void TSessguardStats::Reduce(TSessguardStats& to) const {
        for (const auto& [sessguardKey, counters] : Stats_) {
            to.Stats_[sessguardKey].Add(counters);
        }
    }

    void TSessguardStats::PrintTskv(IOutputStream& stream) const {
        for (const auto& [sesguardKey, counters] : Stats_) {
            NTskvFormat::TLogBuilder tskv;
            tskv.Add("guard_space", sesguardKey.GuardSpace);
            tskv.Add("host", sesguardKey.Host);
            tskv.Add("grant_type", sesguardKey.GrantType == EGrantType::Ip ? "ip" : "tvm2");
            tskv.Add("consumer", sesguardKey.Consumer);
            tskv.Add("method", sesguardKey.Method);

            tskv.Add("reqs_total", ToString(counters.Total));
            tskv.Add("reqs_with_any_sessguard", ToString(counters.WithAnySessguard));
            tskv.Add("reqs_with_non_empty_sessgaurd", ToString(counters.WithNonEmptySessgaurd));

            auto printExamples = [&tskv](const std::vector<TString>& examples, TStringBuf tskvKey) {
                std::vector<TString> tmp(examples);
                std::sort(tmp.begin(), tmp.end());
                TString buf;
                for (const TString& val : tmp) {
                    NUtils::AppendSeparated(buf, '\n', val);
                }
                tskv.Add(tskvKey, NUtils::BinToBase64(buf, true));
            };
            printExamples(counters.ExamplesWithoutSessguard, "examples_without_sessguard");
            printExamples(counters.ExamplesWithEmptySessguard, "examples_with_empty_sessguard");

            tskv.End();
            stream << "tskv\t" << tskv.Str();
        }
    }

    TStringBuf TSessguardStats::GetHost(const TConsumer::TData& data) {
        auto it = data.Params->find("host");
        if (it == data.Params->end()) {
            return {};
        }
        return it->second;
    }

    TStringBuf TSessguardStats::DetectGuardSpace(TStringBuf host) const {
        TStringBuf tld = host.RNextTok('.');
        Y_UNUSED(tld);
        TStringBuf sld = host.RNextTok('.');

        const TSessguardConfigReader::TGuards* gh = Config_->GetGuards(sld);
        if (!gh) {
            gh = Config_->GetGuards(host.RNextTok('.'));
        }
        if (!gh) {
            return {};
        }

        const TStringBuf serviceDomain = host.RNextTok('.');
        if (serviceDomain.empty()) {
            return {};
        }

        auto it = gh->find(serviceDomain);
        return it == gh->end() ? TStringBuf() : it->second;
    }

    TSessguardStats::TCounters& TSessguardStats::GetCounters(EGrantType gt,
                                                             TStringBuf consumer,
                                                             TStringBuf method,
                                                             TStringBuf host,
                                                             TStringBuf guardSpace) {
        auto it = Stats_.find(TSesguardKey<TStringBuf>{
            .GuardSpace = guardSpace,
            .Host = host,
            .GrantType = gt,
            .Consumer = consumer,
            .Method = method,
        });
        if (it != Stats_.end()) {
            return it->second;
        }

        return Stats_.insert({
                                 TSesguardKey<TString>{
                                     .GuardSpace = TString(guardSpace),
                                     .Host = TString(host),
                                     .GrantType = gt,
                                     .Consumer = TString(consumer),
                                     .Method = TString(method),
                                 },
                                 {},
                             })
            .first->second;
    }

    void TSessguardStats::TCounters::Add(const TSessguardStats::TCounters& o) {
        Total += o.Total;
        WithAnySessguard += o.WithAnySessguard;
        WithNonEmptySessgaurd += o.WithNonEmptySessgaurd;

        auto addExample = [](const std::vector<TString>& from, std::vector<TString>& to) {
            for (size_t idx = 0; idx < from.size() && to.size() < EXAMPLES_COUNT; ++idx) {
                to.push_back(from[idx]);
            }
        };
        addExample(o.ExamplesWithoutSessguard, ExamplesWithoutSessguard);
        addExample(o.ExamplesWithEmptySessguard, ExamplesWithEmptySessguard);
    }

    static const TString CONFIG_PATH_PREFIX = "resfs/file/passport/infra/daemons/blackbox/config/configs/";

    TSessguardConfigReader::TSessguardConfigReader()
        : ProdGuards_(BuildGuards(NResource::Find(CONFIG_PATH_PREFIX + "template-blackbox.conf")))
        , YateamGuards_(BuildGuards(NResource::Find(CONFIG_PATH_PREFIX + "yateam-blackbox-include.xml")))
    {
    }

    const TSessguardConfigReader::TGuards* TSessguardConfigReader::GetGuards(TStringBuf sld) const {
        if (sld == "yandex") {
            return &ProdGuards_;
        }

        if (sld == "yandex-team") {
            return &YateamGuards_;
        }

        return nullptr;
    }

    TSessguardConfigReader::TGuards TSessguardConfigReader::BuildGuards(TStringBuf config) {
        NRe2::TRegexGroups re("<guard_space id=\"\\d+\" name=\"guard_([^\"]+)\" hostnames=\"([^\"]+)\"");

        TGuards result;

        std::vector<TStringBuf> vec;
        while (config) {
            TStringBuf line = config.NextTok("\n");
            if (!re.PartialMatch(line, vec)) {
                continue;
            }

            const TStringBuf space = vec.at(0);
            for (TStringBuf host : NUtils::ToVector(vec.at(1), ";")) {
                result[host] = space;
            }
        }

        return result;
    }
}
