#include "stats.h"

#include "utils.h"

namespace NPassport::NBbAccess {
    static const TString METHOD = "method";
    static const TString SESSIONID = "sessionid";

    TStats::TStats() {
        Ip_.reserve(50);
        Tvm2_.reserve(50);
        Unrecognized_.reserve(50);
    }

    void TStats::Map(TStringBuf consumer, TConsumer::TData& data) {
        data.EnsureIsOk();
        TStatsByConsumer* cont = nullptr;

        char delim = ':';
        TStringBuf type = consumer.NextTok(delim);
        TStringBuf name = consumer.NextTok(delim);
        TStringBuf extra = consumer.NextTok(delim);

        std::optional<EGrantType> grantType;
        if (type == "1") {
            cont = &Ip_;
            grantType = EGrantType::Ip;
            Y_UNUSED(extra);
        } else if (type == "3") {
            cont = &Tvm2_;
            grantType = EGrantType::Tvm;
        } else {
            cont = &Unrecognized_;
            name = "<none>";
        }

        TStringBuf m = GetMethod(*data.Params);
        auto it = cont->find(m);
        if (it == cont->end()) {
            it = cont->emplace(m, TMethod()).first;
            it->second.Reserve();
        }

        if (grantType) {
            Sessguard_.Map(*grantType, name, m, data);
        }

        ++TotalRequests_;
        it->second.Map(name, data);
    }

    void TStats::Reduce(TStats& to) const {
        to.TotalRequests_ += TotalRequests_;
        ReduceKv(Ip_, to.Ip_);
        ReduceKv(Tvm2_, to.Tvm2_);
        ReduceKv(Unrecognized_, to.Unrecognized_);

        Sessguard_.Reduce(to.Sessguard_);
    }

    void TStats::PrintRaw(IOutputStream& stream) const {
        auto pr = [&stream](const TStatsByConsumer& cont, TStringBuf grantType) {
            for (const auto& [method, data] : SortCont(cont)) {
                data->PrintRaw(grantType, method, stream);
            }
        };

        pr(Ip_, "ip");
        pr(Tvm2_, "tvm2");
        pr(Unrecognized_, "unrecognized");
    }

    void TStats::PrintPretty(IOutputStream& stream) const {
        auto pr = [this, &stream](const TStatsByConsumer& cont, const TString& title) {
            ui64 totalCont = 0;

            TVector<const TStatsByConsumer::value_type*> sorted;
            sorted.reserve(cont.size());
            for (const auto& pair : cont) {
                sorted.push_back(&pair);
                totalCont += pair.second.TotalRequests();
            }
            std::sort(sorted.begin(), sorted.end(), [](const TStatsByConsumer::value_type* l, const TStatsByConsumer::value_type* r) -> bool {
                return l->second.TotalRequests() > r->second.TotalRequests();
            });

            const TStringBuf eqs = " ================ ";
            stream << eqs << title << " grants " << MakeRatio(totalCont, TotalRequests_) << eqs << Endl;

            for (const TStatsByConsumer::value_type* pair : sorted) {
                pair->second.PrintPretty(pair->first, totalCont, stream);
            }

            stream << Endl;
        };

        TStatsByConsumer summary;
        ReduceKv(Ip_, summary);
        ReduceKv(Tvm2_, summary);
        ReduceKv(Unrecognized_, summary);

        pr(summary, "summary");
        pr(Ip_, "ip");
        pr(Tvm2_, "tvm2");
        pr(Unrecognized_, "unrecognized");
    }

    void TStats::PrintSessguard(IOutputStream& stream) const {
        Sessguard_.PrintTskv(stream);
    }

    TStringBuf TStats::GetMethod(TCgiParams& params) {
        auto itParam = params.find(METHOD);
        if (itParam != params.end()) {
            TStringBuf methodName(itParam->second);
            params.erase(itParam);
            return methodName;
        }
        if (params.find(SESSIONID) != params.end()) {
            return SESSIONID;
        }

        return {};
    }
}
