#include "stats.h"

#include <passport/infra/libs/cpp/utils/string/format.h>

#include <util/stream/format.h>

namespace NPassport::NBbError {
    void SortCountedMessage(TCountedMessage& data) {
        std::sort(data.begin(), data.end(), [](const auto& l, const auto& r) {
            if (l.second == r.second) {
                return l.first > r.first;
            }

            return l.second > r.second;
        });
    }

    static const std::vector<TString> NOISE_MESSAGES = {
        "malformed email in db, ",
        "blackbox performance warning: pdd domains:",
        "sessguard",
        "monitoring warning: root keychain is",
        "blocked stress testing session cookie",
        "duplicated args:",
        "asyncdbwriter ",
        "leaving hosted domains list intact",
        "hosteddomains: performance issue:",
        "got alias with unknown",
        "failed to load random key from db",
    };

    TSortedMessages::TSortedMessages(const TLevelMessages& msg) {
        Common.reserve(msg.size());
        TNoiseMap noise;

        for (const auto& [message, count] : msg) {
            if (IsNoise(message, count, noise)) {
                continue;
            }

            Common.push_back({message, count});
        }

        // Sorting
        SortCountedMessage(Common);

        for (auto& pair : noise) {
            Noise.push_back({pair.first, std::move(pair.second)});
        }
        std::sort(Noise.begin(), Noise.end(), [](const auto& l, const auto& r) {
            return l.second.TotalCount > r.second.TotalCount;
        });
        for (auto& pair : Noise) {
            SortCountedMessage(pair.second.Messages);
        }
    }

    bool TSortedMessages::IsNoise(const TString& message, ui64 count, TNoiseMap& noise) {
        for (const TString& n : NOISE_MESSAGES) {
            if (!NUtils::TolowerCopy(message).Contains(n)) {
                continue;
            }

            auto it = noise.find(n);
            if (it == noise.end()) {
                it = noise.emplace(n, TNoiseMessages{}).first;
            }
            it->second.TotalCount += count;
            it->second.Messages.push_back({message, count});

            return true;
        }

        return false;
    }

    TStats::TStats() {
        Data_.resize(LevelCount);
        for (TLevelMessages& l : Data_) {
            l.reserve(100);
        }
    }

    void TStats::Map(TStringBuf level, TStringBuf msg) {
        std::optional<ELevel> lvl = CastLevel(level);
        if (!lvl) {
            return;
        }

        TLevelMessages& messages = Data_[*lvl];
        auto [it, inserted] = messages.try_emplace(msg, 1);
        if (!inserted) {
            ++it->second;
        }
    }

    void TStats::Reduce(TStats& to) const {
        for (size_t idx = 0; idx < Data_.size(); ++idx) {
            TLevelMessages& messagesTo = to.Data_[idx];

            for (const auto& [message, count] : Data_[idx]) {
                auto [it, inserted] = messagesTo.try_emplace(message, count);
                if (!inserted) {
                    it->second += count;
                }
            }
        }
    }

    void TStats::PrintPretty(IOutputStream& stream) const {
        for (size_t idx = 0; idx < Data_.size(); ++idx) {
            if (Data_[idx].empty()) {
                continue;
            }

            stream << CastLevel((ELevel)idx) << ":" << Endl;

            TSortedMessages sorted(Data_[idx]);
            for (const auto& [message, count] : sorted.Common) {
                stream << LeftPad(count, 16) << ": " << message << Endl;
            }

            if (!sorted.Noise.empty()) {
                stream << Endl << "===Noise: " << Endl;
                for (const auto& [substring, noise] : sorted.Noise) {
                    stream << "total=" << noise.TotalCount << ": " << substring << Endl;
                    size_t printed = 0;

                    for (const auto& [message, count] : noise.Messages) {
                        stream << LeftPad(count, 16) << ": " << message << Endl;
                        if (++printed >= 5) {
                            break;
                        }
                    }

                    if (printed < noise.Messages.size()) {
                        stream << "..." << Endl;
                    }

                    stream << Endl;
                }
            }

            stream << Endl;
        }
    }

    std::optional<TStats::ELevel> TStats::CastLevel(TStringBuf level) {
        if ("ERROR" == level) {
            return Error;
        }
        if ("WARNING" == level) {
            return Warning;
        }

        return {};
    }

    static const THashMap<TStats::ELevel, TString> LEVELS_TO_STRING = {
        {TStats::ELevel::Error, "ERROR"},
        {TStats::ELevel::Warning, "WARNING"},
        {TStats::ELevel::Info, "INFO"},
        {TStats::ELevel::Debug, "DEBUG"},
    };
    TStringBuf TStats::CastLevel(TStats::ELevel level) {
        auto it = LEVELS_TO_STRING.find(level);
        Y_ENSURE(it != LEVELS_TO_STRING.end());
        return it->second;
    }
}
