import argparse
import json
import sys

import requests


def get_abc_service_by_tvm_ids(tvm_ids, oauth_token):
    abcid_by_tvmid = {}
    parent_by_abcid = {}

    page_size = 500

    def chunks():
        for i in range(0, len(tvm_ids), page_size):
            yield tvm_ids[i : i + page_size]

    for id_range in chunks():
        url = (
            'https://abc-back.yandex-team.ru/api/v4/resources/consumers/'
            + '?supplier=14'
            + '&type=47'
            + '&page_size=%s' % page_size
            + '&fields=service.id,service.parent,resource.external_id'
            + '&resource__external_id__in='
            + ','.join(id_range)
        )

        tries = 5
        while tries > 0:
            tries = tries - 1
            try:
                r = requests.get(url, headers={'Authorization': 'OAuth %s' % oauth_token})
                assert r.status_code == 200, r.text
                d = json.loads(r.text)
                for result in d['results']:
                    abcid_by_tvmid[result['resource']['external_id']] = result['service']['id']
                    parent_by_abcid[result['service']['id']] = result['service']['parent']
            except Exception as e:
                print('Failed to get abc ids by tvm id: %s: %s' % (url, e), file=sys.stderr)

    return abcid_by_tvmid, parent_by_abcid


def get_abc_parents(abc_ids, oauth_token):
    parent_by_abcid = {}
    abc_ids = [str(i) for i in abc_ids if i is not None]

    page_size = 500

    def chunks():
        for i in range(0, len(abc_ids), page_size):
            yield abc_ids[i : i + page_size]

    for id_range in chunks():
        url = (
            'https://abc-back.yandex-team.ru/api/v4/services/'
            + '?page_size=%s' % page_size
            + '&fields=id,parent.id'
            + '&id__in='
            + ','.join(id_range)
        )

        tries = 5
        while tries > 0:
            tries = tries - 1
            try:
                r = requests.get(url, headers={'Authorization': 'OAuth %s' % oauth_token})
                assert r.status_code == 200, r.text
                d = json.loads(r.text)
                for result in d['results']:
                    p = result['parent']['id'] if result['parent'] is not None else None
                    parent_by_abcid[result['id']] = p
            except Exception as e:
                print('Failed to get abc parents: %s: %s' % (url, e), file=sys.stderr)

    return parent_by_abcid


def download_grants(git_hash, use_passport_api):
    if use_passport_api:
        url = 'https://raw.github.yandex-team.ru/passport/passport-grants/%s/grants/consumer_grants.production.json'
    else:
        url = 'https://raw.github.yandex-team.ru/passport/blackbox-by-client-grants/%s/grants/consumer_grants.production.json'

    url = url % git_hash

    return json.loads(requests.get(url).text)


def run(git_hash, oauth_token, use_passport_api):
    grants = download_grants(git_hash, use_passport_api)
    tvm_ids = [str(v['client']['client_id']) for k, v in grants.items() if 'client_id' in v['client']]

    abcid_by_tvmid, parent_by_abcid = get_abc_service_by_tvm_ids(tvm_ids, oauth_token)
    parent_by_abcid.update(
        {
            842: None,  # Песочница
        }
    )

    def calc_orphans():
        return list(set([v for k, v in parent_by_abcid.items() if v is not None and v not in parent_by_abcid]))

    orphans = calc_orphans()
    while len(orphans) > 0:
        parent_by_abcid.update(get_abc_parents(orphans, oauth_token))
        orphans = calc_orphans()

    all_tvm_ids = [k for k, v in abcid_by_tvmid.items()]
    all_tvm_ids.sort()

    for tvm_id in all_tvm_ids:
        parent = abcid_by_tvmid[tvm_id]
        parents = []

        while parent is not None:
            parents.append(str(parent))
            parent = parent_by_abcid[parent]

        print('%s %s' % (tvm_id, ','.join(parents)))


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='fetcher of ABC ids from bb grants')
    parser.add_argument('--git_hash', type=str, help='git hash with grants', required=True)
    parser.add_argument('--oauth_token', type=str, help='oauth token to use abc-back', required=True)
    parser.add_argument('--passport_api', action='store_true', help='build for passport_api instead of blackbox')
    args = parser.parse_args()

    run(
        args.git_hash,
        args.oauth_token,
        args.passport_api,
    )
