import argparse
import json
import os

import startrek_client


MEMBERS = ['abc_admins', 'abc_managers', 'c_admins', 'fw_owners', 'abc_owners', 'staff_owners', 'host_owners']


def build_description(name, grants, yqls):
    res = (
        """Внешние аудиторы ежегодно проверяют Паспорт на соответствие требованиям SOC3.
Мы готовимся к очередному ежегодному аудиту, который завершится в 2022 году: https://goals.yandex-team.ru/filter?goal=111307

Одним из требований SOC3 является актуализация прав доступа в сервисы, находящиеся под аудитом.
Пожалуйста, в срок до 1 октября 2021 года проверьте, что у сервиса '%s' нет избыточных грантов, и доступ не открыт на избыточные макросы/кондукторные группы.
Обратите внимание, что в этом тикете мы актуализируем только гранты без TVM и только в продуктивное окружение (не корпоративное, не тестовое, не стрессовое и не Мимино).

Если вы видите избыточные гранты или избыточные макросы/кондукторные группы, напишите это в комментарии и призовите кого:cerevra.
Если что-то не понятно, напишите вопрос в комментарии и призовите кого:cerevra.
Если все гранты, макросы и кондукторные группы нужны, закройте тикет.

**Пожалуйста, начните использовать TVM при работе с Чёрным ящиком! Просим привязать сюда таск на подключение TVM. Спасибо**
------------------------

"""
        % name
    )

    def add_network(grants, type_, desc):
        if type_ not in grants['networks']:
            return ''
        res = 'Гранты предоставлены следующим %s:\n' % desc
        for m in grants['networks'][type_]:
            res += '  * ##%s##\n' % m
        return res

    res += add_network(grants, 'F', 'макросам')
    res += add_network(grants, 'C', 'кондукторным группам')
    res += add_network(grants, 'I', 'IP-адресам')
    res += add_network(grants, 'H', 'хостам')
    res += add_network(grants, 'N', 'подсетям')

    res += 'Гранты предоставлены на:\n'
    for k, v in grants['grants'].items():
        if k == 'has_cred' or k == 'no_cred' or k.startswith('__'):
            continue
        res += '  * ##%s: %s##\n' % (k, v)

    if 'has_cred' in grants['grants']:
        res += 'Гранты предоставлены на получение следующих данных о пользователе по креденшлам (методы ##sessionid##, ##oauth##, ##login##, ##user-ticket##):\n'
        for v in grants['grants']['has_cred']:
            res += '  * ##%s##\n' % v
    if 'no_cred' in grants['grants']:
        res += 'Гранты предоставлены на получение следующих данных о пользователе по идентификатору пользователя (метод ##userinfo##):\n'
        for v in grants['grants']['no_cred']:
            res += '  * ##%s##\n' % v

    res += 'Посмотреть, с каких хостов (ips) в какой метод черного ящика (method) и с какими параметрами были запросы с 1 июня по 1 июля можно тут:\n'
    res += 'https://yql.yandex-team.ru/Operations/%s' % yqls[name]

    return res


def dry_run(grants_file, yql_file, prepared_tasks, use_passport_api, tags, deadline):
    with open(grants_file) as f:
        grants = json.load(f)
    with open(yql_file) as f:
        yqls = json.load(f)

    tasks = {}
    for name, grants in grants.items():
        managers = []
        for m in MEMBERS:
            managers += grants.get(m, [])
        managers.sort()
        assert len(managers) > 0, 'no one manager for %s' % name

        tasks[name] = {
            'summary': 'Актуализация грантов в %s, доступ без TVM, сервис "%s"'
            % (
                "API Паспорта" if use_passport_api else "Черный ящик",
                name,
            ),
            'managers': managers,
            'description': build_description(name, grants, yqls),
            'assignee': managers[0],
            'tags': tags.split(','),
            'deadline': deadline,
        }

    with open(prepared_tasks, 'w') as f:
        f.write(json.dumps(tasks, indent=4, sort_keys=True, ensure_ascii=False))
        f.write('\n')


def create_task(client, task, abc_id):
    users = task['managers']
    # return '%s\thttps://st.yandex-team.ru/%s.\t%s' % (abc_id, "kek", users)
    issue = client.issues.create(
        queue='PASSPORTGRANTS',
        type={'name': 'Task'},
        components=['#GrantsReview'],
        summary=task['summary'],
        description=task['description'],
        followers=users,
        deadline=task['deadline'],
        assignee=task['assignee'],
        tags=task['tags'],
    )
    SKIP_KEY_LEN = len('https://st-api.yandex-team.ru/v2/issues/')
    return '%s\thttps://st.yandex-team.ru/%s.\t%s' % (abc_id, issue.self[SKIP_KEY_LEN:], users)


def wet_run(oauth_token, prepared_tasks, already_created, limit):
    client = startrek_client.Startrek(
        useragent='kek',
        base_url='https://st-api.yandex-team.ru/',
        token=oauth_token,
    )

    with open(prepared_tasks) as f:
        tasks = json.load(f)

    created = set()
    if os.path.isfile(already_created):
        for line in open(already_created):
            abc_id = line.split('\t')[0]
            created.add(abc_id)

    count = 0

    for abc_id, task in tasks.items():
        if abc_id in created:
            print('task for abc_id %s is ready. skip it' % abc_id)
            continue

        try:
            line = create_task(client, task, abc_id)
            print(line)
            with open(already_created, 'a') as f:
                f.write(line)
                f.write('\n')
            created.add(abc_id)
        except Exception as e:
            print('%s : %s' % (abc_id, e))

        count += 1
        if count >= limit:
            break


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='prepares tasks for grants verification')
    parser.add_argument('--grants_file', type=str, help='file with prepared grants', required=True)
    parser.add_argument('--yql_file', type=str, help='file with yql queries', required=True)
    parser.add_argument('--oauth_token', type=str, help='oauth token to use startrek', required=True)
    parser.add_argument('--prepared_tasks', type=str, help='files with prepared tasks', required=True)
    parser.add_argument('--already_created', type=str, help='files with already created tasks', required=True)
    parser.add_argument('--tags', type=str, help='tags for task in startrek: 2021,blackbox', required=True)
    parser.add_argument('--limit', type=int, help='tasks count to create:', default=1)
    parser.add_argument(
        '--deadline', type=str, help='deadline for task in startrek: 2021-07-31T18:00:00.00+0000', required=True
    )
    parser.add_argument('--wet_run', action='store_true', help='actual create task')
    parser.add_argument('--passport_api', action='store_true', help='build for passport_api instead of blackbox')
    args = parser.parse_args()

    if args.wet_run:
        wet_run(
            args.oauth_token,
            args.prepared_tasks,
            args.already_created,
            args.limit,
        )
    else:
        dry_run(
            args.grants_file,
            args.yql_file,
            args.prepared_tasks,
            args.passport_api,
            args.tags,
            args.deadline,
        )
