import argparse
import json

import requests


BLACKBOX_QUERY = """use hahn;

$consumer='%s';

$data = (select
            TableName() as d,
            method,
            consumer,
            grant_type,
            String::SplitToList(String::Base64Decode(all_ip), "\\n", true) as ips,
            String::SplitToList(args, ",", true) as args,
            String::SplitToList(attributes, ",", true) as attributes,
            String::SplitToList(dbfields, ",", true) as dbfields,
            String::SplitToList(phone_attributes, ",", true) as phone_attributes,
            cast(total_reqs as uint64) as total_reqs,
            cast(https as uint64) as https_reqs,
            String::SplitToList(request_number_through_one_connection, ":") as keep_alive,
            String::SplitToList(uid, ":") as uid_bulk,
            cast(cache_hit as uint64) as cached_reqs,
            cast(avg_resp_time as float) as avg_resp_time,
            cast(avg_resp_size as uint32) as avg_resp_size,
            String::SplitToList(String::Base64Decode(some_example), "\n", true) as examples,
        from RANGE (`home/passport/production/reports/blackbox/prod`, `%s`, `%s`));

select
    d,
    method,
    ips,
    args,
    attributes,
    dbfields,
    phone_attributes,
    total_reqs,
    https_reqs,
    cached_reqs,
    cast(keep_alive[2] as uint32) as keep_alive_max,
    cast(keep_alive[1] as float) as keep_alive_mean,
    avg_resp_time,
    avg_resp_size,
    examples,
from $data
where grant_type='ip' and consumer=$consumer
"""


PASSPORTAPI_QUERY = """use hahn;

$consumer='%s';

select
    TableName() as d,
    consumers,
    required_grants,
    count,
    ips,
    tvm_client_id
from range('//home/passport/production/reports/passport-api/grants', 'graphite_grants_%s', 'graphite_grants_%s')
where SetIntersection(
    ToSet(String::SplitToList(consumers, ',')),
    ToSet(AsList($consumer))
) != {}
order by d;"""


def get_consumers(grants_file):
    with open(grants_file) as f:
        grants = json.load(f)

    return grants.keys()


def create_yql_operation(name, oauth_token, use_passport_api, date_from, date_to):
    req_body = {
        'content': (PASSPORTAPI_QUERY if use_passport_api else BLACKBOX_QUERY) % (name, date_from, date_to),
        'action': 'RUN',
        'type': 'SQLv1',
    }
    r = requests.post(
        'https://yql.yandex.net/api/v2/operations',
        data=json.dumps(req_body),
        headers={
            'Content-Type': 'application/json',
            'Authorization': 'OAuth %s' % oauth_token,
        },
    )
    assert r.status_code == 200, 'code=%s. %s' % (r.status_code, r.text)

    id = json.loads(r.text)['id']

    r = requests.get(
        'https://yql.yandex.net/api/v2/operations/%s/share_id' % id,
        headers={
            'Content-Type': 'application/json',
            'Authorization': 'OAuth %s' % oauth_token,
        },
    )
    assert r.status_code == 200, 'code=%s. %s' % (r.status_code, r.text)

    return r.text.strip('"')


def get_operation_status(id, oauth_token):
    r = requests.get(
        'https://yql.yandex.net/api/v2/operations/%s' % id,
        headers={
            'Content-Type': 'application/json',
            'Authorization': 'OAuth %s' % oauth_token,
        },
    )
    assert r.status_code == 200, 'code=%s. %s' % (r.status_code, r.text)
    return json.loads(r.text)['status']


def run(grants_file, oauth_token, out_operations, use_passport_api, date_from, date_to):
    consumers = get_consumers(grants_file)

    res = {}
    for name in consumers:
        shared_link = create_yql_operation(name, oauth_token, use_passport_api, date_from, date_to)
        res[name] = shared_link
        print('name=\'%s\'; shared_link=\'%s\'' % (name, shared_link))

    with open(out_operations, 'w') as f:
        f.write(json.dumps(res, indent=4, sort_keys=True))
        f.write('\n')


def rerun(grants_file, oauth_token, out_operations, use_passport_api, date_from, date_to):
    with open(out_operations) as f:
        operations = json.load(f)

    to_rerun_tvmids = []
    in_flight = 0
    for tvmid, id in operations.items():
        status = get_operation_status(id, oauth_token)
        if status in ['RUNNING', 'PENDING', 'IDLE']:
            in_flight += 1
            print('tvmid=%s. operation is %s' % (tvmid, status))
            continue

        if status == 'COMPLETED':
            continue

        if status != 'ERROR':
            print('WARN: tvmid=%s. operation is %s' % (tvmid, status))
            continue

        print('tvmid=%s. operation is %s' % (tvmid, status))
        to_rerun_tvmids.append(tvmid)

    print('== in flight %s' % in_flight)
    print('== rerun: %s' % len(to_rerun_tvmids))

    name_by_tvmid = get_consumers(grants_file)
    for tvm in to_rerun_tvmids:
        name = name_by_tvmid[int(tvm)]
        id = create_yql_operation(name, oauth_token, use_passport_api, date_from, date_to)
        operations[tvm] = id
        print('name=\'%s\'; shared_link=\'%s\'' % (name, id))

    with open(out_operations, 'w') as f:
        f.write(json.dumps(operations, indent=4, sort_keys=True))
        f.write('\n')


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='prepares tasks for grants verification')
    parser.add_argument('--grants_file', type=str, help='file with prepared grants', required=True)
    parser.add_argument('--oauth_token', type=str, help='oauth token to use yql', required=True)
    parser.add_argument('--out_operations', type=str, help='out file with operations by tvm_id', required=True)
    parser.add_argument('--date_from', type=str, help='for example: 20210601', required=True)
    parser.add_argument('--date_to', type=str, help='for example: 20210701', required=True)
    parser.add_argument('--rerun', action='store_true', help='rerun failed operations')
    parser.add_argument('--passport_api', action='store_true', help='build for passport_api instead of blackbox')
    args = parser.parse_args()

    if args.rerun:
        rerun(
            args.grants_file,
            args.oauth_token,
            args.out_operations,
            args.passport_api,
            args.date_from,
            args.date_to,
        )
    else:
        run(
            args.grants_file,
            args.oauth_token,
            args.out_operations,
            args.passport_api,
            args.date_from,
            args.date_to,
        )
