import argparse
import json
import socket
import sys
import xml.etree.ElementTree as xml_

import requests


TOKEN_HEADER = None
ABC_INFO = {}
CONDUCTOR_INFO = {}
FW_INFO = {
    '_BALANCEBCK_': {'fw_owners': ['lazareva', 'fedusia', 'iandreyev', 'librarian']},
}
HOST_INFO = {}
STAFF_INFO = {}
USER_IS_BAD = {}

SKIP_IP_CONSUMERS = []


def ip_to_hostname(ip):
    try:
        return socket.gethostbyaddr(ip)[0]
    except:
        pass


def try_do(func):
    tries = 5
    while tries > 0:
        tries = tries - 1
        try:
            return func()
        except Exception as e:
            print('exception: %s' % (e), file=sys.stderr)
            # raise


def get_abc_info(abc_id):
    if abc_id in ABC_INFO:
        return ABC_INFO[abc_id]

    def foo():
        next = (
            'https://abc-back.yandex-team.ru/api/v3/services/members/?role__scope=administration&role__scope=services_management&service=%s'
            % abc_id
        )
        res = {
            'abc_admins': [],
            'abc_managers': [],
        }

        while next is not None:
            r = requests.get(next, headers=TOKEN_HEADER, timeout=1)
            if r.status_code != 200 and r.status_code < 500:
                print('abc service %s was not found (%s)' % (abc_id, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            next = j['next']

            res['abc_admins'] = res['abc_admins'] + [
                a['person']['login'] for a in j['results'] if a['role']['scope']['slug'] == 'administration'
            ]
            res['abc_managers'] = res['abc_managers'] + [
                a['person']['login'] for a in j['results'] if a['role']['scope']['slug'] == 'services_management'
            ]

        return res

    res = try_do(foo)
    ABC_INFO[abc_id] = res
    return res


def filter_out_bad_users(logins):
    to_check = []
    result = []
    for log in logins:
        if log in USER_IS_BAD:
            if not USER_IS_BAD[log]:
                result.append(log)
        else:
            to_check.append(log)

    if len(to_check) == 0:
        return result

    def foo():
        next = (
            'https://staff-api.yandex-team.ru/v3/persons?_fields=official.is_dismissed,official.is_robot,login&_limit=100&login=%s'
            % (','.join(to_check))
        )

        while next is not None:
            r = requests.get(next, headers=TOKEN_HEADER, timeout=1)
            if r.status_code != 200 and r.status_code < 500:
                print('logins in staff-api %s was not found (%s)' % (to_check, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            next = None

            res = {}
            for r in j['result']:
                log = r['login']
                is_dismissed = r['official']['is_dismissed']
                is_robot = r['official']['is_robot']
                failed = is_dismissed or is_robot
                if failed:
                    print('%s is dismissed: %s. is robot: %s' % (log, is_dismissed, is_robot), file=sys.stderr)

                res[log] = failed

            for log in to_check:
                USER_IS_BAD[log] = res[log]

    try_do(foo)
    return [a for a in logins if not USER_IS_BAD[a]]


class IpStats:
    def __init__(self):
        self._consumers = {}
        self._networks = {}

    def build(self):
        res = {}

        for name, meta in self._consumers.items():
            nets = self._networks.get(name)
            if nets is None:
                print('no network for %s' % name)
                continue

            r = self.resolve(name, nets)
            out = {'grants': meta, 'networks': nets}

            def add_users(type):
                if type in r and len(r[type]) > 0:
                    out[type] = [x for x in sorted(filter_out_bad_users(list(set(r[type]))))]

            add_users('c_admins')
            add_users('abc_admins')
            add_users('abc_managers')
            add_users('abc_owners')
            add_users('fw_owners')
            add_users('host_owners')
            add_users('staff_owners')

            if (
                'c_admins' not in out
                and 'abc_admins' not in out
                and 'abc_owners' not in out
                and 'fw_owners' not in out
                and 'host_owners' not in out
                and 'staff_owners' not in out
                and 'abc_managers' not in out
            ):
                print('nobody for %s' % r['name'])

            res[r['name']] = out

        return res

    def add(self, consumer, grants):
        if consumer not in self._consumers:
            self._consumers[consumer] = grants

    def tsv_to_macro(self, f):
        for line in open(f):
            macro, type_, name = line.strip().split('\t')

            if name not in self._consumers:
                continue

            d = self._networks.setdefault(name, dict())
            if type_ not in ['F', 'C', 'I', 'H', 'N']:
                raise Exception(line)

            d.setdefault(type_, list()).append(macro)

    def get_conductor_info(self, macro):
        if macro in CONDUCTOR_INFO:
            return CONDUCTOR_INFO[macro]

        def foo():
            r = requests.get(
                'https://c.yandex-team.ru/api/v2/groups/%s/project' % macro.strip('%'), headers=TOKEN_HEADER, timeout=1
            )
            if r.status_code != 200 and r.status_code < 500:
                print('conductor macro %s was not found (%s)' % (macro, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            abc = j['data']['attributes']['abc_service_id']
            if abc is None:
                print('conductor macro %s with none abc' % (macro), file=sys.stderr)
            admins_link = j['data']['relationships']['admins']['links']['related']

            return abc, admins_link

        res = try_do(foo)
        if res is None:
            return

        abc, admins_link = res
        if abc is not None:
            CONDUCTOR_INFO[macro] = {'abc': abc}
            return {'abc': abc}

        def foo():
            r = requests.get(admins_link, headers=TOKEN_HEADER, timeout=1)
            if r.status_code != 200 and r.status_code < 500:
                print('conductor link was bad: %s (%s)' % (admins_link, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            admins = []
            for d in j['data']:
                admins.append(d['attributes']['login'])

            return admins

        admins = try_do(foo)
        if admins is None or len(admins) == 0:
            print('conductor macro %s has no admins' % macro, file=sys.stderr)
            return None

        print('conductor macro %s has admins: %s' % (macro, admins), file=sys.stderr)
        CONDUCTOR_INFO[macro] = {'c_admins': admins}
        return {'c_admins': admins}

    def get_firewall_info(self, macro):
        if macro in FW_INFO:
            return FW_INFO[macro]

        def foo():
            r = requests.get(
                'https://racktables.yandex.net/export/project-id-networks.php?op=show_macro&name=%s' % macro,
                headers=TOKEN_HEADER,
                timeout=1,
            )
            if r.status_code != 200 and r.status_code < 500:
                print('admins for macro faield to be got: %s (%s)' % (macro, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return j['owners']

        res = try_do(foo)
        if res is None:
            FW_INFO[macro] = res
            return

        print('macro -> owners: %s -> %s' % (macro, res), file=sys.stderr)
        res = {'fw_owners': res}
        FW_INFO[macro] = res
        return res

    def get_firewall_hosts(self, macro):
        def foo():
            r = requests.get('http://hbf.yandex.net/macros/%s' % macro, timeout=1)
            if r.status_code != 200 and r.status_code < 500:
                print('hosts for macro faield to be got: %s (%s)' % (macro, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return j

        res = try_do(foo)
        if res is None:
            return

        print('macro -> hosts: %s -> %s' % (macro, res), file=sys.stderr)
        return res

    def get_host_info(self, host):
        if host in HOST_INFO:
            return HOST_INFO[host]

        def foo():
            r = requests.get('http://ro.admin.yandex-team.ru/api/get_object_resps.sbml?object=%s' % host, timeout=1)
            if r.status_code != 200 and r.status_code < 500:
                print('admins for host faield to be got: %s (%s)' % (host, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return [a['name'] for a in j['resps'] + j['resps_expand'] if a['type'] == 'user']

        res = try_do(foo)
        if res is None:
            return

        print('host -> owners: %s -> %s' % (host, res), file=sys.stderr)

        res = {'host_owners': res}
        HOST_INFO[host] = res
        return res

    def get_staff_group_members(self, group):
        if group in STAFF_INFO:
            return STAFF_INFO[group]

        def foo():
            r = requests.get(
                'https://staff-api.yandex-team.ru/v3/groupmembership?_fields=person.login&person.official.is_dismissed=false&group.url=%s'
                % group,
                headers=TOKEN_HEADER,
                timeout=1,
            )
            if r.status_code != 200 and r.status_code < 500:
                print('users for group failed to be got: %s (%s)' % (group, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return [a['person']['login'] for a in j['result']]

        res = try_do(foo)
        if res is None:
            return

        print('group -> users: %s -> %s' % (group, res), file=sys.stderr)

        res = {'owners': res}
        STAFF_INFO[group] = res
        return res

    def resolve(self, name, nets):
        print('===getting "%s" with meta %s' % (name, nets), file=sys.stderr)

        res = {
            'name': name,
            'c_admins': [],
            'abc_admins': [],
            'abc_managers': [],
            'abc_owners': [],
            'fw_owners': [],
            'staff_owners': [],
            'host_owners': [],
        }

        def proc(type, macro):
            if 'C' == type:
                info = self.get_conductor_info(macro)
                if info is None:
                    print('no info for macro %s (%s)' % (macro, name), file=sys.stderr)
                    return

                if 'abc' in info:
                    r = get_abc_info(info['abc'])
                    if r is None:
                        res['error'] = 'macro has bad abc: %s' % info
                        print('macro has bad abc: %s (%s): %s' % (macro, name, info), file=sys.stderr)
                        return
                    res['abc_admins'] = res['abc_admins'] + r['abc_admins']
                    res['abc_managers'] = res['abc_managers'] + r['abc_managers']
                else:
                    res['c_admins'] = res['c_admins'] + info['c_admins']
            elif 'F' == type:
                info = self.get_firewall_info(macro)
                if info is not None:
                    res['fw_owners'] = res['fw_owners'] + [
                        a for a in info['fw_owners'] if not a.startswith('yandex_') and not a.startswith('svc_')
                    ]
                    for gr in [a for a in info['fw_owners'] if a.startswith('yandex_')]:
                        r = self.get_staff_group_members(gr)
                        if r is None:
                            print('group is empty: %s: %s' % (macro, gr), file=sys.stderr)
                        else:
                            res['staff_owners'] = res['staff_owners'] + r['owners']

                    for gr in [a for a in info['fw_owners'] if a.startswith('svc_')]:
                        r = self.get_staff_group_members(gr)
                        if r is None:
                            print('group is empty: %s: %s' % (macro, gr), file=sys.stderr)
                        else:
                            res['abc_owners'] = res['abc_owners'] + r['owners']
                    return
                print('no info for macro %s (%s)' % (macro, name), file=sys.stderr)

                hosts = self.get_firewall_hosts(macro)
                if hosts is None:
                    print('no hosts in macro %s (%s)' % (macro, name), file=sys.stderr)
                    return

                for host in hosts:
                    adm = self.get_host_info(host)
                    if adm is None:
                        print('no adm for host %s (%s)' % (host, name), file=sys.stderr)
                        continue

                    res['host_owners'] = res['host_owners'] + adm['host_owners']
            elif 'I' == type:
                if macro == '::1':
                    res['host_owners'] = res['host_owners'] + ['avmm']
                    return

                h = ip_to_hostname(macro)
                if h is None:
                    print('bad ip %s (%s)' % (macro, name), file=sys.stderr)
                    return

                adm = self.get_host_info(h)
                if adm is None:
                    print('no info for macro %s (%s)' % (macro, name), file=sys.stderr)
                    return

                res['host_owners'] = res['host_owners'] + adm['host_owners']
            elif 'H' == type:
                adm = self.get_host_info(macro)
                if adm is None:
                    print('no info for macro %s (%s)' % (macro, name), file=sys.stderr)
                    return

                res['host_owners'] = res['host_owners'] + adm['host_owners']
            else:
                raise Exception('unknown type %s for %s' % (type, name))

        for type, m in nets.items():
            for macro in m:
                proc(type, macro)

        return res


def parse_ip_grants(f):
    ip_grants = {}

    for ent in xml_.fromstring(f).findall('entry'):
        grants = {}
        for e in ent:
            if e.tag == 'dbfield' or e.tag == 'name':
                continue

            value = e.text or True
            if e.tag == 'allowed_attributes' or e.tag == 'allowed_phone_attributes':
                value = [int(a) for a in value.split(',')]

            grants[e.tag] = value

        dbf = [a.attrib.get('id') for a in ent.findall('dbfield')]
        if len(dbf) > 0:
            grants['dbfields'] = dbf

        if len(grants) == 0:
            continue

        name = ent.find('name').text

        ip_grants[name] = grants

    return ip_grants


def download_grants(git_hash):
    url = 'https://raw.github.yandex-team.ru/passport/blackbox-grants/%s/grants/grants-prod.conf'
    url = url % git_hash
    return requests.get(url).text


def get_grants(git_hash_x, git_hash_y):
    grants_x = parse_ip_grants(download_grants(git_hash_x))
    grants_y = parse_ip_grants(download_grants(git_hash_y))

    print("len grants_x: %s" % len(grants_x))

    result = {}
    for name, v in grants_x.items():
        if name in grants_y:
            result[name] = v
    print("len after filtering too young: %s" % len(result))

    return result


def run(
    git_hash_x,
    git_hash_y,
    networks,
    oauth_token,
    out_grants,
):
    grants = get_grants(git_hash_x, git_hash_y)

    global TOKEN_HEADER
    TOKEN_HEADER = {'Authorization': 'OAuth %s' % oauth_token}

    ip_grants = IpStats()
    for name, v in grants.items():
        ip_grants.add(name, v)

    ip_grants.tsv_to_macro(networks)

    # dummy
    to_print = ip_grants.build()

    with open(out_grants, 'w') as f:
        f.write(json.dumps(to_print, indent=4, sort_keys=True))
        f.write('\n')


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='prepares tasks for grants verification')
    parser.add_argument('--git_hash_x', type=str, help='git hash with grants - X. X > Y', required=True)
    parser.add_argument('--git_hash_y', type=str, help='git hash with grants - Y. X > Y', required=True)
    parser.add_argument('--networks', type=str, help='file with networks from grantushka', required=True)
    parser.add_argument('--oauth_token', type=str, help='oauth token to use abc-back', required=True)
    parser.add_argument('--out_grants', type=str, help='out file with grants', required=True)
    args = parser.parse_args()

    run(
        args.git_hash_x,
        args.git_hash_y,
        args.networks,
        args.oauth_token,
        args.out_grants,
    )
