import argparse
import json
import sys

import requests


def download_grants(git_hash, use_passport_api):
    if use_passport_api:
        url = 'https://raw.github.yandex-team.ru/passport/passport-grants/%s/grants/consumer_grants.production.json'
    else:
        url = 'https://raw.github.yandex-team.ru/passport/blackbox-by-client-grants/%s/grants/consumer_grants.production.json'

    url = url % git_hash

    return json.loads(requests.get(url).text)


def get_tvm_ids_from_grants(grants):
    return [str(v['client']['client_id']) for k, v in grants.items() if 'client_id' in v['client']]


def get_old_grants(git_hash_x, git_hash_y, use_passport_api):
    grants_x = download_grants(git_hash_x, use_passport_api)
    grants_y = download_grants(git_hash_y, use_passport_api)

    tvm_ids_x = set(get_tvm_ids_from_grants(grants_x))
    print("len grants_x: %s" % len(tvm_ids_x))
    tvm_ids_y = set(get_tvm_ids_from_grants(grants_y))

    result_tvm_ids = set([i for i in tvm_ids_x if i in tvm_ids_y])
    return grants_x, result_tvm_ids


def filter_grants_from_file(tvm_ids, skip_tvm_ids_file):
    for line in open(skip_tvm_ids_file):
        try:
            tvm_id, comment = line.split(';')
            tvm_id = tvm_id.strip()
            if tvm_id not in tvm_ids:
                raise Exception('missing tvm_id in grants: %s' % tvm_id)

            tvm_ids.remove(tvm_id)
        except Exception as e:
            print('Failed to process line from %s: %s' % (skip_tvm_ids_file, e))

    return tvm_ids


def get_grants(git_hash_x, git_hash_y, skip_tvm_ids_file, use_passport_api):
    grants, result_tvm_ids = get_old_grants(git_hash_x, git_hash_y, use_passport_api)
    print("len after filtering too young: %s" % len(result_tvm_ids))

    if skip_tvm_ids_file is not None:
        result_tvm_ids = filter_grants_from_file(result_tvm_ids, skip_tvm_ids_file)
        print("len after filtering from file: %s" % len(result_tvm_ids))

    result = {}
    for k, v in grants.items():
        if 'client_id' not in v['client']:
            continue

        if str(v['client']['client_id']) in result_tvm_ids and len(v['networks']) > 0:
            result[k] = v
    print("len after skipping empty: %s" % len(result))

    return result


def get_abc_service_by_tvm_ids(tvm_ids, oauth_token):
    abcid_by_tvmid = {}

    page_size = 100

    def chunks():
        for i in range(0, len(tvm_ids), page_size):
            yield tvm_ids[i : i + page_size]

    for id_range in chunks():
        url = (
            'https://abc-back.yandex-team.ru/api/v4/resources/consumers/'
            + '?supplier=14'
            + '&type=47'
            + '&fields=service.id,resource.external_id'
            + '&resource__external_id__in='
            + ','.join(id_range)
        )

        tries = 5
        while tries > 0:
            tries = tries - 1
            try:
                r = requests.get(url, headers={'Authorization': 'OAuth %s' % oauth_token})
                assert r.status_code == 200, r.text
                d = json.loads(r.text)
                for result in d['results']:
                    abcid_by_tvmid[result['resource']['external_id']] = result['service']['id']
                break
            except Exception as e:
                print('Failed to get abc ids by tvm id: %s: %s' % (url, e), file=sys.stderr)

    return abcid_by_tvmid


def get_abc_ids_by_tvm_ids(tvm_ids, tvm_abc_mapping, oauth_token):
    abcid_by_tvmid = get_abc_service_by_tvm_ids(tvm_ids, oauth_token)

    for line in open(tvm_abc_mapping):
        try:
            tvm_id, abc_id, comment = line.split(';')
            tvm_id = tvm_id.strip()
            abc_id = int(abc_id.strip())

            if tvm_id not in abcid_by_tvmid and tvm_id not in tvm_ids:
                continue
                # raise Exception('tvm_id missing in grants: %s' % tvm_id)

            abcid_by_tvmid[tvm_id] = abc_id
        except Exception as e:
            print('Failed to process line from %s: %s' % (tvm_abc_mapping, e), file=sys.stderr)

    return abcid_by_tvmid


def print_grants(grants, tvm_by_abc, networks, out_grants):
    grant_by_tvmid = {}
    for name, gr in grants.items():
        del gr['networks']

        for k, v in networks[name].items():
            gr.setdefault('networks', dict())[k] = v

        for k, v in gr['grants'].items():
            if v == ['*']:
                gr['grants'][k] = True

        grant_by_tvmid[int(gr['client']['client_id'])] = {
            'name': name,
            'grants': gr,
        }

    to_print = {}
    for abc, tvm_ids in tvm_by_abc.items():
        to_print[abc] = [grant_by_tvmid[int(tvm)] for tvm in tvm_ids]

    with open(out_grants, 'w') as f:
        f.write(json.dumps(to_print, indent=4, sort_keys=True))
        f.write('\n')


def get_abc_roles(abc_ids, oauth_token):
    info = {}
    abc_ids = [str(i) for i in abc_ids if i is not None]
    page_size = 100

    next_url = (
        'https://abc-back.yandex-team.ru/api/v4/services/members/'
        + '?page_size=%s' % page_size
        + '&role__code=tvm_manager'
        + '&ordering=id'
        + '&fields=service.id,person.login'
        + '&service__in='
        + ','.join(abc_ids)
    )

    while next_url:
        tries = 5
        while tries > 0 and next_url:
            tries = tries - 1
            try:
                r = requests.get(next_url, headers={'Authorization': 'OAuth %s' % oauth_token})
                assert r.status_code == 200, r.text
                d = json.loads(r.text)
                next_url = d['next']
                for result in d['results']:
                    info.setdefault(result['service']['id'], set()).add(result['person']['login'])
                break
            except Exception as e:
                print('Failed to get abc roles: %s: %s' % (next_url, e), file=sys.stderr)

    return info


def get_abc_info(abc_ids, oauth_token):
    info = {}
    abc_ids = [str(i) for i in abc_ids if i is not None]
    page_size = 100

    def chunks():
        for i in range(0, len(abc_ids), page_size):
            yield abc_ids[i : i + page_size]

    for id_range in chunks():
        url = (
            'https://abc-back.yandex-team.ru/api/v4/services/'
            + '?page_size=%s' % page_size
            + '&ordering=id'
            + '&role__code=tvm_manager'
            + '&fields=id,name.ru'
            + '&id__in='
            + ','.join(id_range)
        )

        tries = 5
        while tries > 0:
            tries = tries - 1
            try:
                r = requests.get(url, headers={'Authorization': 'OAuth %s' % oauth_token})
                assert r.status_code == 200, r.text
                d = json.loads(r.text)
                for result in d['results']:
                    info[result['id']] = {
                        'name': result['name']['ru'],
                    }
                break
            except Exception as e:
                print('Failed to get abc parents: %s: %s' % (url, e), file=sys.stderr)

    return info


def parse_networks(networks_file):
    res = {}

    for line in open(networks_file):
        macro, type_, name = line.strip().split('\t')

        if name == 'yamoney-frontend':
            name = 'yamoney-frontend-credit'

        d = res.setdefault(name, dict())
        if type_ not in ['F', 'C', 'I', 'H', 'N']:
            raise Exception(line)

        d.setdefault(type_, list()).append(macro)

    return res


def print_abc(abc_ids, abc_managers, tvm_by_abc, oauth_token, out_abc):
    roles = get_abc_roles(abc_ids, oauth_token)
    info = get_abc_info(abc_ids, oauth_token)

    for line in open(abc_managers):
        try:
            abc_id, managers, comment = line.split(';')
            abc_id = int(abc_id.strip())

            roles[abc_id] = [s.strip() for s in managers.split(',')]
        except Exception as e:
            print('Failed to process line from %s: %s' % (abc_managers, e), file=sys.stderr)

    to_print = {}
    for abc in abc_ids:
        managers = roles.get(int(abc))
        if managers is None:
            print('ABC has no manager: %s - by tvm_id: %s' % (abc, tvm_by_abc.get(abc)), file=sys.stderr)
        else:
            managers = list(managers)

        inf = info.get(int(abc))
        if inf is None:
            print('ABC has no name: %s. Skipping it' % abc, file=sys.stderr)
            continue

        to_print[abc] = {
            'name': inf['name'],
            'managers': managers,
        }

    with open(out_abc, 'w') as f:
        f.write(json.dumps(to_print, indent=4, sort_keys=True, ensure_ascii=False))
        f.write('\n')


def run(
    git_hash_x,
    git_hash_y,
    skip_tvm_ids_file,
    tvm_abc_mapping,
    abc_managers,
    networks,
    oauth_token,
    out_grants,
    out_abc,
    use_passport_api,
):
    grants = get_grants(git_hash_x, git_hash_y, skip_tvm_ids_file, use_passport_api)
    tvm_ids = get_tvm_ids_from_grants(grants)

    abc_by_tvm = get_abc_ids_by_tvm_ids(tvm_ids, tvm_abc_mapping, oauth_token)
    missing_tvmids = [tvm for tvm in tvm_ids if tvm not in abc_by_tvm]
    if len(missing_tvmids) > 0:
        print('Missing tvm_ids in ABC: %s' % missing_tvmids, file=sys.stderr)
    print('TVM clients: %s' % len(abc_by_tvm))

    tvm_by_abc = {}
    for tvm, abc in abc_by_tvm.items():
        tvm_by_abc.setdefault(abc, []).append(tvm)
    print('ABC services: %s' % len(tvm_by_abc))

    print_grants(grants, tvm_by_abc, parse_networks(networks), out_grants)
    abc_ids = [abc for abc, tvm in tvm_by_abc.items()]
    print_abc(abc_ids, abc_managers, tvm_by_abc, oauth_token, out_abc)


if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='prepares tasks for grants verification')
    parser.add_argument('--git_hash_x', type=str, help='git hash with grants - X. X > Y', required=True)
    parser.add_argument('--git_hash_y', type=str, help='git hash with grants - Y. X > Y', required=True)
    parser.add_argument('--skip_tvm_ids', type=str, help='file with tvm ids to skip', default=None)
    parser.add_argument('--tvm_abc_mapping', type=str, help='file with tvm<->abc mapping overrides', required=True)
    parser.add_argument('--abc_managers', type=str, help='file with abc manager overridings', required=True)
    parser.add_argument('--networks', type=str, help='file with networks from grantushka', required=True)
    parser.add_argument('--oauth_token', type=str, help='oauth token to use abc-back', required=True)
    parser.add_argument('--out_grants', type=str, help='out file with grants', required=True)
    parser.add_argument('--out_abc', type=str, help='out file with abc info', required=True)
    parser.add_argument('--passport_api', action='store_true', help='build for passport_api instead of blackbox')
    args = parser.parse_args()

    run(
        args.git_hash_x,
        args.git_hash_y,
        args.skip_tvm_ids,
        args.tvm_abc_mapping,
        args.abc_managers,
        args.networks,
        args.oauth_token,
        args.out_grants,
        args.out_abc,
        args.passport_api,
    )
