#include "common.h"

#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/request/test/request.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/env.h>

#include <util/stream/file.h>
#include <util/stream/output.h>
#include <util/string/subst.h>
#include <util/system/env.h>

namespace NPassport::NBb {
    TString TestsDir() {
        return ArcadiaSourceRoot() + "/passport/infra/daemons/blackbox/ut/";
    }

    void PrepareConfig(const TString& name, const TString& testPath) {
        const TString& testdir = TestsDir();

        TFileInput in((TString(testdir + testPath)));
        TString contents = in.ReadAll();
        SubstGlobal(contents, "{tests_dir}", testdir);
        SubstGlobal(contents, "{arcadia_root}", ArcadiaSourceRoot());

        TFileOutput out((TString(name)));
        out << contents;
    }

    const TDbProfile& TTestDbFetcher::DefaultProfile() const {
        return Profiles_.front();
    }

    void TTestDbFetcher::SetTotpEncryptor(const TTotpEncryptor& encryptor) {
        TotpEncryptor_ = &encryptor;
    }

    const TString xpathPrefix = "/config/components/component[@name='blackbox']";

    TTestDbHolder::TTestDbHolder() {
        const NDbPool::TDestinationPtr dsn = NDbPool::TDestination::CreateSqlite(TestsDir() + "db/safeguarddb.sqlite3.sql");
        Pool_ = std::make_unique<NDbPool::TDbPool>(NDbPool::TDbPoolSettings{
            .Dsn = dsn,
            .Size = 2,
        });

        PrepareConfig("./blackbox.conf", "data/blackbox-test.conf");
        auto config = NXml::TConfig::ReadFromFile("./blackbox.conf");

        auto ctx = NDbPool::TDbPoolCtx::Create();
        DbShards_ = std::make_unique<TRangedShardsMap>();
        DbShards_->Init(xpathPrefix, config, ctx);

        Hosts_ = std::make_unique<THostsList>(*Pool_);

        OauthLog_ = std::make_unique<TTskvLog>(std::make_unique<NUtils::TFileLogger>("./blackbox-oauth.log", "DEBUG", false));

        OauthConfig_ = std::make_unique<TOAuthConfig>(*OauthLog_);
        OauthConfig_->Init(config, xpathPrefix, ctx);

        SetLocalTimeZone();
    }

    std::unique_ptr<TTestDbFetcher> TTestDbHolder::CreateFetcher() {
        return std::make_unique<TTestDbFetcher>(*Pool_, *DbShards_, nullptr, NotImplementedDomainFetcher_, nullptr, TSyntheticAttributes(0, {}), nullptr);
    }

    std::unique_ptr<TTestDbFetcher> TTestDbHolder::CreateFetcher(IDomainFetcher& domainfetcher) {
        return std::make_unique<TTestDbFetcher>(*Pool_, *DbShards_, nullptr, domainfetcher, nullptr, TSyntheticAttributes(0, {}), nullptr);
    }

    std::unique_ptr<TDbFieldsConverter> TTestDbHolder::CreateConverter(TDbFetcher& fetcher) {
        return std::make_unique<TDbFieldsConverter>(fetcher, *Hosts_, "1234");
    }

    std::unique_ptr<TDbFieldsConverter> TTestDbHolder::CreateConverter(TDbFetcher& fetcher, const std::vector<TString>& sids) {
        return std::make_unique<TDbFieldsConverter>(fetcher, *Hosts_, "1234", sids);
    }

    std::unique_ptr<TTestOAuthFetcher> TTestDbHolder::CreateOAuthFetcher() {
        return std::make_unique<TTestOAuthFetcher>(*OauthConfig_, "127.0.0.11", "yabro", "127.0.0.1");
    }

    const TOAuthConfig& TTestDbHolder::GetOAuthConfig() const {
        return *OauthConfig_;
    }

    TTestDbHolder& TTestDbHolder::GetSingleton() {
        return *Singleton<TTestDbHolder>();
    }

    std::unique_ptr<TTestDbFetcher> TTestDbHolder::GetMockedDbFetcher(TDomainFetcherMock& domainFetcher, TString domainId, bool addAliases) {
        if (!domainId.empty()) {
            domainFetcher.SetResult(domainId);
        }

        std::unique_ptr dbFetcher = CreateFetcher(domainFetcher);

        if (addAliases) {
            dbFetcher->AddAlias(TAlias::PDD_ALIAS_LOGIN);
            dbFetcher->AddAlias(TAlias::PORTAL_LOGIN);
        }
        return dbFetcher;
    }

    void TTestDbProfile::SetUid(const TString& uid) {
        Uid_ = uid;
    }

    TDbIndex TTestDbProfile::AddAlias(const TString& type, const TString& val) {
        TDbIndex result = TDbProfile::AddAlias(type);
        Aliases_[type].Value = val;
        Aliases_[type].Exists = true;
        return result;
    }

    void TTestDbProfile::AddPddAlias(const TString& val) {
        TDbProfile::AddAlias(TAlias::PDD_ALIAS_LOGIN);
        AliasesPdd_.insert(val);
    }

    void TTestDbProfile::AddOldPublicAlias(const TString& val) {
        TDbProfile::AddAlias(TAlias::OLDPUBLICID);
        AliasesOldPublic_.insert(val);
    }

    TDbIndex TTestDbProfile::AddAttr(const TString& type, const TString& val) {
        TDbIndex result = TDbProfile::AddAttr(type);
        Attrs_[type].Value = val;
        Attrs_[type].Exists = true;
        return result;
    }

    void TTestDbProfile::AddPhoneAttr(const TString& id, const TString& type, const TString& val) {
        AddExtendedPhoneAttr(type);
        ExtendedPhones_[id][type].Value = val;
        ExtendedPhones_[id][type].Exists = true;
    }

    void TTestDbProfile::AddEmailAttr(const TString& id, const TString& type, const TString& val) {
        AddExtendedEmailAttr(type);
        ExtendedEmails_[id][type].Value = val;
        ExtendedEmails_[id][type].Exists = true;
    }

    void TTestDbProfile::AddWebauthnAttr(const TString& id, const TString& type, const TString& val) {
        AddExtendedWebauthnAttr(type);
        ExtendedWebauthn_[id][type].Value = val;
        ExtendedWebauthn_[id][type].Exists = true;
    }

    void TTestDbProfile::FillWithSampleData() {
        Empty_ = false;
        Uid_ = "70501";
        Sid_ = "2";

        DomCache_ = std::make_shared<TDomainCache>();
        DomCache_->LastEventId = "6666";
        DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "356",
            },
        });

        AltDomainId_ = "3";
        AltDomain_ = "mail.ru";
        CatchAll_ = true;

        AddAlias(TAlias::PORTAL_LOGIN, "test-user");
        AddAlias(TAlias::MAIL_LOGIN, "");
        AddAlias(TAlias::PDD_MASTER_LOGIN, "gendir@рога-и-копыта.рф");
        AddPddAlias("admin@рога-и-копыта.рф");
        AddPddAlias("glavbukh@рога-и-копыта.рф");
        AddPddAlias("janitor@рога-и-копыта.рф");
        AddOldPublicAlias("old_public_id_1");
        AddOldPublicAlias("old_public_id_2");

        AddAttr(TAttr::ACCOUNT_HAVE_HINT, "0");
        AddAttr(TAttr::ACCOUNT_HAVE_PASSWORD, "1");
        AddAttr(TAttr::KARMA_VALUE, "100");
        TDbProfile::AddAttr(TAttr::ACCOUNT_BETATESTER);             // NULL value
        TDbProfile::AddAttr(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME); // NULL value

        AddSuid2();
        Suid2_.Value = "12341234";
        Suid2_.Exists = true;

        AddPhoneAttr("5", TPhoneAttr::NUMBER, "79161234567");
        AddPhoneAttr("5", TPhoneAttr::CONFIRMED, "1414141414");

        AddEmailAttr("2", TEmailAttr::ADDRESS, "user@gmail.com");
        AddEmailAttr("2", TEmailAttr::CONFIRMED, "1212121212");
        AddEmailAttr("2", TEmailAttr::IS_UNSAFE, "1");

        AddWebauthnAttr("10", TWebauthnAttr::EXTERNAL_ID, "asdfasdfasdf");
        AddWebauthnAttr("10", TWebauthnAttr::CREATED, "1616161616");

        AddPhoneOperations();
        PhoneOperations_["77"] = "1,2,3,4";
        PhoneOperations_["177"] = "51,52,53,,,54";

        AddPhoneBindings(EPhoneBindingsType::Current);
        PhoneBindings_.push_back({"current", "79162222222", "1", "70501", "1400554360", "1"});

        SetAttrValue(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME, "1");
        SetAttrValue(TAttr::PERSON_LANGUAGE, "en");

        DefaultPhoneId_ = "5";
    }

    bool TTestDbProfile::IsEmpty() const {
        return Empty_;
    }

    TTotpEncryptor CreateTotpEncryptor() {
        return TTotpEncryptor(NAuth::TKeyMap::CreateFromFile(TestsDir() + "data/totp_aes.keys", 2),
                              NAuth::TKeyMap::CreateFromFile(TestsDir() + "data/totp_hmac.keys", 5));
    }

    void SetLocalTimeZone() {
        SetEnv("TZ", "Europe/Moscow");
        tzset();
    }

    TBbHolder::TBbHolder() {
        InitOpenSSL();
        const TString tmpname = "./blackbox.conf";

        PrepareConfig(tmpname, "data/blackbox-test.conf");

        Bb = std::make_unique<NPassport::NBb::TBlackbox>();
        UNIT_ASSERT_NO_EXCEPTION(Bb->Init(NXml::TConfig::ReadFromFile(tmpname)));
    }
}

template <>
void Out<NPassport::NBb::TDbValue>(IOutputStream& o, const NPassport::NBb::TDbValue& value) {
    o << "dbvalue=" << value.Value
      << ". dbindex=" << value.Index
      << ". exists=" << value.Exists
      << Endl;
}

using TStringSet = std::unordered_set<TString>;
template <>
void Out<TStringSet>(IOutputStream& out, const TStringSet& value) {
    out << '{';
    bool first = true;
    for (const auto& val : value) {
        if (!first) {
            out << ", ";
        } else {
            first = false;
        }
        out << '"' << val << '"';
    }
    out << '}';
}

template <>
void Out<std::set<TString>>(IOutputStream& out, const std::set<TString>& value) {
    for (const auto& val : value) {
        out << val << ",";
    }
}

using TStringMap = std::map<TString, TString>;
template <>
void Out<TStringMap>(IOutputStream& out, const TStringMap& value) {
    out << '{';
    bool first = true;
    for (const auto& pair : value) {
        if (!first) {
            out << ", ";
        } else {
            first = false;
        }
        out << '"' << pair.first << '"';
        out << ':';
        out << '"' << pair.second << '"';
    }
    out << '}';
}

using TStringMultimap = std::multimap<TString, TString>;
template <>
void Out<TStringMultimap>(IOutputStream& out, const TStringMultimap& value) {
    out << '{';
    bool first = true;
    for (const auto& pair : value) {
        if (!first) {
            out << ", ";
        } else {
            first = false;
        }
        out << '"' << pair.first << '"';
        out << ':';
        out << '"' << pair.second << '"';
    }
    out << '}';
}

using TDbValueMap = std::map<TString, NPassport::NBb::TDbValue>;
template <>
void Out<TDbValueMap>(IOutputStream& out, const TDbValueMap& value) {
    out << '{';
    bool first = true;
    for (const auto& pair : value) {
        if (!first) {
            out << ", ";
        } else {
            first = false;
        }
        out << '"' << pair.first << '"';
        out << ':';
        out << '"' << pair.second.AsString() << '"';
    }
    out << '}';
}

using TExtendedMap = std::map<TString, TDbValueMap>;
template <>
void Out<TExtendedMap>(IOutputStream& out, const TExtendedMap& value) {
    out << '{';
    bool first = true;
    for (const auto& pair : value) {
        if (!first) {
            out << ", ";
        } else {
            first = false;
        }
        out << '"' << pair.first << '"';
        out << " : ";
        Out<TDbValueMap>(out, pair.second);
    }
    out << '}';
}
