#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/libs/cpp/request/test/request.h>
#include <passport/infra/libs/cpp/unistat/builder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspUtCommon) {
    Y_UNIT_TEST(testRequest) {
        NTest::TRequest r;

        r.SetStatus(HTTP_PARTIAL_CONTENT);
        r.SetHeader("any string", "any other string");
        r.Write("hello");
        r.SetCookie(NPassport::NCommon::TCookie("foo", "bar"));

        UNIT_ASSERT(!r.HasHeader(""));
        UNIT_ASSERT(!r.HasHeader("any string"));

        UNIT_ASSERT_VALUES_EQUAL("", r.GetHeader(""));
        UNIT_ASSERT_VALUES_EQUAL("", r.GetHeader("any string"));

        UNIT_ASSERT(r.HasCookie(""));
        UNIT_ASSERT(r.HasCookie("any string"));

        UNIT_ASSERT_VALUES_EQUAL("", r.GetCookie(""));
        UNIT_ASSERT_VALUES_EQUAL("", r.GetCookie("any string"));

        UNIT_ASSERT_VALUES_EQUAL("", r.GetRemoteAddr());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetUri());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetRequestId());
        UNIT_ASSERT_VALUES_EQUAL("/blackbox", r.GetPath());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetQueryString());
        UNIT_ASSERT_VALUES_EQUAL("GET", r.GetRequestMethod());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetHost());
        UNIT_ASSERT_VALUES_EQUAL(HTTP_PARTIAL_CONTENT, r.GetStatusCode());
        UNIT_ASSERT_VALUES_EQUAL(0, r.GetResponseSize());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetRequestBody());
        UNIT_ASSERT_VALUES_EQUAL("", r.GetRequestCgi());
        UNIT_ASSERT(r.IsSecure());
        UNIT_ASSERT(!r.IsBodyEmpty());

        std::vector<TString> h;
        UNIT_ASSERT(h.empty());

        r.ArgNames(h);
        UNIT_ASSERT(h.empty());
    }

    Y_UNIT_TEST(outTemplates) {
        TString buf;
        TStringOutput str(buf);

        { // std::unordered_set
            buf.clear();
            std::unordered_set<TString> v;
            v.insert("foo");
            v.insert("");

            Out<decltype(v)>(str, v);
            UNIT_ASSERT_VALUES_EQUAL(R"({"", "foo"})", buf);
        }

        { // std::map
            buf.clear();
            std::map<TString, TString> v;
            v["a"] = "";
            v["null"] = "None";

            Out<decltype(v)>(str, v);
            UNIT_ASSERT_VALUES_EQUAL(R"({"a":"", "null":"None"})", buf);
        }

        { // std::multimap
            buf.clear();
            std::multimap<TString, TString> v;
            v.insert(std::make_pair("a", ""));
            v.insert(std::make_pair("null", "None"));
            v.insert(std::make_pair("a", "b"));

            Out<decltype(v)>(str, v);
            UNIT_ASSERT_VALUES_EQUAL(R"({"a":"", "a":"b", "null":"None"})", buf);
        }

        { // std::map of DbValues
            buf.clear();
            std::map<TString, TDbValue> v;
            v["a"] = TDbValue("hi");
            v["null"] = TDbValue();

            Out<decltype(v)>(str, v);
            UNIT_ASSERT_VALUES_EQUAL(R"({"a":"hi", "null":"<NULL>"})", buf);
        }

        { // std::map of map of DbValues
            buf.clear();
            std::map<TString, std::map<TString, TDbValue>> v;
            v["0"]["a"] = TDbValue("hi");
            v["0"]["null"] = TDbValue();
            v["-"][""] = TDbValue("1");
            v[""];

            Out<decltype(v)>(str, v);
            UNIT_ASSERT_VALUES_EQUAL(R"({"" : {}, "-" : {"":"1"}, "0" : {"a":"hi", "null":"<NULL>"}})", buf);
        }
    }

    Y_UNIT_TEST(consumers) {
        TBbHolder bb;

        auto unistatExt = [&bb](const TString& path) {
            TString unistat;
            NUnistat::TBuilder builder(unistat);
            bb.Bb->AddUnistatExtended(path, builder);
            return unistat;
        };

        UNIT_ASSERT_VALUES_EQUAL(R"([])", unistatExt("/consumers/"));
        UNIT_ASSERT_VALUES_EQUAL(R"([])", unistatExt("/consumers/denied/"));

        NTest::TRequest req;
        req.Args = {
            {"method", "userinfo"},
            {"userip", "127.0.0.1"},
            {"login", "malice"},
        };
        req.InHeaders = {
            {"Host", "blackbox.yandex.net"},
        };
        req.RemoteAddr = "127.0.0.1";

        bb.Bb->HandleRequest(req);
        bb.Bb->HandleRequest(req);
        bb.Bb->HandleRequest(req);
        UNIT_ASSERT_VALUES_EQUAL(R"([["IlyaL_WinXP; zeze1; mda-dev;passportdev1_IP_dmmm",3]])",
                                 unistatExt("/consumers/"));
        UNIT_ASSERT_VALUES_EQUAL(R"([])", unistatExt("/consumers/denied/"));

        req.RemoteAddr = "87.250.235.18";
        bb.Bb->HandleRequest(req);
        bb.Bb->HandleRequest(req);

        UNIT_ASSERT_VALUES_EQUAL(R"([["zeze1_IP_dmmm",2],["IlyaL_WinXP; zeze1; mda-dev;passportdev1_IP_dmmm",3]])",
                                 unistatExt("/consumers/"));
        UNIT_ASSERT_VALUES_EQUAL(R"([["zeze1_IP_dmmm",2]])", unistatExt("/consumers/denied/"));

        req.RemoteAddr = "100.50.0.1";
        bb.Bb->HandleRequest(req);
        UNIT_ASSERT_VALUES_EQUAL(R"([["_unknown__dmmm",1],["zeze1_IP_dmmm",2],["IlyaL_WinXP; zeze1; mda-dev;passportdev1_IP_dmmm",3]])",
                                 unistatExt("/consumers/"));
        UNIT_ASSERT_VALUES_EQUAL(R"([["_unknown__dmmm",1],["zeze1_IP_dmmm",2]])", unistatExt("/consumers/denied/"));
    }
}
