#include <passport/infra/daemons/blackbox/src/gammas/fetcher.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NBb;
using namespace NPassport::NBb::NGammas;

Y_UNIT_TEST_SUITE(PasspUtFetcher) {
    Y_UNIT_TEST(getType) {
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::Session,
                                 TFetcher::GetType("session", "some path"));
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::OAuth,
                                 TFetcher::GetType("oauth", "some path"));
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::Smth,
                                 TFetcher::GetType("smth", "some path"));
        UNIT_ASSERT_VALUES_EQUAL(NAuth::EEnitityType::Other,
                                 TFetcher::GetType("other", "some path"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TFetcher::GetType("foo", "some path"),
            yexception,
            "type 'foo' is unknown in some path");
    }

    Y_UNIT_TEST(initKeyspaces) {
        const NXml::TConfig config = NXml::TConfig::ReadFromMemory(R"(
<doc>
    <serve_kspaces>
        <entry id="yandex.ru"       type="session"/>
        <entry id="yandex.kz"       type="session"/>
        <entry id="signsmth"        type="smth"/>
        <entry id="signsmth_long"   type="smth"/>
        <entry id="cookiel"         type="other"/>
    </serve_kspaces>
</doc>
)");
        UNIT_ASSERT_VALUES_EQUAL(
            NAuth::TGammaKeeperSettings::TKeyspaces({
                {"yandex_ru", NAuth::EEnitityType::Session},
                {"yandex_kz", NAuth::EEnitityType::Session},
                {"signsmth", NAuth::EEnitityType::Smth},
                {"signsmth_long", NAuth::EEnitityType::Smth},
                {"cookiel", NAuth::EEnitityType::Other},
                {"oauth", NAuth::EEnitityType::OAuth},
            }),
            TFetcher::InitKeyspaces(config, "/doc"));
    }

    Y_UNIT_TEST(initTimeBounds) {
        const NXml::TConfig config = NXml::TConfig::ReadFromMemory(R"(
<doc>
    <gamma_fetcher>
        <allow_to_check_without_gamma>
            <created_before type="session">16258389000</created_before>
            <created_before type="oauth"  >162583890</created_before>
        </allow_to_check_without_gamma>
    </gamma_fetcher>
</doc>
)");
        UNIT_ASSERT_VALUES_EQUAL(
            NAuth::TGammaKeeperSettings::TTimeBound({
                {NAuth::EEnitityType::Session, TInstant::ParseIso8601("2485-03-16T19:10:00.000000Z")},
                {NAuth::EEnitityType::OAuth, TInstant::ParseIso8601("1975-02-25T18:11:30.000000Z")},
            }),
            TFetcher::InitTimeBounds(config, "/doc"));
    }

    Y_UNIT_TEST(initSignedTypes) {
        const NXml::TConfig config = NXml::TConfig::ReadFromMemory(R"(
<doc>
    <gamma_fetcher>
        <signed_types>
            <item type="oauth">yes</item>
            <item type="session">yes</item>
            <item type="smth">no</item>
        </signed_types>
    </gamma_fetcher>
</doc>
)");
        UNIT_ASSERT_VALUES_EQUAL(
            NAuth::TGammaKeeperSettings::TSignedTypes({
                NAuth::EEnitityType::Session,
                NAuth::EEnitityType::OAuth,
            }),
            TFetcher::InitSignedTypes(config, "/doc"));
    }
}

using namespace NAuth;

template <>
void Out<TGamma>(IOutputStream& out, const TGamma& value) {
    out << "validUpperBound: ";
    if (value.ValidUpperBound) {
        out << *value.ValidUpperBound << Endl;
    } else {
        out << "NULL" << Endl;
    }

    for (auto t : value.TypesToCheck) {
        out << "type to check: " << t << Endl;
    }

    out << "id: " << value.Id << Endl;
    out << "body: " << NUtils::Bin2hex(*value.Body) << Endl;
}

template <>
void Out<TGammaKeeperSettings::TKeyspaces>(IOutputStream& out, const TGammaKeeperSettings::TKeyspaces& value) {
    for (const auto& [key, value] : value) {
        out << key << "->" << value << Endl;
    }
}

template <>
void Out<TGammaKeeperSettings::TTimeBound>(IOutputStream& out, const TGammaKeeperSettings::TTimeBound& value) {
    for (const auto& [key, value] : value) {
        out << key << "->" << value.ToString() << Endl;
    }
}

template <>
void Out<TGammaKeeperSettings::TSignedTypes>(IOutputStream& out, const TGammaKeeperSettings::TSignedTypes& value) {
    for (const auto& value : value) {
        out << value << Endl;
    }
}
