#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/grants/consumer.h>
#include <passport/infra/daemons/blackbox/src/grants/ip_grants.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport::NUtils;

class TTestIpAclMap: public TIpAclMap<TConsumer> {
public:
    using TIpRangeMap = std::multimap<TIpAddr, TAclEntry>;
    TIpRangeMap& GetMap() {
        return RangeMap;
    }
};

Y_UNIT_TEST_SUITE(PasspBbIpGrants) {
    class TGrantsUnitTest: public TIpGrants {
    public:
        TGrantsUnitTest(const TString& path, TDuration timeout)
            : TIpGrants(path, timeout)
        {
        }

        std::shared_ptr<TIpAclMap<TConsumer>> GetAclMap() {
            return Acl();
        }
    };

    const char* const TST_IP[] = {
        "::2:abcd:cd00",
        "::2:abcd:cdff",
        "TEST v6",
        "12.45.67.0",
        "12.45.67.79",
        "TEST BIG",

        "12.45.67.80",
        "12.45.67.81",
        "TEST1",
        "12.45.67.82",
        "12.45.67.83",
        "TEST5",
        "12.45.67.84",
        "12.45.67.87",
        "TEST2",
        "12.45.67.88",
        "12.45.67.89",
        "TEST5",
        "12.45.67.90",
        "12.45.67.92",
        "TEST2",
        "12.45.67.93",
        "12.45.67.94",
        "TEST5",
        "12.45.67.95",
        "12.45.67.95",
        "TEST1",

        "12.45.67.96",
        "12.45.67.98",
        "TEST BIG",

        "12.45.67.99",
        "12.45.67.103",
        "TEST5",
        "12.45.67.104",
        "12.45.67.104",
        "TEST1",
        "12.45.67.105",
        "12.45.67.107",
        "TEST4",
        "12.45.67.108",
        "12.45.67.110",
        "TEST3",
        "12.45.67.111",
        "12.45.67.111",
        "TEST4",
        "12.45.67.112",
        "12.45.67.115",
        "TEST4",
        "12.45.67.116",
        "12.45.67.127",
        "TEST3",
        "12.45.67.128",
        "12.45.67.149",
        "TEST BIG",
        "12.45.67.150",
        "12.45.67.150",
        "TEST2",
        "12.45.67.151",
        "12.45.67.154",
        "TEST3",
        "12.45.67.155",
        "12.45.67.160",
        "TEST4",

        "12.45.67.161",
        "12.45.67.194",
        "TEST BIG",

        "12.45.67.195",
        "12.45.67.198",
        "TEST4",
        "12.45.67.199",
        "12.45.67.201",
        "TEST3",
        "12.45.67.202",
        "12.45.67.203",
        "TEST2",
        "12.45.67.204",
        "12.45.67.206",
        "TEST1",
        "12.45.67.207",
        "12.45.67.209",
        "TEST2",
        "12.45.67.210",
        "12.45.67.212",
        "TEST3",
        "12.45.67.213",
        "12.45.67.215",
        "TEST4",

        "12.45.67.216",
        "12.45.67.255",
        "TEST BIG",
    };

    const std::map<TString, TString> peerMap = {
        {"12.45.67.101", "TEST2"},
        {"12.45.67.115", "TEST v6"},
        {"12.45.67.152", "TEST1"},
        {"77.88.0.197", "mauzer"},
        {"77.88.0.198", "mauzer"},
    };

    Y_UNIT_TEST(aclGrants) {
        TGrantsUnitTest grants(TestsDir() + "data/grants.conf", TDuration::Seconds(120));

        const int testCount = sizeof(TST_IP) / sizeof(char*) / 3;
        std::shared_ptr<TIpAclMap<TConsumer>> aclmap = grants.GetAclMap();
        TTestIpAclMap* testmap = (TTestIpAclMap*)aclmap.get();
        auto p = testmap->GetMap().begin();

        // check map contents
        for (int i = 0; i < testCount; ++i) {
            UNIT_ASSERT_VALUES_EQUAL(TST_IP[3 * i], TString(p->first));
            UNIT_ASSERT(p->second.RangeStart);
            UNIT_ASSERT_VALUES_EQUAL(TST_IP[3 * i + 2], p->second.Value->GetName());
            ++p;
            UNIT_ASSERT_VALUES_EQUAL(p->first.ToString(), TST_IP[3 * i + 1]);
            UNIT_ASSERT(!p->second.RangeStart);
            UNIT_ASSERT_VALUES_EQUAL(TST_IP[3 * i + 2], p->second.Value->GetName());
            ++p;
        }

        for (const auto& pair : peerMap) {
            TIpAddr addr;
            addr.Parse(pair.first);
            auto it = aclmap->SingleIpMap.find(addr);
            UNIT_ASSERT(it != aclmap->SingleIpMap.end());
            UNIT_ASSERT_VALUES_EQUAL(pair.second, it->second->GetName());
        }

        // check searching for peers
        TIpAddr addr;
        addr.Parse("8.8.8.8");
        UNIT_ASSERT(!aclmap->Find(addr));

        addr.Parse("127.0.0.1");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("IlyaL_WinXP; zeze1; mda-dev;passportdev1", aclmap->Find(addr)->GetName());

        addr.Parse("77.88.54.4");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("coinbox; validator-dev; passportdev-iN.yandex.net", aclmap->Find(addr)->GetName());

        addr.Parse("12.45.67.111");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST4", aclmap->Find(addr)->GetName());

        addr.Parse("12.45.67.97");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST BIG", aclmap->Find(addr)->GetName());

        addr.Parse("::ffff:12.45.67.115");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST v6", aclmap->Find(addr)->GetName());

        addr.Parse("::2:abcd:cd00");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST v6", aclmap->Find(addr)->GetName());

        addr.Parse("::2:abcd:ce00");
        UNIT_ASSERT(!aclmap->Find(addr));

        // trypo nets
        addr.Parse("2a02:6b8:c00:123::1");
        UNIT_ASSERT(!aclmap->Find(addr));

        addr.Parse("2a02:6b8:c00:123:0:ed0c:11:1");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST5", aclmap->Find(addr)->GetName());

        addr.Parse("2a02:6b8:c55:ffff:0:ed0c:11:1");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST5", aclmap->Find(addr)->GetName());

        addr.Parse("2a02:6b8:d00:123:0:ed0c:11:1");
        UNIT_ASSERT(!aclmap->Find(addr));

        addr.Parse("2:abcd:cdef:0:0:ed0c::1");
        UNIT_ASSERT(!aclmap->Find(addr));

        addr.Parse("2:abcd:cdef:0:123:ed0c::1");
        UNIT_ASSERT(!aclmap->Find(addr));

        addr.Parse("2620:10f:d000::1111");
        UNIT_ASSERT(!aclmap->Find(addr));

        addr.Parse("2620:10f:1234:56:ac0:ffee::111");
        UNIT_ASSERT(!aclmap->Find(addr));

        addr.Parse("2620:10f:1234:56:0c0:ffee::111");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST BIG", aclmap->Find(addr)->GetName());

        addr.Parse("2620:10f:0:0:0c0:ffee:0:111");
        UNIT_ASSERT(aclmap->Find(addr));
        UNIT_ASSERT_VALUES_EQUAL("TEST BIG", aclmap->Find(addr)->GetName());

        addr.Parse("2620:11f:1234:56:0c0:ffee::111");
        UNIT_ASSERT(!aclmap->Find(addr));
    }
}
