#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/account_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/output/account_chunk.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport;

Y_UNIT_TEST_SUITE(TPasspBbHelpersAccountHelperTest) {
    TTestDbHolder& Db() {
        return TTestDbHolder::GetSingleton();
    }

    static TDbProfile::TAliases AliasesDefault() {
        return {
            {TAlias::PORTAL_LOGIN, TDbValue()},
            {TAlias::PDD_MASTER_LOGIN, TDbValue()},
            {TAlias::ALT_DOMAIN_LOGIN, TDbValue()},
            {TAlias::SOCIAL_LOGIN, TDbValue()},
            {TAlias::LITE_LOGIN, TDbValue()},
            {TAlias::PHONY_LOGIN, TDbValue()},
            {TAlias::MAILISH_LOGIN, TDbValue()},
            {TAlias::UBER_ID, TDbValue()},
            {TAlias::YAMBOT, TDbValue()},
            {TAlias::COLONKISH, TDbValue()},
            {TAlias::KINOPOISK_ID, TDbValue()},
            {TAlias::NEOPHONISH, TDbValue()},
            {TAlias::SCHOLAR, TDbValue()},
        };
    }

    static TDbProfile::TAliases AliasesAll() {
        return {
            {TAlias::PORTAL_LOGIN, TDbValue()},
            {TAlias::MAIL_LOGIN, TDbValue()},
            {TAlias::NAROD_MAIL_LOGIN, TDbValue()},
            {TAlias::LITE_LOGIN, TDbValue()},
            {TAlias::SOCIAL_LOGIN, TDbValue()},
            {TAlias::PDD_MASTER_LOGIN, TDbValue()},
            {TAlias::ALT_DOMAIN_LOGIN, TDbValue()},
            {TAlias::PHONY_LOGIN, TDbValue()},
            {TAlias::PHONE_NUMBER, TDbValue()},
            {TAlias::MAILISH_LOGIN, TDbValue()},
            {TAlias::YANDEXOID_LOGIN, TDbValue()},
            {TAlias::KINOPOISK_ID, TDbValue()},
            {TAlias::UBER_ID, TDbValue()},
            {TAlias::YAMBOT, TDbValue()},
            {TAlias::COLONKISH, TDbValue()},
            {TAlias::PUBLICID, TDbValue()},
            {TAlias::NEOPHONISH, TDbValue()},
            {TAlias::KIDDISH, TDbValue()},
            {TAlias::SCHOLAR, TDbValue()},
            {TAlias::FEDERAL, TDbValue()},
            {TAlias::BANK_PHONE_NUMBER, TDbValue()},
        };
    }

    static TDbProfile::TAttrs AttributesDefault() {
        return {
            {TAttr::ACCOUNT_USER_DEFINED_LOGIN, TDbValue()},
            {TAttr::ACCOUNT_NORMALIZED_LOGIN, TDbValue()},
            {TAttr::ACCOUNT_HAVE_PASSWORD, TDbValue()},
            {TAttr::ACCOUNT_HAVE_HINT, TDbValue()},
            {TAttr::PASSWORD_ENCRYPTED, TDbValue()},
            {TAttr::ACCOUNT_TOTP_SECRET, TDbValue()},
            {TAttr::HINT_QUESTION_SERIALIZED, TDbValue()},
            {TAttr::HINT_ANSWER_ENCRYPTED, TDbValue()},
            {TAttr::ACCOUNT_ENABLE_SEARCH_BY_PHONE_ALIAS, TDbValue()},
        };
    }

    void CheckAccount(const std::unique_ptr<TAccountChunk>& c,
                      const TAccountChunk::TAliasesType& aliases,
                      bool haveHint,
                      bool havePassword,
                      const TString& login) {
        UNIT_ASSERT_VALUES_EQUAL(c->Aliases, aliases);
        UNIT_ASSERT_VALUES_EQUAL(c->HaveHint, haveHint);
        UNIT_ASSERT_VALUES_EQUAL(c->HavePassword, havePassword);
        UNIT_ASSERT_VALUES_EQUAL(c->Login, login);
    }

    Y_UNIT_TEST(AccountHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);
        auto& attrs = const_cast<TDbProfile::TAttrs&>(fetcher->DefaultProfile().Attrs());
        auto& aliases = const_cast<TDbProfile::TAliases&>(fetcher->DefaultProfile().Aliases());

        // no aliases arg
        TAccountHelper h1(*conv, req);

        UNIT_ASSERT_VALUES_EQUAL(aliases, AliasesDefault());
        UNIT_ASSERT_VALUES_EQUAL(attrs, AttributesDefault());

        UNIT_ASSERT(!h1.Result(nullptr));

        // bad alias number, unsupported alias type
        req.Args["aliases"] = "-";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TAccountHelper(*conv, req), TBlackboxError, "invalid aliases value");

        req.Args["aliases"] = "ab";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TAccountHelper(*conv, req), TBlackboxError, "invalid aliases value");

        req.Args["aliases"] = "-1";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TAccountHelper(*conv, req), TBlackboxError, "invalid aliases value");

        req.Args["aliases"] = "0";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TAccountHelper(*conv, req), TBlackboxError, "Unsupported alias type: '0'");

        req.Args["aliases"] = "1,14,18";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TAccountHelper(*conv, req), TBlackboxError, "Unsupported alias type: '14'");

        req.Args["aliases"] = "26";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TAccountHelper(*conv, req), TBlackboxError, "Unsupported alias type: '26'");

        // alias get_social and get_phonenumber
        req.Args["aliases"] = "getsocial";
        TAccountHelper h2(*conv, req);

        UNIT_ASSERT_VALUES_EQUAL(aliases, AliasesDefault());
        UNIT_ASSERT_VALUES_EQUAL(attrs, AttributesDefault());

        UNIT_ASSERT(!h2.Result(nullptr));

        req.Args["aliases"] = "getphonenumber";
        TAccountHelper h3(*conv, req);

        UNIT_ASSERT_VALUES_EQUAL(aliases, TDbProfile::TAliases({
                                              {TAlias::PORTAL_LOGIN, TDbValue()},
                                              {TAlias::PDD_MASTER_LOGIN, TDbValue()},
                                              {TAlias::ALT_DOMAIN_LOGIN, TDbValue()},
                                              {TAlias::SOCIAL_LOGIN, TDbValue()},
                                              {TAlias::LITE_LOGIN, TDbValue()},
                                              {TAlias::PHONY_LOGIN, TDbValue()},
                                              {TAlias::PHONE_NUMBER, TDbValue()},
                                              {TAlias::MAILISH_LOGIN, TDbValue()},
                                              {TAlias::UBER_ID, TDbValue()},
                                              {TAlias::YAMBOT, TDbValue()},
                                              {TAlias::COLONKISH, TDbValue()},
                                              {TAlias::KINOPOISK_ID, TDbValue()},
                                              {TAlias::NEOPHONISH, TDbValue()},
                                              {TAlias::SCHOLAR, TDbValue()},
                                          }));

        UNIT_ASSERT_VALUES_EQUAL(attrs, AttributesDefault());

        UNIT_ASSERT(!h3.Result(nullptr));

        // custom numbers
        req.Args["aliases"] = "13,13 2";
        TAccountHelper h4(*conv, req);

        UNIT_ASSERT_VALUES_EQUAL(aliases, TDbProfile::TAliases({
                                              {TAlias::PORTAL_LOGIN, TDbValue()},
                                              {TAlias::MAIL_LOGIN, TDbValue()},
                                              {TAlias::LITE_LOGIN, TDbValue()},
                                              {TAlias::SOCIAL_LOGIN, TDbValue()},
                                              {TAlias::PDD_MASTER_LOGIN, TDbValue()},
                                              {TAlias::ALT_DOMAIN_LOGIN, TDbValue()},
                                              {TAlias::PHONY_LOGIN, TDbValue()},
                                              {TAlias::PHONE_NUMBER, TDbValue()},
                                              {TAlias::MAILISH_LOGIN, TDbValue()},
                                              {TAlias::YANDEXOID_LOGIN, TDbValue()},
                                              {TAlias::KINOPOISK_ID, TDbValue()},
                                              {TAlias::UBER_ID, TDbValue()},
                                              {TAlias::YAMBOT, TDbValue()},
                                              {TAlias::COLONKISH, TDbValue()},
                                              {TAlias::NEOPHONISH, TDbValue()},
                                              {TAlias::SCHOLAR, TDbValue()},
                                          }));

        UNIT_ASSERT_VALUES_EQUAL(attrs, AttributesDefault());

        UNIT_ASSERT(!h4.Result(nullptr));

        // all_with_hidden, all
        req.Args["aliases"] = "all";
        TAccountHelper h5(*conv, req);

        UNIT_ASSERT_VALUES_EQUAL(aliases, AliasesAll());
        UNIT_ASSERT_VALUES_EQUAL(attrs, AttributesDefault());

        UNIT_ASSERT(!h5.Result(nullptr));

        req.Args["aliases"] = "all_with_hidden";
        TAccountHelper h6(*conv, req);

        UNIT_ASSERT_VALUES_EQUAL(aliases, AliasesAll());
        UNIT_ASSERT_VALUES_EQUAL(attrs, AttributesDefault());

        UNIT_ASSERT(!h6.Result(nullptr));

        TTestDbProfile p(fetcher->DefaultProfile());

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), false, false, "");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({}), false, false, "");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), false, false, "");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({}), false, false, "");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({}), false, false, "");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({}), false, false, "");

        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("user-login");
        p.Attrs_[TAttr::ACCOUNT_USER_DEFINED_LOGIN] = TDbValue("User.Login");
        p.Aliases_[TAlias::KINOPOISK_ID] = TDbValue("9900389340898");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), false, false, "9900389340898");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({}), false, false, "9900389340898");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), false, false, "9900389340898");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({}), false, false, "9900389340898");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"15", "9900389340898"}}), false, false, "9900389340898");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"15", "9900389340898"}}), false, false, "9900389340898");

        p.Aliases_[TAlias::UBER_ID] = TDbValue("84enwhg5hk");
        p.Aliases_[TAlias::MAIL_LOGIN] = TDbValue("smtp@ya.ru");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), false, false, "84enwhg5hk");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({}), false, false, "84enwhg5hk");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), false, false, "84enwhg5hk");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}}), false, false, "84enwhg5hk");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     false, false, "84enwhg5hk");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     false, false, "84enwhg5hk");

        p.Aliases_[TAlias::MAILISH_LOGIN] = TDbValue("yndx-test-mailish-x6bo50@test.ru");
        p.Aliases_[TAlias::SOCIAL_LOGIN] = TDbValue("uid-noobn3yl");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), false, false, "yndx-test-mailish-x6bo50@test.ru");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), false, false, "yndx-test-mailish-x6bo50@test.ru");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), false, false, "yndx-test-mailish-x6bo50@test.ru");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}}), false, false, "yndx-test-mailish-x6bo50@test.ru");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"6", "uid-noobn3yl"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}), false, false, "yndx-test-mailish-x6bo50@test.ru");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"6", "uid-noobn3yl"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}), false, false, "yndx-test-mailish-x6bo50@test.ru");

        p.Aliases_[TAlias::LITE_LOGIN] = TDbValue("masha@gmail.com");
        p.Attrs_[TAttr::ACCOUNT_HAVE_HINT] = TDbValue("1");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, false, "masha@gmail.com");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, false, "masha@gmail.com");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), true, false, "masha@gmail.com");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}}), true, false, "masha@gmail.com");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}), true, false, "masha@gmail.com");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}), true, false, "masha@gmail.com");

        p.Aliases_[TAlias::ALT_DOMAIN_LOGIN] = TDbValue("ahmed@galatasaray.net");
        p.Aliases_[TAlias::YANDEXOID_LOGIN] = TDbValue("art");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, false, "ahmed@galatasaray.net");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, false, "ahmed@galatasaray.net");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), true, false, "ahmed@galatasaray.net");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, false, "ahmed@galatasaray.net");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"9", "ahmed@galatasaray.net"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     true, false, "ahmed@galatasaray.net");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"9", "ahmed@galatasaray.net"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     true, false, "ahmed@galatasaray.net");

        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue("user-login");
        p.Attrs_[TAttr::ACCOUNT_HAVE_PASSWORD] = TDbValue("2");
        p.Aliases_[TAlias::PHONE_NUMBER] = TDbValue("89169161616");
        p.Aliases_[TAlias::NAROD_MAIL_LOGIN] = TDbValue("mail@narod.ru");
        p.Aliases_[TAlias::PUBLICID] = TDbValue("SuperSuperMacho");
        p.AliasesOldPublic_.insert("Macho");
        p.AliasesOldPublic_.insert("RealMacho");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "User.Login");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "User.Login");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), true, true, "User.Login");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "User.Login");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"1", "user-login"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"9", "ahmed@galatasaray.net"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     true, true, "User.Login");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"1", "user-login"}, {"11", "89169161616"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"9", "ahmed@galatasaray.net"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}}),
                     true, true, "User.Login");

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "test.ru",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Тестер и Сыновья LTD",
            },
        });
        p.Aliases_[TAlias::PDD_MASTER_LOGIN] = TDbValue("tester@test.ru");
        p.AliasesPdd_.insert("son1@test.ru");
        p.AliasesPdd_.insert("son2@test.ru");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "User.Login");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "User.Login");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), true, true, "User.Login");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "User.Login");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"1", "user-login"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"9", "ahmed@galatasaray.net"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     true, true, "User.Login");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"1", "user-login"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"9", "ahmed@galatasaray.net"}, {"11", "89169161616"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}}),
                     true, true, "User.Login");

        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue();

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "ahmed@galatasaray.net");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "ahmed@galatasaray.net");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), true, true, "ahmed@galatasaray.net");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "ahmed@galatasaray.net");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"9", "ahmed@galatasaray.net"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     true, true, "ahmed@galatasaray.net");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"9", "ahmed@galatasaray.net"}, {"11", "89169161616"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}}),
                     true, true, "ahmed@galatasaray.net");

        p.Aliases_[TAlias::ALT_DOMAIN_LOGIN] = TDbValue();

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "masha@gmail.com");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "masha@gmail.com");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), true, true, "masha@gmail.com");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "masha@gmail.com");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}), true, true, "masha@gmail.com");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"5", "masha@gmail.com"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"11", "89169161616"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}}), true, true, "masha@gmail.com");

        p.Aliases_[TAlias::LITE_LOGIN] = TDbValue();

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "tester@test.ru");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "tester@test.ru");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({}), true, true, "tester@test.ru");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "tester@test.ru");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}}),
                     true, true, "tester@test.ru");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"11", "89169161616"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}}),
                     true, true, "tester@test.ru");

        p.Attrs_[TAttr::ACCOUNT_ENABLE_SEARCH_BY_PHONE_ALIAS] = TDbValue("1");
        p.Aliases_[TAlias::COLONKISH] = TDbValue("kolonkish-cev3bzeq");
        p.Aliases_[TAlias::YAMBOT] = TDbValue("yambot-3peeyx3m");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "tester@test.ru");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "tester@test.ru");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}}), true, true, "tester@test.ru");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "tester@test.ru");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"17", "yambot-3peeyx3m"}, {"18", "kolonkish-cev3bzeq"}}), true, true, "tester@test.ru");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"17", "yambot-3peeyx3m"}, {"18", "kolonkish-cev3bzeq"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}}), true, true, "tester@test.ru");

        p.Aliases_[TAlias::NEOPHONISH] = TDbValue("nphne-neo");
        p.Aliases_[TAlias::KIDDISH] = TDbValue("kid-alias");
        p.Aliases_[TAlias::SCHOLAR] = TDbValue("Школьник1");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "tester@test.ru");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "tester@test.ru");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}}), true, true, "tester@test.ru");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "tester@test.ru");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"17", "yambot-3peeyx3m"}, {"18", "kolonkish-cev3bzeq"}, {"21", "nphne-neo"}, {"22", "kid-alias"}, {"23", "Школьник1"}}), true, true, "tester@test.ru");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"17", "yambot-3peeyx3m"}, {"18", "kolonkish-cev3bzeq"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}, {"21", "nphne-neo"}, {"22", "kid-alias"}, {"23", "Школьник1"}}), true, true, "tester@test.ru");

        p.Aliases_[TAlias::BANK_PHONE_NUMBER] = TDbValue("79012345678");

        CheckAccount(h1.Result(&p), TAccountChunk::TAliasesType({}), true, true, "tester@test.ru");
        CheckAccount(h2.Result(&p), TAccountChunk::TAliasesType({{"6", "uid-noobn3yl"}}), true, true, "tester@test.ru");
        CheckAccount(h3.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}}), true, true, "tester@test.ru");
        CheckAccount(h4.Result(&p), TAccountChunk::TAliasesType({{"2", "smtp@ya.ru"}, {"13", "art"}}), true, true, "tester@test.ru");
        CheckAccount(h5.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"17", "yambot-3peeyx3m"}, {"18", "kolonkish-cev3bzeq"}, {"21", "nphne-neo"}, {"22", "kid-alias"}, {"23", "Школьник1"}}), true, true, "tester@test.ru");
        CheckAccount(h6.Result(&p), TAccountChunk::TAliasesType({{"11", "89169161616"}, {"2", "smtp@ya.ru"}, {"3", "mail@narod.ru"}, {"6", "uid-noobn3yl"}, {"7", "tester@test.ru"}, {"8", "son1@test.ru"}, {"8", "son2@test.ru"}, {"12", "yndx-test-mailish-x6bo50@test.ru"}, {"13", "art"}, {"15", "9900389340898"}, {"16", "84enwhg5hk"}, {"17", "yambot-3peeyx3m"}, {"18", "kolonkish-cev3bzeq"}, {"19", "SuperSuperMacho"}, {"20", "Macho"}, {"20", "RealMacho"}, {"21", "nphne-neo"}, {"22", "kid-alias"}, {"23", "Школьник1"}, {"25", "79012345678"}}), true, true, "tester@test.ru");
    }

    Y_UNIT_TEST(CheckGrants) {
        NTest::TRequest req;
        TConsumer c;
        TGrantsChecker checker(req, c, false);

        auto check = [&](const std::set<TString>& expected) {
            TAccountHelper::CheckGrants(checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };

        check({});

        req.Args["aliases"] = "123";
        check({});

        req.Args["aliases"] = "all_with_hidden";
        check({"no grants to use aliases=all_with_hidden"});

        c.SetAllow(TBlackboxFlags::GetHiddenAliases, true);
        check({});
    }
};
