#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/billing_features_helper.h>
#include <passport/infra/daemons/blackbox/src/output/billing_features_chunk.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspUtBillingFeaturesHelper) {
    static TTestDbHolder& Db() {
        return TTestDbHolder::GetSingleton();
    }

    Y_UNIT_TEST(checkGrants) {
        NTest::TRequest req;
        TConsumer c;
        TGrantsChecker checker(req, c, false);

        auto check = [&](const std::set<TString>& expected) {
            TBillingFeaturesHelper::CheckGrants(checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };

        check({});

        req.Args["get_billing_features"] = "";
        check({"no grants for arg 'get_billing_features'"});

        c.SetAllow(TBlackboxFlags::GetBillingFeatures, true);
        check({});
    }

    Y_UNIT_TEST(billingFeaturesHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TAttrs&>(fetcher->DefaultProfile().Attrs());

        // no get_billing_features arg
        TBillingFeaturesHelper h1(*fetcher, req);

        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT(!h1.Result(&fetcher->DefaultProfile()));

        // want billing features but no grant
        req.Args["get_billing_features"] = "yes of course";

        // bad billing features arg, has grant
        UNIT_ASSERT_EXCEPTION_CONTAINS(TBillingFeaturesHelper(*fetcher, req),
                                       TBlackboxError,
                                       "Unsupported get_billing_features value: 'yes of course'");

        UNIT_ASSERT(attrs.empty());

        // good get_billing_features arg
        req.Args["get_billing_features"] = "all";
        TBillingFeaturesHelper h2(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TAttrs({{TAttr::ACCOUNT_BILLING_FEATURES, TDbValue()}}));

        // broken protobuf
        attrs.begin()->second = TDbValue(NUtils::Base64ToBin("abc"));

        UNIT_ASSERT(!h2.Result(nullptr));
        UNIT_ASSERT(!h2.Result(&fetcher->DefaultProfile()));

        attrs.begin()->second = TDbValue(NUtils::Base64ToBin("ChwKDE11c2ljUHJlbWl1bRIMCAEQzsLxBRgAINUBCh8KDFRlc3RfZmVhdHVyZRIPCAAQZCoJQmlnIGJyYW5k"));

        UNIT_ASSERT(!h2.Result(nullptr));
        std::unique_ptr<TBillingFeaturesChunk> c = h2.Result(&fetcher->DefaultProfile());
        UNIT_ASSERT(c);
        UNIT_ASSERT_VALUES_EQUAL(2, c->Features.size());

        const auto& featureAttrs = c->Features["MusicPremium"];
        UNIT_ASSERT(featureAttrs.InTrial);
        UNIT_ASSERT_VALUES_EQUAL(true, *featureAttrs.InTrial);
        UNIT_ASSERT(featureAttrs.TrialDuration);
        UNIT_ASSERT_VALUES_EQUAL(12345678, *featureAttrs.TrialDuration);
        UNIT_ASSERT(featureAttrs.PaidTrial);
        UNIT_ASSERT_VALUES_EQUAL(false, *featureAttrs.PaidTrial);
        UNIT_ASSERT(featureAttrs.RegionId);
        UNIT_ASSERT_VALUES_EQUAL(213, *featureAttrs.RegionId);
        UNIT_ASSERT(!featureAttrs.Brand);

        const auto& featureAttrs2 = c->Features["Test_feature"];
        UNIT_ASSERT(featureAttrs2.InTrial);
        UNIT_ASSERT_VALUES_EQUAL(false, *featureAttrs2.InTrial);
        UNIT_ASSERT(featureAttrs2.TrialDuration);
        UNIT_ASSERT_VALUES_EQUAL(100, *featureAttrs2.TrialDuration);
        UNIT_ASSERT(!featureAttrs2.PaidTrial);
        UNIT_ASSERT(!featureAttrs2.RegionId);
        UNIT_ASSERT_VALUES_EQUAL("Big brand", *featureAttrs2.Brand);
    }
}
