#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/display_name_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/output/display_name_chunk.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport;

// test helpers
void CheckSocialTarget(const TString& profileId, const TString& target) {
    if (profileId.empty()) {
        UNIT_ASSERT(target.empty());
        return;
    }
    TString id = NUtils::CreateStr(".", profileId, ".");

    TString::size_type dot = target.find('.');
    UNIT_ASSERT(dot != TString::npos);
    UNIT_ASSERT(dot < target.size() - 1);

    dot = target.find('.', dot + 1);
    UNIT_ASSERT(dot != TString::npos);
    UNIT_ASSERT(dot < target.size() - 1);

    UNIT_ASSERT_VALUES_EQUAL(id, target.substr(dot, id.size()));
    UNIT_ASSERT(target.find('.', dot + id.size() + 1) == TString::npos);
}

void CheckDisplayNameImpl(const std::unique_ptr<TDisplayNameChunk>& c,
                          const TString& regname,
                          const TString& displayName,
                          const TString& avatar,
                          bool avatarEmpty,
                          bool social,
                          bool publicName) {
    UNIT_ASSERT_VALUES_EQUAL(c->Regname, regname);
    UNIT_ASSERT_VALUES_EQUAL(c->DisplayName, displayName);
    UNIT_ASSERT_VALUES_EQUAL(c->Avatar, avatar);
    UNIT_ASSERT_VALUES_EQUAL(c->IsAvatarEmpty, avatarEmpty);

    // for social accounts all should be non-empty, and vice versa
    UNIT_ASSERT_VALUES_EQUAL(c->Social, social);
    UNIT_ASSERT_VALUES_EQUAL(!c->ProfileId.empty(), social);
    UNIT_ASSERT_VALUES_EQUAL(!c->Provider.empty(), social);
    UNIT_ASSERT_VALUES_EQUAL(!c->Target.empty(), social);

    UNIT_ASSERT_VALUES_EQUAL(c->IsDisplayNameEmpty.has_value(), publicName);
    UNIT_ASSERT_VALUES_EQUAL(c->PublicName.has_value(), publicName);
}

void CheckDisplayName(const std::unique_ptr<TDisplayNameChunk>& c,
                      const TString& regname,
                      const TString& displayName,
                      const TString& avatar,
                      bool avatarEmpty) {
    CheckDisplayNameImpl(c, regname, displayName, avatar, avatarEmpty, false, false);
}

void CheckDisplayNameSocial(const std::unique_ptr<TDisplayNameChunk>& c,
                            const TString& regname,
                            const TString& displayName,
                            const TString& avatar,
                            bool avatarEmpty,
                            const TString& profileId,
                            const TString& provider) {
    CheckDisplayNameImpl(c, regname, displayName, avatar, avatarEmpty, true, false);

    UNIT_ASSERT_VALUES_EQUAL(c->ProfileId, profileId);
    UNIT_ASSERT_VALUES_EQUAL(c->Provider, provider);
    CheckSocialTarget(profileId, c->Target);
}

void CheckDisplayNamePublic(const std::unique_ptr<TDisplayNameChunk>& c,
                            const TString& regname,
                            const TString& displayName,
                            const TString& avatar,
                            bool avatarEmpty,
                            bool displayNameEmpty,
                            const TString& publicName) {
    CheckDisplayNameImpl(c, regname, displayName, avatar, avatarEmpty, false, true);

    UNIT_ASSERT_VALUES_EQUAL(*c->IsDisplayNameEmpty, displayNameEmpty);
    UNIT_ASSERT_VALUES_EQUAL(c->PublicName->Name, publicName);
}

void CheckDisplayNamePublicSocial(const std::unique_ptr<TDisplayNameChunk>& c,
                                  const TString& regname,
                                  const TString& displayName,
                                  const TString& avatar,
                                  bool avatarEmpty,
                                  bool displayNameEmpty,
                                  const TString& publicName,
                                  const TString& profileId,
                                  const TString& provider) {
    CheckDisplayNameImpl(c, regname, displayName, avatar, avatarEmpty, true, true);

    UNIT_ASSERT_VALUES_EQUAL(c->ProfileId, profileId);
    UNIT_ASSERT_VALUES_EQUAL(c->Provider, provider);
    CheckSocialTarget(profileId, c->Target);

    UNIT_ASSERT_VALUES_EQUAL(*c->IsDisplayNameEmpty, displayNameEmpty);
    UNIT_ASSERT_VALUES_EQUAL(c->PublicName->Name, publicName);
}

Y_UNIT_TEST_SUITE(TPasspBbHelpersDisplayNameHelper) {
    static TTestDbHolder& Db() {
        return TTestDbHolder::GetSingleton();
    }

    static TDbProfile::TAliases AliasesDefault() {
        return {
            {TAlias::PORTAL_LOGIN, TDbValue()},
            {TAlias::PDD_MASTER_LOGIN, TDbValue()},
            {TAlias::ALT_DOMAIN_LOGIN, TDbValue()},
            {TAlias::SOCIAL_LOGIN, TDbValue()},
            {TAlias::LITE_LOGIN, TDbValue()},
            {TAlias::PHONY_LOGIN, TDbValue()},
            {TAlias::MAILISH_LOGIN, TDbValue()},
            {TAlias::UBER_ID, TDbValue()},
            {TAlias::YAMBOT, TDbValue()},
            {TAlias::COLONKISH, TDbValue()},
            {TAlias::KINOPOISK_ID, TDbValue()},
            {TAlias::NEOPHONISH, TDbValue()},
            {TAlias::SCHOLAR, TDbValue()},
        };
    }

    static TDbProfile::TAttrs AttrsSimple() {
        return {
            {TAttr::ACCOUNT_USER_DEFINED_LOGIN, TDbValue()},
            {TAttr::ACCOUNT_NORMALIZED_LOGIN, TDbValue()},
            {TAttr::ACCOUNT_DISPLAY_NAME, TDbValue()},
            {TAttr::AVATAR_DEFAULT, TDbValue()},
            {TAttr::PERSON_FIRSTNAME, TDbValue()},
            {TAttr::PERSON_LASTNAME, TDbValue()},
            {TAttr::ACCOUNT_IS_VERIFIED, TDbValue()},
        };
    }

    static TDbProfile::TAttrs AttrsPublicName() {
        return {
            {TAttr::ACCOUNT_REGISTRATION_DATETIME, TDbValue()},
            {TAttr::ACCOUNT_USER_DEFINED_LOGIN, TDbValue()},
            {TAttr::ACCOUNT_NORMALIZED_LOGIN, TDbValue()},
            {TAttr::ACCOUNT_DISPLAY_NAME, TDbValue()},
            {TAttr::AVATAR_DEFAULT, TDbValue()},
            {TAttr::PERSON_FIRSTNAME, TDbValue()},
            {TAttr::PERSON_LASTNAME, TDbValue()},
            {TAttr::PERSON_LANGUAGE, TDbValue()},
            {TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME, TDbValue()},
            {TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME, TDbValue()},
            {TAttr::ACCOUNT_PERSONAL_DATA_PUBLIC_ACCESS_ALLOWED, TDbValue()},
            {TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED, TDbValue()},
            {TAttr::ACCOUNT_IS_VERIFIED, TDbValue()},
        };
    }

    Y_UNIT_TEST(DisplayNameBuilder) {
        TTestDbProfile p;
        p.FillWithSampleData();
        TDisplayNameBuilder b(p.AddAttr(TAttr::PERSON_FIRSTNAME, "Васисуалий"),
                              p.AddAttr(TAttr::PERSON_LASTNAME, "Лоханкин"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(b.BuildDisplayName(nullptr, "", ""), yexception, "format is too small ''");

        UNIT_ASSERT_VALUES_EQUAL("", b.BuildDisplayName(nullptr, "t:", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("", b.BuildDisplayName(&p, "t:", "SuperSuperMacho"));

        UNIT_ASSERT_VALUES_EQUAL("customN@me", b.BuildDisplayName(nullptr, "t:customN@me", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("Got%", b.BuildDisplayName(nullptr, "t:Got%", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("Got%%", b.BuildDisplayName(nullptr, "t:Got%%", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("Got%%%", b.BuildDisplayName(nullptr, "t:Got%%%", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("Got%login", b.BuildDisplayName(nullptr, "t:Got%login", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("25%f5%%", b.BuildDisplayName(nullptr, "t:25%f5%%", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("a%FOO%b%BAR_", b.BuildDisplayName(nullptr, "t:a%FOO%b%BAR_", "SuperSuperMacho"));

        UNIT_ASSERT_VALUES_EQUAL("I'm Васисуалий + Лоханкин aka SuperSuperMacho", b.BuildDisplayName(&p, "t:I'm %firstname% + %lastname% aka %login%", "SuperSuperMacho"));

        // not pdd account, %pdd_username empty
        UNIT_ASSERT_VALUES_EQUAL("@рога-и-копыта.рф/рога-и-копыта.рф", b.BuildDisplayName(&p, "t:%pdd_username%@%pdd_domain%/%display_domain%", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("@рога-и-копыта.рф/рога-и-копыта.рф", b.BuildDisplayName(&p, "t:%pdd_username%@%pdd_domain%/%display_domain%", "SuperSuperMacho@display.ru"));

        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue(); // now it is pdd account
        UNIT_ASSERT_VALUES_EQUAL("@рога-и-копыта.рф/рога-и-копыта.рф", b.BuildDisplayName(&p, "t:%pdd_username%@%pdd_domain%/%display_domain%", "SuperSuperMacho"));
        UNIT_ASSERT_VALUES_EQUAL("SuperSuperMacho@рога-и-копыта.рф/рога-и-копыта.рф", b.BuildDisplayName(&p, "t:%pdd_username%@%pdd_domain%/%display_domain%", "SuperSuperMacho@display.ru"));

        TTestDbProfile p2;
        TDisplayNameBuilder b2(p2.AddAttr(TAttr::PERSON_FIRSTNAME), p2.AddAttr(TAttr::PERSON_COUNTRY));

        UNIT_ASSERT_VALUES_EQUAL("I'm  +  aka SuperSuperMacho@mail.ru", b2.BuildDisplayName(&p2, "t:I'm %firstname% + %lastname% aka %login%", "SuperSuperMacho@mail.ru"));
        UNIT_ASSERT_VALUES_EQUAL("@/", b2.BuildDisplayName(&p2, "t:%pdd_username%@%pdd_domain%/%display_domain%", "SuperSuperMacho@display.ru"));
    }

    Y_UNIT_TEST(DisplayNameStatic) {
        // displayNameEmpty
        UNIT_ASSERT(TDisplayNameHelper::DisplayNameEmpty(""));
        UNIT_ASSERT(TDisplayNameHelper::DisplayNameEmpty("!"));
        UNIT_ASSERT(TDisplayNameHelper::DisplayNameEmpty("wow"));
        UNIT_ASSERT(TDisplayNameHelper::DisplayNameEmpty("foo:bar"));
        UNIT_ASSERT(TDisplayNameHelper::DisplayNameEmpty("q:"));

        UNIT_ASSERT(!TDisplayNameHelper::DisplayNameEmpty("p:"));
        UNIT_ASSERT(!TDisplayNameHelper::DisplayNameEmpty("s:"));
        UNIT_ASSERT(!TDisplayNameHelper::DisplayNameEmpty("pna:"));
        UNIT_ASSERT(!TDisplayNameHelper::DisplayNameEmpty("t:"));

        // buildPublicName
        UNIT_ASSERT_VALUES_EQUAL("", TDisplayNameHelper::BuildPublicName("", "Smith", false));
        UNIT_ASSERT_VALUES_EQUAL("John", TDisplayNameHelper::BuildPublicName("John", "", false));
        UNIT_ASSERT_VALUES_EQUAL("John", TDisplayNameHelper::BuildPublicName("John", "   ", false));
        UNIT_ASSERT_VALUES_EQUAL("John S.", TDisplayNameHelper::BuildPublicName("John", "Smith", false));
        UNIT_ASSERT_VALUES_EQUAL("John S.", TDisplayNameHelper::BuildPublicName("John", "    Smith", false));
        UNIT_ASSERT_VALUES_EQUAL("John S.", TDisplayNameHelper::BuildPublicName("John", "Smith     ", false));
        UNIT_ASSERT_VALUES_EQUAL("John", TDisplayNameHelper::BuildPublicName("John", "S", false));
        UNIT_ASSERT_VALUES_EQUAL("John", TDisplayNameHelper::BuildPublicName("John", "  S  ", false));
        UNIT_ASSERT_VALUES_EQUAL("John 😂.", TDisplayNameHelper::BuildPublicName("John", "😂 Я 'Тестовый\" / &<тестер>", false));
        UNIT_ASSERT_VALUES_EQUAL("Smith", TDisplayNameHelper::BuildPublicName("", "  Smith", true));
        UNIT_ASSERT_VALUES_EQUAL("John", TDisplayNameHelper::BuildPublicName("John ", "", true));
        UNIT_ASSERT_VALUES_EQUAL("John Smith", TDisplayNameHelper::BuildPublicName(" John", "Smith ", true));

        UNIT_ASSERT_VALUES_EQUAL("", TDisplayNameHelper::BuildPublicName("", "Пупкин", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася", TDisplayNameHelper::BuildPublicName("Вася", "", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася", TDisplayNameHelper::BuildPublicName("  Вася", "", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася", TDisplayNameHelper::BuildPublicName("Вася  ", "", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася П.", TDisplayNameHelper::BuildPublicName("  Вася  ", "Пупкин", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася П.", TDisplayNameHelper::BuildPublicName("  Вася  ", "   Пупкин", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася П.", TDisplayNameHelper::BuildPublicName("Вася", "Пупкин  ", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася", TDisplayNameHelper::BuildPublicName("Вася", "П", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася", TDisplayNameHelper::BuildPublicName("Вася", "  П  ", false));
        UNIT_ASSERT_VALUES_EQUAL("Вася 😂.", TDisplayNameHelper::BuildPublicName("Вася", "😂 Я 'Тестовый\" / &<тестер>", false));
        UNIT_ASSERT_VALUES_EQUAL("Пупкин", TDisplayNameHelper::BuildPublicName("", " Пупкин", true));
        UNIT_ASSERT_VALUES_EQUAL("Вася", TDisplayNameHelper::BuildPublicName("  Вася  ", "", true));
        UNIT_ASSERT_VALUES_EQUAL("Вася Пупкин", TDisplayNameHelper::BuildPublicName("  Вася  ", " Пупкин   ", true));

        // buildIncognitoName
        UNIT_ASSERT_VALUES_EQUAL("Инкогнито 3055", TDisplayNameHelper::BuildIncognitoName("ru", "4086333055"));
        UNIT_ASSERT_VALUES_EQUAL("Incognito 7129", TDisplayNameHelper::BuildIncognitoName("en", "4092777129"));
        UNIT_ASSERT_VALUES_EQUAL("Incognito 501", TDisplayNameHelper::BuildIncognitoName("", "501"));

        // suggestPublicName
        UNIT_ASSERT_VALUES_EQUAL("", TDisplayNameHelper::SuggestPublicName("", " "));
        UNIT_ASSERT_VALUES_EQUAL("", TDisplayNameHelper::SuggestPublicName("John", ""));
        UNIT_ASSERT_VALUES_EQUAL("", TDisplayNameHelper::SuggestPublicName("John", " "));
        UNIT_ASSERT_VALUES_EQUAL("", TDisplayNameHelper::SuggestPublicName("John", "\t"));
        UNIT_ASSERT_VALUES_EQUAL("Smith", TDisplayNameHelper::SuggestPublicName("  ", "Smith "));
        UNIT_ASSERT_VALUES_EQUAL("John Smith", TDisplayNameHelper::SuggestPublicName("\t John   ", "Smith"));
    }

    Y_UNIT_TEST(DisplayNameHelperEmpty) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);
        auto& attrs = const_cast<TDbProfile::TAttrs&>(fetcher->DefaultProfile().Attrs());
        auto& aliases = const_cast<TDbProfile::TAliases&>(fetcher->DefaultProfile().Aliases());
        auto& phoneattrs = const_cast<TDbProfile::TExtendedEntities&>(fetcher->DefaultProfile().ExtendedPhoneAttrs());

        // no regname arg
        TDisplayNameHelper h1(*conv, req, 0, 0);

        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(aliases.empty());
        UNIT_ASSERT(phoneattrs.empty());
        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT(!h1.Result(&fetcher->DefaultProfile()));

        // bad regname arg
        req.Args["regname"] = "no";
        TDisplayNameHelper h2(*conv, req, 0, 0);

        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(aliases.empty());
        UNIT_ASSERT(phoneattrs.empty());
        UNIT_ASSERT(!h2.Result(nullptr));
        UNIT_ASSERT(!h2.Result(&fetcher->DefaultProfile()));

        // regname is off
        req.Args["regname"] = "off";
        TDisplayNameHelper h3(*conv, req, 0, 0);

        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(aliases.empty());
        UNIT_ASSERT(phoneattrs.empty());
        UNIT_ASSERT(!h3.Result(nullptr));
        UNIT_ASSERT(!h3.Result(&fetcher->DefaultProfile()));

        // regname on but empty profile
        req.Args["regname"] = "yes";
        TDisplayNameHelper hSimple(*conv, req, 0, 0);

        UNIT_ASSERT_VALUES_EQUAL(attrs, AttrsSimple());
        UNIT_ASSERT_VALUES_EQUAL(aliases, AliasesDefault());
        UNIT_ASSERT_VALUES_EQUAL(phoneattrs, TDbProfile::TExtendedEntities({{"",
                                                                             TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()},
                                                                                                 {TPhoneAttr::E164_NUMBER, TDbValue()},
                                                                                                 {TPhoneAttr::MASKED_FORMATTED_NUMBER, TDbValue()}})}}));
        UNIT_ASSERT(!hSimple.Result(nullptr));

        req.Args["get_public_name"] = "no";
        TDisplayNameHelper hSimple2(*conv, req, 0, 0);

        UNIT_ASSERT_VALUES_EQUAL(attrs, AttrsSimple());
        UNIT_ASSERT_VALUES_EQUAL(aliases, AliasesDefault());
        UNIT_ASSERT_VALUES_EQUAL(phoneattrs, TDbProfile::TExtendedEntities({{"",
                                                                             TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()},
                                                                                                 {TPhoneAttr::E164_NUMBER, TDbValue()},
                                                                                                 {TPhoneAttr::MASKED_FORMATTED_NUMBER, TDbValue()}})}}));

        UNIT_ASSERT(!hSimple2.Result(nullptr));

        // with public name
        req.Args["get_public_name"] = "1";
        req.Args["is_display_name_empty"] = "True";
        TDisplayNameHelper hPublic(*conv, req, 0, 0);

        UNIT_ASSERT_VALUES_EQUAL(attrs, AttrsPublicName());
        UNIT_ASSERT_VALUES_EQUAL(aliases, AliasesDefault());
        UNIT_ASSERT_VALUES_EQUAL(phoneattrs, TDbProfile::TExtendedEntities({{"",
                                                                             TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()},
                                                                                                 {TPhoneAttr::E164_NUMBER, TDbValue()},
                                                                                                 {TPhoneAttr::MASKED_FORMATTED_NUMBER, TDbValue()}})}}));
        UNIT_ASSERT(!hPublic.Result(nullptr));
    }

    Y_UNIT_TEST(DisplayNameHelperPhonish) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);

        // regname ok, no public name
        req.Args["get_public_name"] = "no";
        TDisplayNameHelper hSimple(*conv, req, 1500000000, 1600000000);

        req.Args["get_public_name"] = "yes";
        req.Args["is_display_name_empty"] = "True";
        TDisplayNameHelper hPublic(*conv, req, 1500000000, 1600000000);

        TTestDbProfile p(fetcher->DefaultProfile());

        // phonish alias
        p.Aliases_[TAlias::PHONY_LOGIN] = TDbValue("phne-eprst");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("phne-eprst");

        CheckDisplayName(hSimple.Result(&p), "phne-eprst", "phne-eprst", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "phne-eprst", "phne-eprst", "0/0-0", true, true, "phne-eprst");

        // has phonish alias + phone number
        p.ExtendedPhones_["333"] = TDbProfile::TAttrs({{"1", TDbValue("79252252525")},
                                                       {"103", TDbValue("+7925*****25")}});

        CheckDisplayName(hSimple.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true, false, "+7925*****25");

        // setting display name doesn't work for phonish
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:vk:Супер Мачо");

        CheckDisplayName(hSimple.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true, false, "+7925*****25");

        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("p:анонимус");

        CheckDisplayName(hSimple.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true, false, "+7925*****25");

        // setting fio doesn't work for phonish
        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue("Васисуалий");
        p.Attrs_[TAttr::PERSON_LASTNAME] = TDbValue("Лоханкин");

        CheckDisplayName(hSimple.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "phne-eprst", "+7925*****25", "0/0-0", true, false, "+7925*****25");

        // phonish with portal alias
        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue("superuser");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("superuser");

        CheckDisplayName(hSimple.Result(&p), "superuser", "анонимус", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "анонимус", "0/0-0", true, false, "анонимус");

        // all other tests should be the same as for portal user, here are just two of them to be sure
        // phonish with portal login and fio
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("broken");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "0/0-0", true, true, "Васисуалий Л.");

        // phonish with portal login + fio + avatar + show FIO
        p.Attrs_[TAttr::AVATAR_DEFAULT] = TDbValue("1/2-3");
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("100");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "1/2-3", false, true, "Васисуалий Лоханкин");

        // phonish with portal login + fio + avatar + no show FIO but registered after fullFioStartTime
        p.Attrs_[TAttr::ACCOUNT_REGISTRATION_DATETIME] = TDbValue("1600000000");
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("0");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "1/2-3", false, true, "Васисуалий Лоханкин");

        // phonish with portal login + fio + avatar + no show FIO registered before fullFioStartTime
        p.Attrs_[TAttr::ACCOUNT_REGISTRATION_DATETIME] = TDbValue("1500000000");
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("0");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "1/2-3", false, true, "Васисуалий Л.");
    }

    Y_UNIT_TEST(DisplayNameHelperNeophonish) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);

        // regname ok, no public name
        req.Args["get_public_name"] = "no";
        TDisplayNameHelper hSimple(*conv, req, 1500000000, 1600000000);

        req.Args["get_public_name"] = "yes";
        req.Args["is_display_name_empty"] = "True";
        TDisplayNameHelper hPublic(*conv, req, 1500000000, 1600000000);

        TTestDbProfile p(fetcher->DefaultProfile());

        // neophonish alias
        p.Aliases_[TAlias::NEOPHONISH] = TDbValue("nphne-abcdef");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("nphne-abcdef");

        // no FIO
        p.Uid_ = "4086330501";
        CheckDisplayName(hSimple.Result(&p), "nphne-abcdef", "nphne-abcdef", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "nphne-abcdef", "nphne-abcdef", "0/0-0", true, true, "Incognito 0501");

        p.Attrs_[TAttr::PERSON_LANGUAGE] = TDbValue("ru");
        CheckDisplayNamePublic(hPublic.Result(&p), "nphne-abcdef", "nphne-abcdef", "0/0-0", true, true, "Инкогнито 0501");

        p.ExtendedPhones_["333"] = TDbProfile::TAttrs({{"1", TDbValue("89252252525")},
                                                       {"102", TDbValue("+79252252525")}});
        CheckDisplayName(hSimple.Result(&p), "nphne-abcdef", "+79252252525", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "nphne-abcdef", "+79252252525", "0/0-0", true, true, "Инкогнито 0501");

        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue("  ");
        p.Attrs_[TAttr::PERSON_LASTNAME] = TDbValue("\t\t");

        CheckDisplayName(hSimple.Result(&p), "nphne-abcdef", "+79252252525", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "nphne-abcdef", "+79252252525", "0/0-0", true, true, "Инкогнито 0501");

        // correct neophonish with full FIO
        p.Attrs_[TAttr::PERSON_LASTNAME] = TDbValue("Пупкин    ");

        CheckDisplayName(hSimple.Result(&p), "nphne-abcdef", "Пупкин", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "nphne-abcdef", "Пупкин", "0/0-0", true, true, "Инкогнито 0501");

        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue(" Василий ");

        CheckDisplayName(hSimple.Result(&p), "nphne-abcdef", "Василий Пупкин", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "nphne-abcdef", "Василий Пупкин", "0/0-0", true, true, "Василий П.");

        // neophonish with display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("p:анонимус");

        CheckDisplayName(hSimple.Result(&p), "nphne-abcdef", "анонимус", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "nphne-abcdef", "анонимус", "0/0-0", true, false, "анонимус");

        // neophonish with social display name, if somehow happened
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:vk:Супер Мачо");

        CheckDisplayNameSocial(hSimple.Result(&p), "nphne-abcdef", "Супер Мачо", "0/0-0", true, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "nphne-abcdef", "Супер Мачо", "0/0-0", true, false, "Супер Мачо", "12345", "vk");

        // neophonish that got portal alias
        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue("superuser");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("superuser");

        CheckDisplayNameSocial(hSimple.Result(&p), "superuser", "Супер Мачо", "0/0-0", true, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "superuser", "Супер Мачо", "0/0-0", true, false, "Супер Мачо", "12345", "vk");

        // neophonish with portal alias and display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("p:анонимус");

        CheckDisplayName(hSimple.Result(&p), "superuser", "анонимус", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "анонимус", "0/0-0", true, false, "анонимус");

        // neophonish with portal alias and no display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "0/0-0", true, true, "Василий П.");
    }

    Y_UNIT_TEST(DisplayNameHelperScholar) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);

        // regname ok, no public name
        req.Args["get_public_name"] = "no";
        TDisplayNameHelper hSimple(*conv, req, 1500000000, 1600000000);

        req.Args["get_public_name"] = "yes";
        req.Args["is_display_name_empty"] = "True";
        TDisplayNameHelper hPublic(*conv, req, 1500000000, 1600000000);

        TTestDbProfile p(fetcher->DefaultProfile());
        // scholar alias
        p.Aliases_[TAlias::SCHOLAR] = TDbValue("школьниквася");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("школьниквася");
        p.Attrs_[TAttr::ACCOUNT_USER_DEFINED_LOGIN] = TDbValue("ШкольникВася");

        CheckDisplayName(hSimple.Result(&p), "школьниквася", "школьниквася", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "школьниквася", "школьниквася", "0/0-0", true, true, "школьниквася");

        // scholar with FIO
        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue("Василий");
        p.Attrs_[TAttr::PERSON_LASTNAME] = TDbValue("Пупкин");

        CheckDisplayName(hSimple.Result(&p), "школьниквася", "школьниквася", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "школьниквася", "школьниквася", "0/0-0", true, true, "Василий П.");

        // scholar with display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("p:анонимус");

        CheckDisplayName(hSimple.Result(&p), "школьниквася", "анонимус", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "школьниквася", "анонимус", "0/0-0", true, false, "анонимус");

        // scholar with social display name, if somehow happened
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:vk:Супер Мачо");

        CheckDisplayNameSocial(hSimple.Result(&p), "школьниквася", "Супер Мачо", "0/0-0", true, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "школьниквася", "Супер Мачо", "0/0-0", true, false, "Супер Мачо", "12345", "vk");

        // scholar that got portal alias
        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue("scholar-login");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("Scholar-Login");

        CheckDisplayNameSocial(hSimple.Result(&p), "Scholar-Login", "Супер Мачо", "0/0-0", true, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "Scholar-Login", "Супер Мачо", "0/0-0", true, false, "Супер Мачо", "12345", "vk");

        // scholar with portal alias and no display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("");

        CheckDisplayName(hSimple.Result(&p), "Scholar-Login", "Scholar-Login", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "Scholar-Login", "Scholar-Login", "0/0-0", true, true, "Василий П.");
    }

    Y_UNIT_TEST(DisplayNameHelperSocial) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);

        // regname ok, no public name
        req.Args["get_public_name"] = "no";
        TDisplayNameHelper hSimple(*conv, req, 1500000000, 1600000000);

        req.Args["get_public_name"] = "yes";
        req.Args["is_display_name_empty"] = "True";
        TDisplayNameHelper hPublic(*conv, req, 1500000000, 1600000000);

        TTestDbProfile p(fetcher->DefaultProfile());

        // social alias
        p.Aliases_[TAlias::SOCIAL_LOGIN] = TDbValue("uid-brbrbr");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("uid-brbrbr");

        CheckDisplayName(hSimple.Result(&p), "uid-brbrbr", "uid-brbrbr", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "uid-brbrbr", "uid-brbrbr", "0/0-0", true, true, "uid-brbrbr");

        // empty social displayname
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:");

        CheckDisplayName(hSimple.Result(&p), "uid-brbrbr", "uid-brbrbr", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "uid-brbrbr", "uid-brbrbr", "0/0-0", true, true, "uid-brbrbr");

        // bad social displayname + some attrs
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345");
        p.Attrs_[TAttr::AVATAR_DEFAULT] = TDbValue("0/0-0");

        CheckDisplayName(hSimple.Result(&p), "uid-brbrbr", "uid-brbrbr", "0/0-0", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "uid-brbrbr", "uid-brbrbr", "0/0-0", false, true, "uid-brbrbr");

        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:");
        p.Attrs_[TAttr::AVATAR_DEFAULT] = TDbValue("1/2-3");

        CheckDisplayName(hSimple.Result(&p), "uid-brbrbr", "uid-brbrbr", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "uid-brbrbr", "uid-brbrbr", "1/2-3", false, true, "uid-brbrbr");

        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:vk");
        p.Attrs_[TAttr::PERSON_LASTNAME] = TDbValue("Лоханкин");

        CheckDisplayName(hSimple.Result(&p), "uid-brbrbr", "uid-brbrbr", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "uid-brbrbr", "uid-brbrbr", "1/2-3", false, true, "uid-brbrbr");

        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:vk:");
        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue("Васисуалий");

        CheckDisplayName(hSimple.Result(&p), "uid-brbrbr", "uid-brbrbr", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "uid-brbrbr", "uid-brbrbr", "1/2-3", false, true, "Васисуалий Л.");

        // has social displayname
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:vk:Супер Мачо");

        CheckDisplayNameSocial(hSimple.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, false, "Супер Мачо", "12345", "vk");

        // has social displayname and can't use it as publicname
        p.Attrs_[TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME] = TDbValue("1");

        CheckDisplayNameSocial(hSimple.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, false, "Васисуалий Л.", "12345", "vk");

        // social that allowed full name in public name
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("100");

        CheckDisplayNameSocial(hSimple.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, false, "Васисуалий Лоханкин", "12345", "vk");

        // social that registered after fullFioStartTime
        p.Attrs_[TAttr::ACCOUNT_REGISTRATION_DATETIME] = TDbValue("1700000000");
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("00");

        CheckDisplayNameSocial(hSimple.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "uid-brbrbr", "Супер Мачо", "1/2-3", false, false, "Васисуалий Лоханкин", "12345", "vk");

        // social that got portal alias
        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue("superuser");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("superuser");

        CheckDisplayNameSocial(hSimple.Result(&p), "superuser", "Супер Мачо", "1/2-3", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "superuser", "Супер Мачо", "1/2-3", false, false, "Васисуалий Лоханкин", "12345", "vk");

        // ex-social that has social display name that is not allowed and full fio not allowed
        p.Attrs_[TAttr::ACCOUNT_REGISTRATION_DATETIME] = TDbValue("1400000000");
        CheckDisplayNameSocial(hSimple.Result(&p), "superuser", "Супер Мачо", "1/2-3", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "superuser", "Супер Мачо", "1/2-3", false, false, "Васисуалий Л.", "12345", "vk");

        // ex-social that has social display name allowed to show
        p.Attrs_[TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME] = TDbValue("000");

        CheckDisplayNameSocial(hSimple.Result(&p), "superuser", "Супер Мачо", "1/2-3", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "superuser", "Супер Мачо", "1/2-3", false, false, "Супер Мачо", "12345", "vk");

        // ex-social that got passport display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("p:анонимус");

        CheckDisplayName(hSimple.Result(&p), "superuser", "анонимус", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "анонимус", "1/2-3", false, false, "анонимус");

        // ex-social with broken display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("ping");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "1/2-3", false, true, "Васисуалий Л.");

        // ex-social with phone alias in display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("pna:Телефонный Алиас");

        CheckDisplayName(hSimple.Result(&p), "superuser", "Телефонный Алиас", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "Телефонный Алиас", "1/2-3", false, false, "Телефонный Алиас");
    }

    Y_UNIT_TEST(DisplayNameHelperPdd) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);

        // regname ok, no public name
        req.Args["get_public_name"] = "no";
        TDisplayNameHelper hSimple(*conv, req, 1500000000, 1600000000);

        req.Args["get_public_name"] = "yes";
        req.Args["is_display_name_empty"] = "True";
        TDisplayNameHelper hPublic(*conv, req, 1500000000, 1600000000);

        TTestDbProfile p(fetcher->DefaultProfile());

        p.DomCache_ = std::make_shared<TDomainCache>();
        p.DomCache_->LastEventId = "6666";
        p.DomItem_ = TDomain({
            .Id = "111",
            .Master = "4",
            .Name = "xn-----6kccl6ap0agkc7a2i.xn--p1ai",
            .DefaultUid = "400500600",
            .Options = {
                .OrganizationName = "Рога && Копыта LTD",
                .OrganizationId = "356",
            },
        });

        p.Aliases_[TAlias::PDD_MASTER_LOGIN] = TDbValue("Фунт@рога-и-копыта.рф");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("Фунт@рога-и-копыта.рф");

        CheckDisplayName(hSimple.Result(&p), "Фунт@рога-и-копыта.рф", "Фунт@рога-и-копыта.рф", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "Фунт@рога-и-копыта.рф", "Фунт@рога-и-копыта.рф", "0/0-0", true, true, "Фунт@рога-и-копыта.рф");

        // public_name for PDD
        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue("Фунт");
        p.Attrs_[TAttr::PERSON_LASTNAME] = TDbValue("Лиха");
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("1");

        CheckDisplayName(hSimple.Result(&p), "Фунт@рога-и-копыта.рф", "Фунт@рога-и-копыта.рф", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "Фунт@рога-и-копыта.рф", "Фунт@рога-и-копыта.рф", "0/0-0", true, true, "Фунт Лиха");

        // PDD with template displayname
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("t: I'm %pdd_username% from %pdd_domain% (%firstname% %lastname% aka %login%) %cool%!");

        CheckDisplayName(hSimple.Result(&p), "Фунт@рога-и-копыта.рф", " I'm Фунт from рога-и-копыта.рф (Фунт Лиха aka Фунт@рога-и-копыта.рф) %cool%!", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "Фунт@рога-и-копыта.рф", " I'm Фунт from рога-и-копыта.рф (Фунт Лиха aka Фунт@рога-и-копыта.рф) %cool%!", "0/0-0", true, false, " I'm Фунт from рога-и-копыта.рф (Фунт Лиха aka Фунт@рога-и-копыта.рф) %cool%!");

        // PDD with passport display name and avatar
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("p:анонимус");
        p.Attrs_[TAttr::AVATAR_DEFAULT] = TDbValue("100/500-1000000");

        CheckDisplayName(hSimple.Result(&p), "Фунт@рога-и-копыта.рф", "анонимус", "100/500-1000000", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "Фунт@рога-и-копыта.рф", "анонимус", "100/500-1000000", false, false, "анонимус");

        // PDD with social display name
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("s:12345:vk:Супер Мачо");

        CheckDisplayNameSocial(hSimple.Result(&p), "Фунт@рога-и-копыта.рф", "Супер Мачо", "100/500-1000000", false, "12345", "vk");
        CheckDisplayNamePublicSocial(hPublic.Result(&p), "Фунт@рога-и-копыта.рф", "Супер Мачо", "100/500-1000000", false, false, "Супер Мачо", "12345", "vk");
    }

    Y_UNIT_TEST(DisplayNameHelper) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);

        // regname ok, no public name
        req.Args["get_public_name"] = "no";
        TDisplayNameHelper hSimple(*conv, req, 1500000000, 1600000000);

        req.Args["get_public_name"] = "yes";
        req.Args["is_display_name_empty"] = "True";
        TDisplayNameHelper hPublic(*conv, req, 1500000000, 1600000000);

        TTestDbProfile p(fetcher->DefaultProfile());

        // empty profile/attrs
        CheckDisplayName(hSimple.Result(&p), "", "", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "", "", "0/0-0", true, true, "");

        // has only portal login
        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue("superuser");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("superuser");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "0/0-0", true, true, "superuser");

        // has portal login + fio
        p.Attrs_[TAttr::PERSON_FIRSTNAME] = TDbValue("Васисуалий");
        p.Attrs_[TAttr::PERSON_LASTNAME] = TDbValue("Лоханкин");

        CheckDisplayName(hSimple.Result(&p), "superuser", "superuser", "0/0-0", true);
        CheckDisplayNamePublic(hPublic.Result(&p), "superuser", "superuser", "0/0-0", true, true, "Васисуалий Л.");

        // has default avatar and user defined login
        p.Attrs_[TAttr::AVATAR_DEFAULT] = TDbValue("0/0-0");
        p.Attrs_[TAttr::ACCOUNT_USER_DEFINED_LOGIN] = TDbValue("SuperUser");
        {
            // here is simple hack to rebuild regname dbfield inside TDisplayNameHelper
            // if we don't change to different profile, it will always return cached value
            TTestDbProfile dummy(fetcher->DefaultProfile());
            dummy.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("another_user");

            CheckDisplayName(hSimple.Result(&dummy), "another_user", "another_user", "0/0-0", true);
            CheckDisplayNamePublic(hPublic.Result(&dummy), "another_user", "another_user", "0/0-0", true, true, "another_user");
        }

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "SuperUser", "0/0-0", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "SuperUser", "0/0-0", false, true, "Васисуалий Л.");

        // has non-default avatar
        p.Attrs_[TAttr::AVATAR_DEFAULT] = TDbValue("1/2-3");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "SuperUser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "SuperUser", "1/2-3", false, true, "Васисуалий Л.");

        // show full FIO is on
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("100");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "SuperUser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "SuperUser", "1/2-3", false, true, "Васисуалий Лоханкин");

        // has broken displayname
        p.Attrs_[TAttr::PERSON_SHOW_FIO_IN_PUBLIC_NAME] = TDbValue("01");
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("Василий Великий");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "SuperUser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "SuperUser", "1/2-3", false, true, "Васисуалий Л.");

        // has passport displayname
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("p:Василий Великий");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "Василий Великий", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "Василий Великий", "1/2-3", false, false, "Василий Великий");

        // has displayname but it is turned off
        p.Attrs_[TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME] = TDbValue("1");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "Василий Великий", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "Василий Великий", "1/2-3", false, false, "Васисуалий Л.");

        // has phonealias displayname
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("pna:Телефонный Алиас");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "Телефонный Алиас", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "Телефонный Алиас", "1/2-3", false, false, "Васисуалий Л.");

        // using displayname as publicname is on again
        p.Attrs_[TAttr::PERSON_DONT_USE_DISPLAYNAME_AS_PUBLICNAME] = TDbValue("");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "Телефонный Алиас", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "Телефонный Алиас", "1/2-3", false, false, "Телефонный Алиас");

        // bad displayname prefix
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("png:Василий Великий");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "SuperUser", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "SuperUser", "1/2-3", false, true, "Васисуалий Л.");

        // template displayname
        p.Attrs_[TAttr::ACCOUNT_DISPLAY_NAME] = TDbValue("t:Вася (%lastname% %firstname% aka %login%) %cool%!");

        CheckDisplayName(hSimple.Result(&p), "SuperUser", "Вася (Лоханкин Васисуалий aka SuperUser) %cool%!", "1/2-3", false);
        CheckDisplayNamePublic(hPublic.Result(&p), "SuperUser", "Вася (Лоханкин Васисуалий aka SuperUser) %cool%!", "1/2-3", false, false, "Вася (Лоханкин Васисуалий aka SuperUser) %cool%!");
    }

    Y_UNIT_TEST(VerifiedFlag) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);
        TDisplayNameHelper h(*conv, req, 0, 0);

        TTestDbProfile p(fetcher->DefaultProfile());

        // account is verified, check for different users
        p.Attrs_[TAttr::ACCOUNT_IS_VERIFIED] = TDbValue("1");

        // phonish
        p.Aliases_[TAlias::PHONY_LOGIN] = TDbValue("phne-eprst");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("phne-eprst");

        UNIT_ASSERT(h.Result(&p)->Verified);

        // neophonish
        p.Aliases_[TAlias::NEOPHONISH] = TDbValue("nphne-abcdef");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("nphne-abcdef");

        UNIT_ASSERT(h.Result(&p)->Verified);

        // social
        p.Aliases_[TAlias::SOCIAL_LOGIN] = TDbValue("uid-brbrbr");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("uid-brbrbr");

        UNIT_ASSERT(h.Result(&p)->Verified);

        // with portal login
        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue("superuser");
        p.Attrs_[TAttr::ACCOUNT_NORMALIZED_LOGIN] = TDbValue("superuser");

        UNIT_ASSERT(h.Result(&p)->Verified);
    }

    Y_UNIT_TEST(PublicDataFlags) {
        NTest::TRequest req;
        req.Args["regname"] = "yes";

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> conv = Db().CreateConverter(*fetcher);

        req.Args["get_public_name"] = "yes";
        TDisplayNameHelper h(*conv, req, 1500000000, 1600000000);

        TTestDbProfile p(fetcher->DefaultProfile());

        int flag1 = p.AddAttr(TAttr::ACCOUNT_PERSONAL_DATA_PUBLIC_ACCESS_ALLOWED);
        int flag2 = p.AddAttr(TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED);

        // old users
        p.Attrs_[TAttr::ACCOUNT_REGISTRATION_DATETIME] = TDbValue("1400000000");

        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag2));

        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_PUBLIC_ACCESS_ALLOWED] = TDbValue("0");

        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag2));

        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED] = TDbValue("0");

        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag2));

        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED] = TDbValue("1");

        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag2));

        std::optional<TPublicNameData> data = h.GetPublicName(&p, "");
        UNIT_ASSERT(data);
        UNIT_ASSERT_VALUES_EQUAL("", data->Name);
        UNIT_ASSERT_VALUES_EQUAL(false, data->HasPublicProfile);
        UNIT_ASSERT_VALUES_EQUAL(true, data->ThirdPartyCanUse);

        // new users
        p.Attrs_[TAttr::ACCOUNT_REGISTRATION_DATETIME] = TDbValue("1600000000");
        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_PUBLIC_ACCESS_ALLOWED] = TDbValue("");
        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED] = TDbValue("");

        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag2));

        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_PUBLIC_ACCESS_ALLOWED] = TDbValue("1");

        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag2));

        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED] = TDbValue("2");

        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag2));

        p.Attrs_[TAttr::ACCOUNT_PERSONAL_DATA_THIRD_PARTY_PROCESSING_ALLOWED] = TDbValue("0");

        UNIT_ASSERT_VALUES_EQUAL(true, h.GetPublicDataFlag(&p, flag1));
        UNIT_ASSERT_VALUES_EQUAL(false, h.GetPublicDataFlag(&p, flag2));

        data = h.GetPublicName(&p, "My Name");
        UNIT_ASSERT(data);
        UNIT_ASSERT_VALUES_EQUAL("My Name", data->Name);
        UNIT_ASSERT_VALUES_EQUAL(true, data->HasPublicProfile);
        UNIT_ASSERT_VALUES_EQUAL(false, data->ThirdPartyCanUse);
    }

    Y_UNIT_TEST(CheckGrants) {
        NTest::TRequest req;
        TConsumer c;
        TGrantsChecker checker(req, c, false);

        auto check = [&](const std::set<TString>& expected) {
            TDisplayNameHelper::CheckGrants(checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };
        check({});

        req.Args["regname"] = "yes";
        check({});

        req.Args["get_public_name"] = "yes";
        check({"no grants for get_public_name=yes"});

        req.Args["regname"] = "no";
        TDisplayNameHelper::CheckGrants(checker);
        check({});

        req.Args["regname"] = "yes";
        c.SetAllow(TBlackboxFlags::GetPublicName, true);
        check({});
    }
}
