#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/oauth_attrs_helper.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>
#include <passport/infra/daemons/blackbox/src/output/attributes_chunk.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspUtOAuthAttrsHelper) {
    static TTestDbHolder& Db() {
        return TTestDbHolder::GetSingleton();
    }

    static void CheckOAuthAttrs(const TTestOAuthFetcher& f,
                                const TTestOAuthFetcher::TAttrSet& clientAttrs,
                                bool allClientAttrs = false) {
        UNIT_ASSERT_VALUES_EQUAL(clientAttrs, f.AddClientAttrs_);
        UNIT_ASSERT_VALUES_EQUAL(allClientAttrs, f.AllClientAttrs_);
    }

    Y_UNIT_TEST(oauthAttrsHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestOAuthFetcher> fetcher = Db().CreateOAuthFetcher();

        // no oauth attributes arg, bad arg
        TOAuthAttrsHelper h(*fetcher, req);
        CheckOAuthAttrs(*fetcher, TTestOAuthFetcher::TAttrSet());

        req.Args["oauth_token_attributes"] = "yes";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TOAuthAttrsHelper(*fetcher, req), TBlackboxError, "invalid oauth_token_attributes value");

        req.Args["oauth_token_attributes"] = "";
        req.Args["oauth_client_attributes"] = "yes";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TOAuthAttrsHelper(*fetcher, req), TBlackboxError, "invalid oauth_client_attributes value");

        // numeric attrs
        req.Args["oauth_token_attributes"] = "4,5,1,0,13,1001,13 1001,6 ";
        req.Args["oauth_client_attributes"] = "";
        TOAuthAttrsHelper h1(*fetcher, req);
        CheckOAuthAttrs(*fetcher, TTestOAuthFetcher::TAttrSet());

        req.Args["oauth_client_attributes"] = "1 0 0 1, 5,6,7,8,10001,8,1";
        TOAuthAttrsHelper h2(*fetcher, req);
        CheckOAuthAttrs(*fetcher, TTestOAuthFetcher::TAttrSet({"1", "5", "6"}));

        // all attrs
        req.Args["oauth_token_attributes"] = "all";
        TOAuthAttrsHelper h3(*fetcher, req);
        CheckOAuthAttrs(*fetcher, TTestOAuthFetcher::TAttrSet({"1", "5", "6"}));

        req.Args["oauth_client_attributes"] = "all";
        TOAuthAttrsHelper h4(*fetcher, req);
        CheckOAuthAttrs(*fetcher, TTestOAuthFetcher::TAttrSet({"1", "5", "6"}), true);

        TOAuthTokenInfo t(true);
        t.TokenAttrs = TOAuthTokenInfo::TAttrMap({{"0", "123456"}, {"1", "ddeaabccaeda"}, {"4", "5511523"}, {"12", "ddd"}});
        t.ClientAttrs = TOAuthTokenInfo::TAttrMap({{"1", "70500"}, {"6", "1"}, {"10", "VIP client"}});

        UNIT_ASSERT(!h.TokenAttrsChunk(nullptr));
        UNIT_ASSERT(!h.ClientAttrsChunk(nullptr));
        UNIT_ASSERT(!h.TokenAttrsChunk(&t));
        UNIT_ASSERT(!h.ClientAttrsChunk(&t));

        UNIT_ASSERT(!h1.TokenAttrsChunk(nullptr));
        UNIT_ASSERT(!h1.ClientAttrsChunk(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h1.TokenAttrsChunk(&t)->Attrs,
                                 TAttributesChunk::TAttributesType({{"0", "123456"}, {"1", "ddeaabccaeda"}, {"4", "5511523"}}));
        UNIT_ASSERT(!h1.ClientAttrsChunk(&t));

        UNIT_ASSERT(!h2.TokenAttrsChunk(nullptr));
        UNIT_ASSERT(!h2.ClientAttrsChunk(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h2.TokenAttrsChunk(&t)->Attrs,
                                 TAttributesChunk::TAttributesType({{"0", "123456"}, {"1", "ddeaabccaeda"}, {"4", "5511523"}}));
        UNIT_ASSERT_VALUES_EQUAL(h2.ClientAttrsChunk(&t)->Attrs,
                                 TAttributesChunk::TAttributesType({{"1", "70500"}, {"6", "1"}}));

        UNIT_ASSERT(!h3.TokenAttrsChunk(nullptr));
        UNIT_ASSERT(!h3.ClientAttrsChunk(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h3.TokenAttrsChunk(&t)->Attrs,
                                 TAttributesChunk::TAttributesType({{"0", "123456"}, {"1", "ddeaabccaeda"}, {"4", "5511523"}, {"12", "ddd"}}));
        UNIT_ASSERT_VALUES_EQUAL(h3.ClientAttrsChunk(&t)->Attrs,
                                 TAttributesChunk::TAttributesType({{"1", "70500"}, {"6", "1"}}));

        UNIT_ASSERT(!h4.TokenAttrsChunk(nullptr));
        UNIT_ASSERT(!h4.ClientAttrsChunk(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h4.TokenAttrsChunk(&t)->Attrs,
                                 TAttributesChunk::TAttributesType({{"0", "123456"}, {"1", "ddeaabccaeda"}, {"4", "5511523"}, {"12", "ddd"}}));
        UNIT_ASSERT_VALUES_EQUAL(h4.ClientAttrsChunk(&t)->Attrs,
                                 TAttributesChunk::TAttributesType({{"1", "70500"}, {"6", "1"}, {"10", "VIP client"}}));
    }

    Y_UNIT_TEST(checkGrants) {
        NTest::TRequest req;
        TConsumer c;
        TGrantsChecker checker(req, c, false);

        auto check = [&](const std::set<TString>& expected) {
            TOAuthAttrsHelper::CheckGrants(checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };

        check({});

        req.Args["oauth_token_attributes"] = "foo";
        check({
            "no grants for oauth attributes",
        });

        req.Args["oauth_client_attributes"] = "bar";
        check({
            "no grants for oauth attributes",
        });

        req.Args["oauth_token_attributes"] = "";
        check({
            "no grants for oauth attributes",
        });

        req.Args["oauth_client_attributes"] = "";
        check({});
    }
}
