#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/grants/grants_checker.h>
#include <passport/infra/daemons/blackbox/src/helpers/attributes_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/base_result_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/billing_features_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/email_attrs_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/family_info_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/karma_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/oauth_attrs_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/partitions_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/phone_attrs_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/public_id_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/test_pin_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/uid_helper.h>
#include <passport/infra/daemons/blackbox/src/helpers/webauthn_attrs_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/attributes.h>
#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/partitions.h>
#include <passport/infra/daemons/blackbox/src/oauth/token_info.h>
#include <passport/infra/daemons/blackbox/src/output/attributes_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/auth_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/authid_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/billing_features_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/dbfields_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/emails_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/ext_attrs_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/family_info_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/karma_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/new_session_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/phone_bindings_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/phone_operations_chunk.h>
#include <passport/infra/daemons/blackbox/src/output/uid_chunk.h>

#include <passport/infra/libs/cpp/auth_core/public_id_encryptor.h>
#include <passport/infra/libs/cpp/request/test/request.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;
using namespace NPassport;

Y_UNIT_TEST_SUITE(PasspBbHelpersTestSuite) {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    std::unique_ptr<TTestDbHolder> DB;

    TTestDbHolder& Db() {
        if (!DB) {
            DB = std::make_unique<TTestDbHolder>();
        }
        return *DB;
    }

    Y_UNIT_TEST(AttributesSettings) {
        TAttributesSettings settings(
            "1,2 3 7,,,8  9 2 5",
            "8,  5, 6,5,5",
            "666,777 777",
            "8,9");

        UNIT_ASSERT(!settings.RequiresGrant("0"));
        UNIT_ASSERT(settings.RequiresGrant("1"));
        UNIT_ASSERT(settings.RequiresGrant("2"));
        UNIT_ASSERT(!settings.RequiresGrant("4"));
        UNIT_ASSERT(settings.RequiresGrant("5"));
        UNIT_ASSERT(settings.RequiresGrant("7"));
        UNIT_ASSERT(!settings.RequiresGrant("abc"));
        UNIT_ASSERT(!settings.RequiresGrant(" "));
        UNIT_ASSERT(!settings.RequiresGrant(""));
        UNIT_ASSERT(!settings.RequiresGrant(","));

        UNIT_ASSERT(!settings.IsBinary(""));
        UNIT_ASSERT(!settings.IsBinary(" "));
        UNIT_ASSERT(!settings.IsBinary("0"));
        UNIT_ASSERT(!settings.IsBinary("1"));
        UNIT_ASSERT(!settings.IsBinary("2"));
        UNIT_ASSERT(settings.IsBinary("5"));
        UNIT_ASSERT(settings.IsBinary("6"));
        UNIT_ASSERT(!settings.IsBinary("7"));
        UNIT_ASSERT(settings.IsBinary("8"));
        UNIT_ASSERT(!settings.IsBinary("abc"));

        UNIT_ASSERT(!settings.RequiresGrantPhone(""));
        UNIT_ASSERT(!settings.RequiresGrantPhone(","));
        UNIT_ASSERT(settings.RequiresGrantPhone("666"));
        UNIT_ASSERT(!settings.RequiresGrantPhone("777 777"));
        UNIT_ASSERT(settings.RequiresGrantPhone("777"));
        UNIT_ASSERT(!settings.RequiresGrantPhone("666,777"));

        UNIT_ASSERT(!settings.IsDeleted(""));
        UNIT_ASSERT(!settings.IsDeleted("1"));
        UNIT_ASSERT(!settings.IsDeleted("5"));
        UNIT_ASSERT(!settings.IsDeleted("abc"));
        UNIT_ASSERT(!settings.IsDeleted("666"));
        UNIT_ASSERT(settings.IsDeleted("8"));
        UNIT_ASSERT(settings.IsDeleted("9"));
    }

    Y_UNIT_TEST(AttributesHelperGrants) {
        NTest::TRequest req;
        TConsumer c;
        TGrantsChecker checker(req, c, false);

        TAttributesSettings settings("asd,foo,bar,", "", "", "");
        c.AddAttr("foo", TConsumer::ERank::HasCred);
        c.AddAttr("bar", TConsumer::ERank::NoCred);

        req.Args["attributes"] = "asd,asd,foo,foo,bar,bar,kek,kek";

        auto check = [&](TConsumer::ERank rank, const std::set<TString>& expected) {
            TAttributesHelper::CheckGrants(rank, settings, checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };

        check(TConsumer::ERank::NoCred,
              {
                  "no grants for attribute 'asd'",
                  "no grants for attribute 'foo'",
              });
        check(TConsumer::ERank::HasCred,
              {
                  "no grants for attribute 'asd'",
              });
    }

    Y_UNIT_TEST(AttributesHelper) {
        TAttributesSettings settings(
            "1,2 3 7,,,8  9 2 5",
            "8,  5, 6,5,5",
            "666,777 777",
            "8,9");

        NTest::TRequest req;
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TAttrs&>(fetcher->DefaultProfile().Attrs());

        TAttributesHelper h1(*fetcher, settings, req);

        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT(h1.Result(&fetcher->DefaultProfile())->Attrs.empty());

        req.Args["attributes"] = "1,1,9,8,7,5,";
        TAttributesHelper h2(*fetcher, settings, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TAttrs(
                                            {{"1", TDbValue()}, {"5", TDbValue()}, {"7", TDbValue()}}));

        UNIT_ASSERT(!h2.Result(nullptr));
        UNIT_ASSERT(h2.Result(&fetcher->DefaultProfile())->Attrs.empty());

        attrs["1"] = TDbValue("12345");
        attrs["5"] = TDbValue("12345");

        UNIT_ASSERT(!h2.Result(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h2.Result(&fetcher->DefaultProfile())->Attrs,
                                 TAttributesChunk::TAttributesType({{"1", "12345"}, {"5", "MTIzNDU="}}));
    }

    Y_UNIT_TEST(EmailAttrsHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TExtendedEntities&>(fetcher->DefaultProfile().ExtendedEmailAttrs());

        // no getemails arg, bad getemails arg
        UNIT_ASSERT_EXCEPTION_CONTAINS(TEmailAttrsHelper(*fetcher, req), TBlackboxError, "unknown getemails parameter value: ''");
        UNIT_ASSERT(attrs.empty());

        req.Args["getemails"] = "one";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TEmailAttrsHelper(*fetcher, req), TBlackboxError, "unknown getemails parameter value: 'one'");
        UNIT_ASSERT(attrs.empty());

        // getemails all
        req.Args["getemails"] = "all";
        req.Args["email_attributes"] = "3,5,15,,5,3";
        TEmailAttrsHelper h1(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TExtendedEntities({{"",
                                                                        TDbProfile::TAttrs({{TEmailAttr::ADDRESS, TDbValue()},
                                                                                            {"3", TDbValue()},
                                                                                            {"5", TDbValue()},
                                                                                            {"15", TDbValue()}})}}));

        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT(h1.Result(&fetcher->DefaultProfile())->Attrs.empty());

        attrs[""]["1"] = TDbValue("not used, because");
        attrs[""]["5"] = TDbValue("values from empty entity ignored");
        attrs["777"] = TDbProfile::TAttrs({{"1", TDbValue("a@b.c")}, {"2", TDbValue("1414153621")}, {"3", TDbValue("1517193621")}, {"5", TDbValue("1")}});
        attrs["555"] = TDbProfile::TAttrs({{"1", TDbValue("")}, {"2", TDbValue("1414159995")}, {"5", TDbValue("")}});

        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h1.Result(&fetcher->DefaultProfile())->Attrs,
                                 TExtAttrsChunk::TExtAttrs({{"555", TExtAttrsChunk::TAttrs({{"5", ""}})},
                                                            {"777", TExtAttrsChunk::TAttrs({{"3", "1517193621"}, {"5", "1"}})}}));
    }

    Y_UNIT_TEST(EmailAttrsHelperAll) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TExtendedEntities&>(fetcher->DefaultProfile().ExtendedEmailAttrs());

        // all email_attributes
        req.Args["email_attributes"] = "all";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TEmailAttrsHelper(*fetcher, req), TBlackboxError, "unknown getemails parameter value: ''");
        UNIT_ASSERT(attrs.empty());

        req.Args["getemails"] = "all";
        TEmailAttrsHelper h(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TExtendedEntities({{"",
                                                                        TDbProfile::TAttrs({{"1", TDbValue()},
                                                                                            {"2", TDbValue()},
                                                                                            {"3", TDbValue()},
                                                                                            {"4", TDbValue()},
                                                                                            {"5", TDbValue()},
                                                                                            {"6", TDbValue()},
                                                                                            {"7", TDbValue()}})}}));
    }

    static void CheckKarma(const std::unique_ptr<TKarmaChunk>& c, long k, long v, const TString& s = "") {
        UNIT_ASSERT_VALUES_EQUAL(k, c->Karma);
        UNIT_ASSERT_VALUES_EQUAL(v, c->RawValue);
        UNIT_ASSERT_VALUES_EQUAL(s, c->Bantime);
    }

    Y_UNIT_TEST(KarmaHelper) {
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TAttrs&>(fetcher->DefaultProfile().Attrs());

        TKarmaHelper h1(*fetcher, false);
        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TAttrs({{TAttr::KARMA_VALUE, TDbValue()}}));

        TKarmaHelper h2(*fetcher, true);
        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TAttrs({{TAttr::KARMA_VALUE, TDbValue()},
                                                            {TAttr::KARMA_ACTIVATION_DATETIME, TDbValue()}}));

        // null profile, no value
        CheckKarma(h2.Result(nullptr), 0, 0);

        // bad karma value
        attrs[TAttr::KARMA_VALUE] = TDbValue("abc");

        CheckKarma(h2.Result(&fetcher->DefaultProfile()), 0, 0);

        // negative with prefix
        attrs[TAttr::KARMA_VALUE] = TDbValue("-2048");
        attrs[TAttr::KARMA_ACTIVATION_DATETIME] = TDbValue("1234567890");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), -48, -2048);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), -48, -2048);

        // negative without prefix
        attrs[TAttr::KARMA_VALUE] = TDbValue("-42");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), -42, -42);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), -42, -42, "1234567890");

        // without prefix and bad bantime
        attrs[TAttr::KARMA_ACTIVATION_DATETIME] = TDbValue("1234");
        attrs[TAttr::KARMA_VALUE] = TDbValue("42");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), 42, 42);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), 42, 42);

        attrs[TAttr::KARMA_ACTIVATION_DATETIME] = TDbValue("1234567890");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), 42, 42);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), 42, 42, "1234567890");

        attrs[TAttr::KARMA_VALUE] = TDbValue("1042");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), 100, 1042);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), 100, 1042);

        attrs[TAttr::KARMA_VALUE] = TDbValue("3042");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), 100, 3042);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), 100, 3042);

        attrs[TAttr::KARMA_VALUE] = TDbValue("6042");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), 0, 6042);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), 0, 6042);

        attrs[TAttr::KARMA_VALUE] = TDbValue("7042");

        CheckKarma(h1.Result(&fetcher->DefaultProfile()), 85, 7042);
        CheckKarma(h2.Result(&fetcher->DefaultProfile()), 85, 7042);
    }

    Y_UNIT_TEST(PhoneAttrsHelperGrants) {
        NTest::TRequest req;
        TConsumer c;
        TGrantsChecker checker(req, c, false);

        TAttributesSettings settings("", "", "asd,foo,bar,101", "");
        c.AddPhoneAttr("foo", TConsumer::ERank::HasCred);
        c.AddPhoneAttr("bar", TConsumer::ERank::NoCred);

        req.Args["phone_attributes"] = "asd,asd,foo,foo,bar,bar,kek,kek";

        auto check = [&](TConsumer::ERank rank, const std::set<TString>& expected) {
            TPhoneAttrsHelper::CheckGrants(rank, settings, checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };

        check(TConsumer::ERank::NoCred, {});
        check(TConsumer::ERank::HasCred, {});

        req.Args["getphones"] = "";

        check(TConsumer::ERank::NoCred,
              {
                  "no grants for phone attribute 'asd'",
                  "no grants for phone attribute 'foo'",
              });
        check(TConsumer::ERank::HasCred,
              {
                  "no grants for phone attribute 'asd'",
              });

        req.Args["phone_attributes"] = "all";
        check(TConsumer::ERank::NoCred,
              {
                  "no grants for phone attribute '101'",
              });
        check(TConsumer::ERank::HasCred,
              {
                  "no grants for phone attribute '101'",
              });
    }

    Y_UNIT_TEST(PhoneAttrsHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TExtendedEntities&>(fetcher->DefaultProfile().ExtendedPhoneAttrs());

        // no getphones arg, bad getphones arg
        UNIT_ASSERT_EXCEPTION_CONTAINS(TPhoneAttrsHelper(*fetcher, req), TBlackboxError, "unknown getphones parameter value: ''");
        UNIT_ASSERT(attrs.empty());

        req.Args["getphones"] = "one";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TPhoneAttrsHelper(*fetcher, req), TBlackboxError, "unknown getphones parameter value: 'one'");
        UNIT_ASSERT(attrs.empty());

        // getphones all
        req.Args["getphones"] = "all";
        req.Args["phone_attributes"] = "5,2,2, ,105,5,2";
        TPhoneAttrsHelper h1(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TExtendedEntities({{"",
                                                                        TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()},
                                                                                            {"2", TDbValue()},
                                                                                            {"4", TDbValue()},
                                                                                            {"5", TDbValue()},
                                                                                            {"105", TDbValue()}})}}));

        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT(h1.Result(&fetcher->DefaultProfile())->Attrs.empty());

        // getphones bound
        req.Args["getphones"] = "bound";
        req.Args["phone_attributes"] = "3,1";
        TPhoneAttrsHelper h2(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TExtendedEntities({{"",
                                                                        TDbProfile::TAttrs({{TPhoneAttr::NUMBER, TDbValue()},
                                                                                            {TPhoneAttr::BOUND, TDbValue()},
                                                                                            {"2", TDbValue()},
                                                                                            {"4", TDbValue()},
                                                                                            {"5", TDbValue()},
                                                                                            {"105", TDbValue()}})}}));

        UNIT_ASSERT(!h2.Result(nullptr));
        UNIT_ASSERT(h2.Result(&fetcher->DefaultProfile())->Attrs.empty());

        attrs[""]["1"] = TDbValue("not used, because");
        attrs[""]["3"] = TDbValue("values from empty entity ignored");
        attrs[""]["5"] = TDbValue("for extended phone attributes too");
        attrs["2234"] = TDbProfile::TAttrs({{"1", TDbValue("79169161616")}, {"2", TDbValue("1414155651")}, {"5", TDbValue("1517457621")}, {"6", TDbValue("122")}});
        attrs["22"] = TDbProfile::TAttrs({{"1", TDbValue()}, {"2", TDbValue("1414159995")}, {"4", TDbValue()}, {"3", TDbValue("1414159995")}, {"105", TDbValue("1")}});
        attrs["223"] = TDbProfile::TAttrs({{"1", TDbValue("79252252525")}, {"4", TDbValue("1")}, {"3", TDbValue("1414159995")}, {"105", TDbValue("")}});

        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h1.Result(&fetcher->DefaultProfile())->Attrs,
                                 TExtAttrsChunk::TExtAttrs({{"22", TExtAttrsChunk::TAttrs({{"105", "1"}, {"2", "1414159995"}})},
                                                            {"223", TExtAttrsChunk::TAttrs({{"105", ""}})},
                                                            {"2234", TExtAttrsChunk::TAttrs({{"2", "1414155651"}, {"5", "1517457621"}})}}));

        UNIT_ASSERT(!h2.Result(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h2.Result(&fetcher->DefaultProfile())->Attrs,
                                 TExtAttrsChunk::TExtAttrs({{"223", TExtAttrsChunk::TAttrs({{"1", "79252252525"}, {"3", "1414159995"}})}}));
    }

    Y_UNIT_TEST(PhoneAttrsHelperAll) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TExtendedEntities&>(fetcher->DefaultProfile().ExtendedPhoneAttrs());

        // all phone_attributes
        req.Args["phone_attributes"] = "all";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TPhoneAttrsHelper(*fetcher, req), TBlackboxError, "unknown getphones parameter value: ''");
        UNIT_ASSERT(attrs.empty());

        req.Args["getphones"] = "all";
        TPhoneAttrsHelper h(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TExtendedEntities({{"",
                                                                        TDbProfile::TAttrs({{"1", TDbValue()},
                                                                                            {"2", TDbValue()},
                                                                                            {"3", TDbValue()},
                                                                                            {"4", TDbValue()},
                                                                                            {"5", TDbValue()},
                                                                                            {"6", TDbValue()},
                                                                                            {"101", TDbValue()},
                                                                                            {"102", TDbValue()},
                                                                                            {"103", TDbValue()},
                                                                                            {"104", TDbValue()},
                                                                                            {"105", TDbValue()},
                                                                                            {"106", TDbValue()},
                                                                                            {"107", TDbValue()},
                                                                                            {"108", TDbValue()},
                                                                                            {"109", TDbValue()}})}}));
    }

    Y_UNIT_TEST(PublicIdHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& aliases = const_cast<TDbProfile::TAliases&>(fetcher->DefaultProfile().Aliases());
        auto& attrs = const_cast<TDbProfile::TAttrs&>(fetcher->DefaultProfile().Attrs());

        NAuth::TPublicIdEncryptor enc(NAuth::TKeyMap::CreateFromRawMap(
            {{1, NUtils::Hex2bin("d96c0686dd45d82326ef13b3140038d4672d36858b1c4bcbe62759ed578e389b")},
             {2, NUtils::Hex2bin("7784a739e1d836297bf933005c486ec3cef9e0722adcc5ea81d83e46f8820516")}},
            1));

        // no get_public_id arg
        TPublicIdHelper h1(*fetcher, req, &enc);

        UNIT_ASSERT(aliases.empty());
        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT(!h1.Result(&fetcher->DefaultProfile()));

        // bad get_public_id arg
        req.Args["get_public_id"] = "hochu!";
        TPublicIdHelper h2(*fetcher, req, &enc);

        UNIT_ASSERT(aliases.empty());
        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(!h2.Result(nullptr));
        UNIT_ASSERT(!h2.Result(&fetcher->DefaultProfile()));

        // no encryptor
        req.Args["get_public_id"] = "yes";
        TPublicIdHelper h3(*fetcher, req, nullptr);

        UNIT_ASSERT(aliases.empty());
        UNIT_ASSERT(attrs.empty());
        UNIT_ASSERT(!h3.Result(nullptr));
        UNIT_ASSERT(!h3.Result(&fetcher->DefaultProfile()));

        // good get_public_id arg, no uid
        TPublicIdHelper h4(*fetcher, req, &enc);
        TTestDbProfile p(fetcher->DefaultProfile());

        UNIT_ASSERT_VALUES_EQUAL(aliases, TDbProfile::TAliases({{TAlias::PUBLICID, TDbValue()}}));
        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TAttrs({{TAttr::ACCOUNT_PUBLIC_ID_VERSION, TDbValue()}, {TAttr::ACCOUNT_USER_DEFINED_PUBLIC_ID, TDbValue()}}));
        UNIT_ASSERT(!h4.Result(nullptr));
        UNIT_ASSERT(!h4.Result(&p));

        // good uid, no alias
        p.SetUid("100500");
        UNIT_ASSERT_STRINGS_EQUAL("fucka6ynnuv6pyk1b79v1pucgt", *h4.Result(&p));

        p.AddAttr(TAttr::ACCOUNT_PUBLIC_ID_VERSION, "3");
        UNIT_ASSERT_STRINGS_EQUAL("774c8uamha06ga154dx70j8vh5", *h4.Result(&p));

        p.AddAttr(TAttr::ACCOUNT_PUBLIC_ID_VERSION, "no");
        UNIT_ASSERT_STRINGS_EQUAL("fucka6ynnuv6pyk1b79v1pucgt", *h4.Result(&p));

        // has alias
        p.AddAlias(TAlias::PUBLICID, "super-macho");
        UNIT_ASSERT_STRINGS_EQUAL("super-macho", *h4.Result(&p));

        // alias and bad user defined attr
        p.AddAttr(TAttr::ACCOUNT_USER_DEFINED_PUBLIC_ID, "old.alias");
        UNIT_ASSERT_STRINGS_EQUAL("super-macho", *h4.Result(&p));

        // alias and good user defined attr
        p.AddAttr(TAttr::ACCOUNT_USER_DEFINED_PUBLIC_ID, "Super.Macho");
        UNIT_ASSERT_STRINGS_EQUAL("Super.Macho", *h4.Result(&p));
    }

    static void CheckUidChunk(const TUidChunk& c, bool hosted, const TString& uid, const TString& domain,
                              const TString& domId, bool mx, bool domEna, bool lite, bool catchAll) {
        UNIT_ASSERT_VALUES_EQUAL(c.Hosted, hosted);
        UNIT_ASSERT_VALUES_EQUAL(c.Uid, uid);
        UNIT_ASSERT_VALUES_EQUAL(c.Domain, domain);
        UNIT_ASSERT_VALUES_EQUAL(c.DomainId, domId);
        UNIT_ASSERT_VALUES_EQUAL(c.Mx, mx);
        UNIT_ASSERT_VALUES_EQUAL(c.DomainEna, domEna);
        UNIT_ASSERT_VALUES_EQUAL(c.Lite, lite);
        UNIT_ASSERT_VALUES_EQUAL(c.CatchAll, catchAll);
    }

    Y_UNIT_TEST(UidHelper) {
        std::unique_ptr<TUidChunk> c;
        TTestDbProfile p;
        p.FillWithSampleData();
        TDomain emptyDomain(TDomain::TDomainParams{});
        TDomain someDomain({
            .Id = "123",
            .Name = "some.domain.ru",
        });
        TDomain utfDomain({
            .Id = "12345",
            .Master = "123",
            .Name = "xn----htbdnepicdpwl.xn--p1ai",
            .Ena = true,
            .Mx = true,
        });

        c = TUidHelper::Result(nullptr, true, emptyDomain);
        CheckUidChunk(*c, false, "", "", "", false, false, false, false);

        c = TUidHelper::Result(nullptr, false, someDomain);
        CheckUidChunk(*c, true, "", "some.domain.ru", "123", false, false, false, false);

        c = TUidHelper::Result(&p, true, emptyDomain);
        CheckUidChunk(*c, false, "70501", "", "", false, false, true, false);

        c = TUidHelper::Result(&p, false, someDomain);
        CheckUidChunk(*c, false, "70501", "some.domain.ru", "123", false, false, false, true);

        p.Aliases_[TAlias::PORTAL_LOGIN] = TDbValue();
        c = TUidHelper::Result(&p, false, someDomain);
        CheckUidChunk(*c, true, "70501", "some.domain.ru", "123", false, false, false, true);

        c = TUidHelper::Result(&p, true, utfDomain);
        CheckUidChunk(*c, true, "70501", "xn----htbdnepicdpwl.xn--p1ai", "12345", true, true, true, true);

        p.CatchAll_ = false;
        c = TUidHelper::Result(&p, false, utfDomain);
        CheckUidChunk(*c, true, "70501", "xn----htbdnepicdpwl.xn--p1ai", "12345", true, true, false, false);
    }

    Y_UNIT_TEST(WebauthnAttrsHelperGrants) {
        NTest::TRequest req;
        TConsumer c;
        TGrantsChecker checker(req, c, false);

        auto check = [&](const std::set<TString>& expected) {
            TWebauthnAttrsHelper::CheckGrants(checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };

        check({});

        req.Args["get_webauthn_credentials"] = "";
        check({"no grants for arg 'get_webauthn_credentials'"});

        c.SetAllow(TBlackboxFlags::GetWebauthnCredentials, true);
        check({});

        req.Args["webauthn_credential_attributes"] = "all";
        check({});
    }

    Y_UNIT_TEST(WebauthnAttrsHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TExtendedEntities&>(fetcher->DefaultProfile().ExtendedWebauthnAttrs());

        // no get_webauthn_credentials arg, bad get_webauthn_credentials arg
        UNIT_ASSERT_EXCEPTION_CONTAINS(TWebauthnAttrsHelper(*fetcher, req), TBlackboxError, "unknown get_webauthn_credentials parameter value: ''");
        UNIT_ASSERT(attrs.empty());

        req.Args["get_webauthn_credentials"] = "one";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TWebauthnAttrsHelper(*fetcher, req), TBlackboxError, "unknown get_webauthn_credentials parameter value: 'one'");
        UNIT_ASSERT(attrs.empty());

        // get_webauthn_credentials all
        req.Args["get_webauthn_credentials"] = "all";
        req.Args["webauthn_credential_attributes"] = "3,5,10,,";
        TWebauthnAttrsHelper h1(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TExtendedEntities({{"",
                                                                        TDbProfile::TAttrs({{TWebauthnAttr::EXTERNAL_ID, TDbValue()},
                                                                                            {"3", TDbValue()},
                                                                                            {"5", TDbValue()},
                                                                                            {"10", TDbValue()}})}}));

        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT(h1.Result(&fetcher->DefaultProfile())->Attrs.empty());

        attrs[""]["1"] = TDbValue("not used, because");
        attrs[""]["5"] = TDbValue("values from empty entity ignored");
        attrs[""]["7"] = TDbValue("for extended webauthn attributes too");
        attrs["13"] = TDbProfile::TAttrs({{"1", TDbValue("asdfasdfasdf")}, {"2", TDbValue("qwertyuiop")}, {"3", TDbValue("150")}, {"5", TDbValue("1517193621")}});
        attrs["333"] = TDbProfile::TAttrs({{"1", TDbValue("")}, {"4", TDbValue("iPhone")}, {"5", TDbValue("1414153621")}});

        UNIT_ASSERT(!h1.Result(nullptr));
        UNIT_ASSERT_VALUES_EQUAL(h1.Result(&fetcher->DefaultProfile())->Attrs,
                                 TExtAttrsChunk::TExtAttrs({{"13", TExtAttrsChunk::TAttrs({{"3", "150"}, {"5", "1517193621"}})},
                                                            {"333", TExtAttrsChunk::TAttrs({{"5", "1414153621"}})}}));
    }

    Y_UNIT_TEST(WebauthnAttrsHelperAll) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& attrs = const_cast<TDbProfile::TExtendedEntities&>(fetcher->DefaultProfile().ExtendedWebauthnAttrs());

        // all webauthn_credential_attributes
        req.Args["webauthn_credential_attributes"] = "all";
        UNIT_ASSERT_EXCEPTION_CONTAINS(TWebauthnAttrsHelper(*fetcher, req), TBlackboxError, "unknown get_webauthn_credentials parameter value: ''");
        UNIT_ASSERT(attrs.empty());

        req.Args["get_webauthn_credentials"] = "all";
        TWebauthnAttrsHelper h(*fetcher, req);

        UNIT_ASSERT_VALUES_EQUAL(attrs, TDbProfile::TExtendedEntities({{"",
                                                                        TDbProfile::TAttrs({{"1", TDbValue()},
                                                                                            {"2", TDbValue()},
                                                                                            {"3", TDbValue()},
                                                                                            {"4", TDbValue()},
                                                                                            {"5", TDbValue()},
                                                                                            {"6", TDbValue()},
                                                                                            {"7", TDbValue()},
                                                                                            {"8", TDbValue()}})}}));
    }

    Y_UNIT_TEST(FamilyInfoHelper) {
        NTest::TRequest req;

        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        auto& family = const_cast<std::optional<TFamilyInfo>&>(fetcher->DefaultProfile().FamilyInfo());

        { // nothing in request
            TFamilyInfoHelper h(*fetcher, req);

            UNIT_ASSERT(!h.Result(nullptr));
            UNIT_ASSERT(!h.Result(&fetcher->DefaultProfile()));
        }

        req.Args["get_family_info"] = "yes";
        {
            TFamilyInfoHelper h(*fetcher, req);

            UNIT_ASSERT(!h.Result(nullptr));
            std::unique_ptr<TFamilyInfoChunk> chunk = h.Result(&fetcher->DefaultProfile());
            UNIT_ASSERT(chunk);
            UNIT_ASSERT_VALUES_EQUAL("", chunk->FamilyId);
            UNIT_ASSERT_VALUES_EQUAL("", chunk->AdminUid);
        }

        family = TFamilyInfo{"__some_family_id__", "__some_uid__", "__some_place__"};
        {
            TFamilyInfoHelper h(*fetcher, req);

            UNIT_ASSERT(!h.Result(nullptr));
            std::unique_ptr<TFamilyInfoChunk> chunk = h.Result(&fetcher->DefaultProfile());
            UNIT_ASSERT(chunk);
            UNIT_ASSERT_VALUES_EQUAL("f__some_family_id__", chunk->FamilyId);
            UNIT_ASSERT_VALUES_EQUAL("__some_uid__", chunk->AdminUid);
        }
    }

    Y_UNIT_TEST(PartitionsHelperGrants) {
        NTest::TRequest req;

        TConsumer c;
        c.AddPartition("foo");
        c.AddPartition("bar");
        TGrantsChecker checker(req, c, false);

        TPartitionsSettings settings(
            TPartitionsSettings::TPartitionIdByName{
                {"default", 0},
                {"foo", 1},
                {"bar", 2},
                {"forbidden", 3},
            },
            "default");

        auto check = [&](const std::set<TString>& expected) {
            TPartitionsHelper::CheckGrants(settings, checker);
            UNIT_ASSERT_VALUES_EQUAL(checker.GetResult().Errors, expected);
            const_cast<decltype(checker.GetResult().Errors)*>(&checker.GetResult().Errors)->clear();
        };

        check({});

        req.Args["partition"] = "";
        check({});

        req.Args["partition"] = ",default,,forbidden,   foo, bar  ,";
        check({"no grants for partition 'forbidden'"});

        req.Args["partition"] = "default,unsupported,foo,bar,forbidden";
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TPartitionsHelper::CheckGrants(settings, checker),
            TBlackboxError,
            "Unsupported partition 'unsupported'");
    }

    Y_UNIT_TEST(PartitionsHelperParse) {
        NTest::TRequest req;

        TPartitionsSettings partitionsSettings(
            TPartitionsSettings::TPartitionIdByName{
                {"default", 0},
                {"foo", 1},
                {"bar", 2},
            },
            "default");

        struct TCase {
            TString PartitionArg;
            TPartitionsHelper::TSettings Settings = {};
            std::vector<ui64> ExpectedPartitionIds;
            TString ExpectedError;
        };

        std::vector<TCase> cases({
            TCase{
                .PartitionArg = "",
                .ExpectedPartitionIds = {0},
            },
            TCase{
                .PartitionArg = "default",
                .ExpectedPartitionIds = {0},
            },
            TCase{
                .PartitionArg = "foo",
                .ExpectedPartitionIds = {1},
            },
            TCase{
                .PartitionArg = "bar,foo,default",
                .ExpectedPartitionIds = {2, 0, 1},
            },
            TCase{
                .PartitionArg = ",,foo,   default,foo,,  bar ,",
                .ExpectedPartitionIds = {2, 0, 1},
            },
            TCase{
                .PartitionArg = "foo,unsupported,bar",
                .ExpectedError = "Unsupported partition 'unsupported'",
            },
            TCase{
                .PartitionArg = "foo",
                .Settings = TPartitionsHelper::TSettings{
                    .Method = "test forbid array",
                    .ForbidArray = true,
                },
                .ExpectedPartitionIds = {1},
            },
            TCase{
                .PartitionArg = "foo,default,bar",
                .Settings = TPartitionsHelper::TSettings{
                    .Method = "test forbid array",
                    .ForbidArray = true,
                },
                .ExpectedError = "Array is not allowed in 'partition' param for method=test forbid array",
            },
            TCase{
                .PartitionArg = "default",
                .Settings = TPartitionsHelper::TSettings{
                    .Method = "test forbid non default",
                    .ForbidNonDefault = true,
                },
                .ExpectedPartitionIds = {0},
            },
            TCase{
                .PartitionArg = ",default,,default",
                .Settings = TPartitionsHelper::TSettings{
                    .Method = "test forbid non default",
                    .ForbidNonDefault = true,
                },
                .ExpectedPartitionIds = {0},
            },
            TCase{
                .PartitionArg = "foo,default,bar",
                .Settings = TPartitionsHelper::TSettings{
                    .Method = "test forbid non default",
                    .ForbidNonDefault = true,
                },
                .ExpectedError = "Partitions are not supported for method=test forbid non default",
            },
        });

        for (const TCase& c : cases) {
            req.Args["partition"] = c.PartitionArg;
            if (c.ExpectedError.empty()) {
                UNIT_ASSERT_VALUES_EQUAL_C(
                    c.ExpectedPartitionIds,
                    TPartitionsHelper::ParsePartitionArg(partitionsSettings, req, c.Settings),
                    c.PartitionArg);
            } else {
                UNIT_ASSERT_EXCEPTION_CONTAINS_C(
                    TPartitionsHelper::ParsePartitionArg(partitionsSettings, req, c.Settings),
                    TBlackboxError,
                    c.ExpectedError,
                    c.PartitionArg);
            }
        }
    }
}
