#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/blackbox.h>
#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/methods/check_device_signature.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/signer.h>
#include <passport/infra/daemons/blackbox/src/output/check_device_signature_result.h>

#include <passport/infra/libs/cpp/request/test/request.h>
#include <passport/infra/libs/cpp/utils/crypto/rsa.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspMethodCheclDeviceSignature) {
    static const TString BAD_DEVICE_ID = "bad_device_id";
    static const TString OK_DEVICE_ID = "good_device_id";
    static const TString SIGN_SPACE = "kekeke";

    // openssl genrsa -out rsa.private 2048
    // openssl rsa -in rsa.private -inform PEM -outform PEM -pubout -out rsa.public
    static const TString PRIVATE_KEY = R"(
-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAo6kuQneY+KPoD+KhxO4EDvWZVKXxT3CHcNTJNWYmNfJ4ufGa
iy1xANq3WTRTXqmEXyteWYsW7dcIaYbjjdORllebSsxnsxTlA8SwOKIZfLuy2Es6
eXk0ryTShppB43mLtmWXYcHdphCCAKdvU1VK8yFJ9BXl6e7doGvKJEby6KK73LFS
FeEp5Ah+ySSLwGOTDZbDvX3XNNBUafkm/IVguPI6Xv+LMdqYIZ4SitSEWNMeJieg
nm5Q4y0IXmcsfkgFYR0on5knnM73d9UL1iFkX+iZQjOcLwOIrVnprc+OzZopU4zV
UfX8lIVGVljTFy39NRpqDDbFJCY4BT7CMJIGtQIDAQABAoIBAQCJhbonhBP4qtJv
tsK0lkS2xU3IwYBafnZxK3y/8EwffNZReWGhndedHix/OubrXoqq5ehsWeI0jUK2
WfCQ1r05lfgaDFT/OImQdxCVJjFAjQClm+FQ/bZ1jf2RucGwAgySgh+It70mtCxp
nEiBv+QxFUHtZxFBV8TH7ot3nF61d+Eg+ft6JvtzKlBmLC/DBW2og+8/yQJigQLT
tc8CQ2smmrINvUw3xOJlcB8eyNdyyoZxIL1gFochmHHAkPYYMn0VpS5G2UmwJf1C
KJYrGDlLkrGNqEWQAampG9syaJbC035CMOjmpktE7bdwE8yvwOBNrbpHpBOj99hB
GxpXqxhBAoGBANSfRQLt4VgCKiZPYuMHU+0ZWZLBrWJOIDvYdL1JygnOp+fawCUB
/JMN1tOqkODlzeMLdED8mDroCZFJtih0XSx596olIdAE0YuxCmcipFxCm4UexDYN
wMzuWmWPUy2vzlNdeeJRBm+E0Euiwql7QnknAqvQziDawzcaDbpsM0XpAoGBAMUM
yLGmmRGDFYbgPym+n+kgg7PtcuXFqFOaSBbEK+UUV7BQihDOKEH2qCmLPa/oyfbk
VYi0H9qbr4Dk31KFU5MuocCU7Z6P8wUnF+Iqsi6rzzwruEZEVWKedb92Vi5IHb1V
ke0VVnG0AOYlb17uUHcXk2f3OvVXBzhTcdW1AR7tAoGAKCbLiXoSi+QcwY9MRUMk
/ilDWQtqCp5mR+bgZL6CgMedvZL5pgNyBvMeFmqnYVEESFPpaDxLIxSeeelJlOok
jjmaI29Pq1C9oXuVU1SfZSzO4ZylwO9n8Usxluwx8Pa6J9QHKE3UwUp7a00ZJZg5
PVe4kWmlA19MbiNlZEb6h8ECgYA5BN3+KgsY+il8gd9FIRf8yZ/SrN8Whf+TUY2S
qOlavGwzZNrOyhj0HX/DpbsP7/ihLKljiVu+UlaSoafNpS7t5AyCQZYQz+6uiwVq
OJie0LCC5NPJ4XQjuV6xLHj7o3qXku9K/2WoOUIZVrPHcIZwL65D41J4M9qu/dZu
1JWicQKBgDPSSfvlfIyhxjdhz3t45sf1HoksWhq/kwi0b65AbeAK3VmPezquKhsb
m7ot3K3GR26YeeVNo/uHJ2mlj2UCQkRVA4Sbnw14uqivqoQROarS4IacxEuGzEwQ
rjmvfrwOKR3fmy3hcMqrl0/WcVVkhjobX04i+ToNm0bRkqo5YlLE
-----END RSA PRIVATE KEY-----)";
    static const TString PUBLIC_KEY = R"(
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAo6kuQneY+KPoD+KhxO4E
DvWZVKXxT3CHcNTJNWYmNfJ4ufGaiy1xANq3WTRTXqmEXyteWYsW7dcIaYbjjdOR
llebSsxnsxTlA8SwOKIZfLuy2Es6eXk0ryTShppB43mLtmWXYcHdphCCAKdvU1VK
8yFJ9BXl6e7doGvKJEby6KK73LFSFeEp5Ah+ySSLwGOTDZbDvX3XNNBUafkm/IVg
uPI6Xv+LMdqYIZ4SitSEWNMeJiegnm5Q4y0IXmcsfkgFYR0on5knnM73d9UL1iFk
X+iZQjOcLwOIrVnprc+OzZopU4zVUfX8lIVGVljTFy39NRpqDDbFJCY4BT7CMJIG
tQIDAQAB
-----END PUBLIC KEY-----)"; // stored in db

    Y_UNIT_TEST_F(common, TBlackboxFixture) {
        TConsumer peer;
        NTest::TRequest request;
        const TSigner& signer = Bb->Impl->Signer();
        TCheckDeviceSignatureProcessor proc(*Bb->Impl, request);
        const time_t now = time(nullptr);

        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [](const TBlackboxError& e) { return e.Status() == TBlackboxError::EType::AccessDenied; });
        peer.SetAllow(TBlackboxMethods::CheckDeviceSignature, true);

        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [](const TBlackboxError& e) { return e.Status() == TBlackboxError::EType::AccessDenied; });
        peer.AddCheckDeviceSignatureSignspace(SIGN_SPACE);
        peer.AddCheckDeviceSignatureSignspace("bad_space");
        request.Args["nonce_sign_space"] = "bad_space";

        request.Args["timestamp"] = IntToString<10>(now);
        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [](const TBlackboxError& e) { return e.Status() == TBlackboxError::EType::AccessDenied; });
        peer.SetAllow(TBlackboxFlags::CheckDeviceSignatureWithDebugMode, true);

        auto checkMissingParamAndFillIt = [&](TString arg) {
            UNIT_ASSERT_EXCEPTION_SATISFIES_C(
                proc.Process(peer),
                TBlackboxError,
                [=](const TBlackboxError& e) {
                    UNIT_ASSERT_VALUES_EQUAL_C("Missing " + arg + " argument", e.what(), arg);
                    return e.Status() == TBlackboxError::EType::InvalidParams;
                },
                arg);
            request.Args[arg] = "~~~foobar~~~";
        };

        checkMissingParamAndFillIt("nonce");
        checkMissingParamAndFillIt("device_id");
        checkMissingParamAndFillIt("signature");

        // Common
        std::unique_ptr<TCheckDeviceSignatureResult> result;

        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("SIGNATURE.INVALID", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("invalid base64 in signature", result->Error);
        request.Args["signature"] = "qqqq";

        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("NONCE.INVALID", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("Nonce is invalid", result->Error);

        request.Args["nonce"] = NUtils::CreateStr(now + 100500, ".123.lTRoIAoJ6DHK9_sM.uLX-_X0LhcYsetrgf8g.C7pZdvzB-O01g3BOs2PDrw");
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("NONCE.NO_KEY", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("Nonce is not expired, but key_id==123 was not found", result->Error);

        request.Args["nonce"] = signer.Sign(BAD_DEVICE_ID, SIGN_SPACE, "-777", now);
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("NONCE.EXPIRED", result->Status);
        UNIT_ASSERT_VALUES_EQUAL(NUtils::CreateStr("Nonce expires at ", now - 777, ". now is ", now), result->Error);

        request.Args["timestamp"] = IntToString<10>(now - 778);
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("NONCE.INVALID", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("Nonce is invalid", result->Error);

        request.Args["nonce_sign_space"] = SIGN_SPACE;
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("DEVICE_ID.INVALID", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("Device in nonce: 'bad_device_id' but in request: '~~~foobar~~~'", result->Error);

        request.Args["device_id"] = BAD_DEVICE_ID;
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("PUBLIC_KEY.NOT_FOUND", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("Public key is absent for device_id=bad_device_id", result->Error);

        request.Args["device_id"] = OK_DEVICE_ID;
        request.Args["nonce"] = signer.Sign(OK_DEVICE_ID, SIGN_SPACE, "777", now);
        request.Args.erase("timestamp");
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("SIGNATURE.INVALID", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("wrong signature length", result->Error);

        request.Args["signature"] = NUtils::BinToBase64(
            NUtils::TRsaPrivateEvp::FromPem(PRIVATE_KEY).SignWithSha256(request.Args["nonce"].Value));
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("OK", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("", result->Error);

        request.Args["version"] = "lol";
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            proc.Process(peer),
            TBlackboxError,
            "'public_key' and 'version' should be both empty OR both non-empty");

        request.Args.erase("version");
        request.Args["public_key"] = "lol";

        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [](const TBlackboxError& e) { return e.Status() == TBlackboxError::EType::AccessDenied; });
        peer.SetAllow(TBlackboxFlags::CheckDeviceSignatureWithPublicKey, true);

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            proc.Process(peer),
            TBlackboxError,
            "'public_key' and 'version' should be both empty OR both non-empty");

        request.Args["version"] = "lol";
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("PUBLIC_KEY.UNSUPPORTED_VERSION", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("Supported versions: 1. got: lol", result->Error);

        request.Args["version"] = "1";
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("PUBLIC_KEY.INVALID", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("Public key is invalid: Failed PEM_read_bio_PUBKEY: no start line", result->Error);

        request.Args["public_key"] = PUBLIC_KEY;
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("OK", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("", result->Error);

        // check max age
        request.Args["device_id"] = OK_DEVICE_ID;
        request.Args["nonce"] = signer.Sign(OK_DEVICE_ID, SIGN_SPACE, "-777", now);
        request.Args["signature"] = NUtils::BinToBase64(
            NUtils::TRsaPrivateEvp::FromPem(PRIVATE_KEY).SignWithSha256(request.Args["nonce"].Value));
        request.Args["timestamp"] = IntToString<10>(now);
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("NONCE.EXPIRED", result->Status);
        UNIT_ASSERT_VALUES_EQUAL(NUtils::CreateStr("Nonce expires at ", now - 777, ". now is ", now), result->Error);

        request.Args["max_age"] = "1777";
        UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
        UNIT_ASSERT_VALUES_EQUAL("OK", result->Status);
        UNIT_ASSERT_VALUES_EQUAL("", result->Error);

        request.Args["max_age"] = "1234567890";
        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [=](const TBlackboxError& e) {
                UNIT_ASSERT_VALUES_EQUAL("Arg 'max_age' has too large value: max allowed: 86400: '1234567890'", e.what());
                return e.Status() == TBlackboxError::EType::InvalidParams;
            });
    }
}
