#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/blackbox.h>
#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/methods/create_pwd_hash.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/misc/password_checker.h>
#include <passport/infra/daemons/blackbox/src/output/typed_value_result.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspMethodCreatePwdHash) {
    Y_UNIT_TEST(errors) {
        TBbHolder bb;
        TConsumer peer;
        NTest::TRequest request;
        TCreatePwdHashProcessor proc(*bb.Bb->Impl, request);

        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [](const TBlackboxError& e) { return e.Status() == TBlackboxError::EType::AccessDenied; });

        peer.SetAllow(TBlackboxMethods::CreatePwdHash, true);

        auto checkInvalidParams = [&](const TString& errorMessage) {
            UNIT_ASSERT_EXCEPTION_SATISFIES(
                proc.Process(peer),
                TBlackboxError,
                [=](const TBlackboxError& e) {
                    UNIT_ASSERT_STRINGS_EQUAL(errorMessage, e.what());
                    return e.Status() == TBlackboxError::EType::InvalidParams;
                });
        };

        checkInvalidParams("No password, md5crypt or rawmd5 argument found in request, please specify one.");

        // by rawmd5
        request.Args["rawmd5"] = "qwerty";
        request.Args["uid"] = "someone";
        checkInvalidParams("invalid rawmd5 hash format: qwerty");

        request.Args["rawmd5"] = "abcdefghabcdefghabcdefghabcdefgh";
        checkInvalidParams("invalid uid value: 'someone'");

        request.Args["uid"] = "1";
        request.Args["ver"] = "one";
        checkInvalidParams("invalid ver value: 'one'");

        request.Args["ver"] = "6";
        checkInvalidParams("can not convert rawmd5 hash to version 6");

        // by md5crypt
        request.Args["md5crypt"] = "qwerty";
        request.Args["uid"] = "someone";
        checkInvalidParams("invalid md5crypt hash format: qwerty");

        request.Args["md5crypt"] = "qwertyqwertyqwertyqwerty!";
        checkInvalidParams("invalid md5crypt hash format: qwertyqwertyqwertyqwerty!");

        request.Args["md5crypt"] = "$1$rtyqwertyqwertyqwerty!";
        checkInvalidParams("invalid uid value: 'someone'");

        request.Args["uid"] = "1";
        request.Args["ver"] = "one";
        checkInvalidParams("invalid ver value: 'one'");

        request.Args["ver"] = "4";
        checkInvalidParams("can not convert md5crypt hash to version 4");

        // by password
        request.Args["password"] = std::string(256, 'q');
        checkInvalidParams("too long password.");

        request.Args["password"] = "qwerty";
        request.Args["uid"] = "someone";
        checkInvalidParams("invalid uid value: 'someone'");

        request.Args["uid"] = "1";
        request.Args["ver"] = "one";
        checkInvalidParams("invalid ver value: 'one'");

        // unsupported versions
        request.Args["ver"] = "1";
        checkInvalidParams("not supported hash version: '1'");

        request.Args["ver"] = "3";
        checkInvalidParams("not supported hash version: '3'");

        request.Args["ver"] = "5";
        checkInvalidParams("not supported hash version: '5'");
    }

    Y_UNIT_TEST_F(common, TBlackboxFixture) {
        TPasswordChecker pc;
        TConsumer peer;
        NTest::TRequest request;
        TCreatePwdHashProcessor proc(*Bb->Impl, request);
        peer.SetAllow(TBlackboxMethods::CreatePwdHash, true);
        std::unique_ptr<TTypedValueResult> result;
        request.Args["uid"] = "100500";

        // TODO configure argon to make some hashes here
        // by rawmd5
        request.Args["rawmd5"] = "d8578edf8458ce06fbc5bb76a58c5ca4";
        request.Args["ver"] = "7";
        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [=](const TBlackboxError& e) {
                UNIT_ASSERT_STRINGS_EQUAL("Argon is not configured", e.what());
                return e.Status() == TBlackboxError::EType::Unknown;
            });

        // by md5crypt
        request.Args["md5crypt"] = "$1$.YTetdyy$glvDCsyAUqOJgLgGvLp0A0";
        request.Args["ver"] = "6";
        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [=](const TBlackboxError& e) {
                UNIT_ASSERT_STRINGS_EQUAL("Argon is not configured", e.what());
                return e.Status() == TBlackboxError::EType::Unknown;
            });
    }
}
