#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/blackbox.h>
#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/methods/family_info.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/output/family_info_result.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

/*
sqlite> CREATE TABLE `family_info` (
  `family_id` INTEGER PRIMARY KEY,
  `admin_uid` INTEGER NOT NULL,
  `meta` varbinary(128) NOT NULL,
  UNIQUE (`admin_uid`)
);

sqlite> CREATE TABLE `family_members` (
  `family_id` INTEGER,
  `uid` INTEGER PRIMARY KEY,
  `place` INTEGER
);

sqlite> select * from family_info;
1|42|
2|100501|
sqlite> select * from family_members;
1|42|1
1|43|2
1|44|4
2|100500|1
2|100501|3
*/

Y_UNIT_TEST_SUITE(PasspMethodFamilyInfo) {
    Y_UNIT_TEST(common) {
        TBbHolder bb;
        TConsumer peer;
        NTest::TRequest request;
        TFamilyInfoProcessor proc(*bb.Bb->Impl, request);

        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [](const TBlackboxError& e) { return e.Status() == TBlackboxError::EType::AccessDenied; });
        peer.SetAllow(TBlackboxMethods::FamilyInfo, true);

        // invalid args
        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [=](const TBlackboxError& e) {
                UNIT_ASSERT_VALUES_EQUAL("Missing family_id argument", e.what());
                return e.Status() == TBlackboxError::EType::InvalidParams;
            });

        request.Args["family_id"] = "146";
        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [=](const TBlackboxError& e) {
                UNIT_ASSERT_VALUES_EQUAL("family_id does not look like valid value: '146'", e.what());
                return e.Status() == TBlackboxError::EType::InvalidParams;
            });

        // Common
        auto check = [&](const TString& familyId,
                         const TString& adminUid,
                         const TString& status,
                         const std::vector<TFamilyInfoResult::TUser>& users) {
            request.Args["family_id"] = familyId;
            std::unique_ptr<TFamilyInfoResult> result;
            UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
            UNIT_ASSERT_VALUES_EQUAL_C(status, result->Status.Value, familyId);
            UNIT_ASSERT_VALUES_EQUAL_C(adminUid, result->AdminUid, familyId);
            UNIT_ASSERT_VALUES_EQUAL_C(users, result->Users, familyId);
        };

        check("f1", "42", "OK", {
                                    {.Uid = "42"},
                                    {.Uid = "43"},
                                    {.Uid = "44"},
                                });
        check("f2", "100501", "OK", {
                                        {.Uid = "100500"},
                                        {.Uid = "100501"},
                                    });
        check("f3", "", "MISSING_FAMILY", {});

        request.Args["get_place"] = "yes";
        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [=](const TBlackboxError& e) {
                UNIT_ASSERT_VALUES_EQUAL("no grants for get_place=yes. consumer:  (tvm_id=NULL;IP=)", e.what());
                return e.Status() == TBlackboxError::EType::AccessDenied;
            });
        peer.SetAllow(TBlackboxFlags::FamilyInfoPlace, true);

        check("f1", "42", "OK", {
                                    {.Uid = "42", .Place = "1"},
                                    {.Uid = "43", .Place = "2"},
                                    {.Uid = "44", .Place = "4"},
                                });
        check("f2", "100501", "OK", {
                                        {.Uid = "100500", .Place = "1"},
                                        {.Uid = "100501", .Place = "3"},
                                    });
        check("f3", "", "MISSING_FAMILY", {});
    }

    Y_UNIT_TEST(toInternalFamilyId) {
        auto assertError = [](TString err) {
            UNIT_ASSERT_EXCEPTION_SATISFIES(
                TFamilyInfoProcessor::ToInternalFamilyId(err),
                TBlackboxError,
                [=](const TBlackboxError& e) {
                    UNIT_ASSERT_VALUES_EQUAL("family_id does not look like valid value: '" + err + "'", e.what());
                    return e.Status() == TBlackboxError::EType::InvalidParams;
                });
        };

        assertError("");
        assertError("100500");
        assertError("0100500");
        assertError("f0100500");
        assertError("f0");

        UNIT_ASSERT_VALUES_EQUAL("100500", TFamilyInfoProcessor::ToInternalFamilyId("f100500"));
        UNIT_ASSERT_VALUES_EQUAL("1", TFamilyInfoProcessor::ToInternalFamilyId("f1"));
    }

    Y_UNIT_TEST(fromInternalFamilyId) {
        UNIT_ASSERT_VALUES_EQUAL("f", TFamilyInfoProcessor::FromInternalFamilyId(""));
        UNIT_ASSERT_VALUES_EQUAL("f100500", TFamilyInfoProcessor::FromInternalFamilyId("100500"));
        UNIT_ASSERT_VALUES_EQUAL("f0100500", TFamilyInfoProcessor::FromInternalFamilyId("0100500"));
        UNIT_ASSERT_VALUES_EQUAL("f0", TFamilyInfoProcessor::FromInternalFamilyId("0"));
        UNIT_ASSERT_VALUES_EQUAL("ff0", TFamilyInfoProcessor::FromInternalFamilyId("f0"));
    }
}

template <>
void Out<TFamilyInfoResult::TUser>(IOutputStream& o, const TFamilyInfoResult::TUser& value) {
    o << "uid=" << value.Uid << ",";
    o << "place=" << (value.Place ? *value.Place : "(NULL)") << ";";
}

template <>
void Out<std::vector<TFamilyInfoResult::TUser>>(IOutputStream& o, const std::vector<TFamilyInfoResult::TUser>& value) {
    for (const TFamilyInfoResult::TUser& u : value) {
        o << u << Endl;
    }
}
