#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/blackbox.h>
#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/methods/get_device_public_key.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/output/get_device_public_key_result.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(PasspMethodGetDevicePublicKey) {
    static const TString MISSING_DEVICE_ID = "missing_device_id";
    static const TString BAD_VERSION = "bad_version";
    static const TString BAD_PUBLIC_KEY = "bad_public_key";
    static const TString OK_DEVICE_ID = "good_device_id";

    Y_UNIT_TEST_F(common, TBlackboxFixture) {
        TConsumer peer;
        NTest::TRequest request;
        TGetDevicePublicKeyProcessor proc(*Bb->Impl, request);

        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [](const TBlackboxError& e) { return e.Status() == TBlackboxError::EType::AccessDenied; });
        peer.SetAllow(TBlackboxMethods::GetDevicePublicKey, true);

        UNIT_ASSERT_EXCEPTION_SATISFIES(
            proc.Process(peer),
            TBlackboxError,
            [=](const TBlackboxError& e) {
                UNIT_ASSERT_VALUES_EQUAL("Missing device_id argument", e.what());
                return e.Status() == TBlackboxError::EType::InvalidParams;
            });

        // Common
        auto check = [&](const TString& deviceId,
                         const TString& status,
                         const TString& value,
                         const TString& version,
                         const TString& ownerId,
                         const TString& error) {
            request.Args["device_id"] = deviceId;
            std::unique_ptr<TGetDevicePublicKeyResult> result;
            UNIT_ASSERT_NO_EXCEPTION(result = proc.Process(peer));
            UNIT_ASSERT_VALUES_EQUAL_C(status, result->Status, deviceId);
            UNIT_ASSERT_VALUES_EQUAL_C(value, result->Value, deviceId);
            UNIT_ASSERT_VALUES_EQUAL_C(version, result->Version, deviceId);
            UNIT_ASSERT_VALUES_EQUAL_C(ownerId, result->OwnerId, deviceId);
            UNIT_ASSERT_VALUES_EQUAL_C(error, result->Error, deviceId);
        };

        check(MISSING_DEVICE_ID,
              "PUBLIC_KEY.NOT_FOUND",
              "",
              "",
              "",
              "Public key is absent for device_id=missing_device_id");

        check(BAD_VERSION,
              "PUBLIC_KEY.UNSUPPORTED_VERSION",
              "3333",
              "17",
              "2",
              "Supported versions: 1. got: 17");

        check(BAD_PUBLIC_KEY,
              "PUBLIC_KEY.INVALID",
              "3333",
              "1",
              "2",
              "Public key is invalid: Failed PEM_read_bio_PUBKEY: no start line");

        check(OK_DEVICE_ID,
              "OK",
              "-----BEGIN PUBLIC KEY-----\nMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAo6kuQneY+KPoD+KhxO4E\nDvWZVKXxT3CHcNTJNWYmNfJ4ufGaiy1xANq3WTRTXqmEXyteWYsW7dcIaYbjjdOR\nllebSsxnsxTlA8SwOKIZfLuy2Es6eXk0ryTShppB43mLtmWXYcHdphCCAKdvU1VK\n8yFJ9BXl6e7doGvKJEby6KK73LFSFeEp5Ah+ySSLwGOTDZbDvX3XNNBUafkm/IVg\nuPI6Xv+LMdqYIZ4SitSEWNMeJiegnm5Q4y0IXmcsfkgFYR0on5knnM73d9UL1iFk\nX+iZQjOcLwOIrVnprc+OzZopU4zVUfX8lIVGVljTFy39NRpqDDbFJCY4BT7CMJIG\ntQIDAQAB\n-----END PUBLIC KEY-----",
              "1",
              "2",
              "");
    }
}
