#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/blackbox.h>
#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/methods/get_oauth_tokens.h>
#include <passport/infra/daemons/blackbox/src/misc/exception.h>
#include <passport/infra/daemons/blackbox/src/output/get_oauth_tokens_result.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NBb;

static std::vector<TOAuthChunk> CollectVectorFromArgs(std::vector<TOAuthChunk> vec) {
    return vec;
}

static std::vector<TOAuthChunk> CollectVectorFromArgs(std::vector<TOAuthChunk> vec,
                                                      TOAuthChunk&& chunk) {
    vec.push_back(std::move(chunk));
    return vec;
}

template <class... T>
static std::vector<TOAuthChunk> CollectVectorFromArgs(std::vector<TOAuthChunk> vec,
                                                      TOAuthChunk&& chunk,
                                                      T&&... arg) {
    vec.push_back(std::move(chunk));
    return CollectVectorFromArgs(std::move(vec), std::move(arg)...);
}

template <class... T>
static void checkResult(TString location,
                        std::unique_ptr<TGetOAuthTokensResult> actualResult,
                        T&&... arg) {
    const std::vector<TOAuthChunk> expectedResult = CollectVectorFromArgs(
        std::vector<TOAuthChunk>{},
        std::move(arg)...);

    UNIT_ASSERT_VALUES_EQUAL_C(actualResult->Tokens.size(), expectedResult.size(), location);

    for (size_t idx = 0; idx < expectedResult.size(); ++idx) {
        TString checkId = TStringBuilder() << location << "; token=" << idx;

        const TOAuthChunk& actual = actualResult->Tokens[idx];
        const TOAuthChunk& expected = expectedResult[idx];
        UNIT_ASSERT_C(actual.TokenInfo, checkId);
        // TODO
        checkId += "; token_id=" + actual.TokenInfo->TokenId;
        UNIT_ASSERT_C(expected.TokenInfo, checkId);

        UNIT_ASSERT_VALUES_EQUAL_C(actual.Status, expected.Status, checkId);
        UNIT_ASSERT_VALUES_EQUAL_C(actual.Comment, expected.Comment, checkId);
        UNIT_ASSERT_VALUES_EQUAL_C(actual.LoginId, expected.LoginId, checkId);
        UNIT_ASSERT_VALUES_EQUAL_C(actual.TokenInfo->TokenId, expected.TokenInfo->TokenId, checkId);
        UNIT_ASSERT_VALUES_EQUAL_C((bool)actual.TokenAttrs, (bool)expected.TokenAttrs, checkId);
        if (expected.TokenAttrs) {
            UNIT_ASSERT_VALUES_EQUAL_C(actual.TokenAttrs->Attrs, expected.TokenAttrs->Attrs, checkId);
        }
        UNIT_ASSERT_VALUES_EQUAL_C((bool)actual.ClientAttrs, (bool)expected.ClientAttrs, checkId);
        if (actual.ClientAttrs) {
            UNIT_ASSERT_VALUES_EQUAL_C(actual.ClientAttrs->Attrs, expected.ClientAttrs->Attrs, checkId);
        }
    }
}

static std::unique_ptr<TOAuthTokenInfo> CreateTokenInfo(const TString& tokenId) {
    auto res = std::make_unique<TOAuthTokenInfo>();
    res->TokenId = tokenId;
    return res;
}

using TOptionalAttrs = std::optional<TAttributesChunk::TAttributesType>;

static TOptionalAttrs CreateAttrs(TAttributesChunk::TAttributesType val) {
    return val;
}

static TOptionalAttrs CreateAttrs() {
    return {};
}

static TConsumer CreateConsumer() {
    TConsumer res;

    res.SetAllow(TBlackboxMethods::GetOAuthTokens, true);
    res.SetAllow(TBlackboxFlags::FullInfo, true);
    res.SetAllow(TBlackboxFlags::OAuthAttributes, true);

    return res;
}

static void AddAttrs(NTest::TRequest& request) {
    request.Args["oauth_token_attributes"] = "4,5";
    request.Args["oauth_client_attributes"] = "13";
}

#define BB_CHECK(ACTUAL, ...) \
    checkResult(TStringBuilder() << __SOURCE_FILE__ << ":" << __LINE__, ACTUAL, __VA_ARGS__)

#define BB_EMPTY(ACTUAL) \
    checkResult(TStringBuilder() << __SOURCE_FILE__ << ":" << __LINE__, ACTUAL)

Y_UNIT_TEST_SUITE_F(PasspMethodGetOAuthTokens, TBlackboxFixture) {
    static TOAuthChunk CreateChunk(TString tokenId,
                                   TOAuthError error,
                                   TOptionalAttrs tokenAttrs,
                                   TOptionalAttrs clientAttrs,
                                   TString loginId = {}) {
        return TOAuthChunk{
            .TokenInfo = CreateTokenInfo(tokenId),
            .TokenAttrs = tokenAttrs
                              ? std::make_unique<TAttributesChunk>(std::move(*tokenAttrs))
                              : nullptr,
            .ClientAttrs = clientAttrs
                               ? std::make_unique<TAttributesChunk>(std::move(*clientAttrs))
                               : nullptr,
            .Status = error.ConvertToStatus(),
            .Comment = error.Msg(),
            .LoginId = loginId,
        };
    }

    TOAuthChunk Chunk700101(bool withAttrs = false, bool withLoginId = false) {
        return CreateChunk(
            "700101",
            TOAuthError::Ok,
            withAttrs ? CreateAttrs({
                            {"4", "1001"},
                            {"5", "qqqqqqqqqqqqqqqq"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001001"}})
                      : CreateAttrs(),
            withLoginId ? "t:700101"
                        : "");
    }

    TOAuthChunk Chunk700102(bool withAttrs = false) {
        return CreateChunk(
            "700102",
            TOAuthError::TokenExpired,
            withAttrs ? CreateAttrs({
                            {"4", "1002"},
                            {"5", "qqqqqqqqqqqqqqqq"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001002"}})
                      : CreateAttrs());
    }

    TOAuthChunk Chunk700103(bool withAttrs = false, bool withLoginId = false) {
        return CreateChunk(
            "700103",
            TOAuthError::Ok,
            withAttrs ? CreateAttrs({
                            {"4", "1003"},
                            {"5", "qqqqqqqqqqqqqqqq"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001003"}})
                      : CreateAttrs(),
            withLoginId ? "s:1234567890:123:my_custom_login_id"
                        : "");
    }

    TOAuthChunk Chunk700104(bool withAttrs = false) {
        return CreateChunk(
            "700104",
            TOAuthError::ClientNotFound,
            withAttrs ? CreateAttrs({
                            {"4", "1004"},
                            {"5", "qqqqqqqqqqqqqqqq"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001004"}})
                      : CreateAttrs());
    }

    TOAuthChunk Chunk700105(bool withAttrs = false, bool withLoginId = false) {
        return CreateChunk(
            "700105",
            TOAuthError::Ok,
            withAttrs ? CreateAttrs({
                            {"4", "1005"},
                            {"5", "wwwwwwwwwwwwwwww"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001005"}})
                      : CreateAttrs(),
            withLoginId ? "t:700110"
                        : "");
    }

    TOAuthChunk Chunk700106(bool withAttrs = false) {
        return CreateChunk(
            "700106",
            TOAuthError::ClientBlocked,
            withAttrs ? CreateAttrs({
                            {"4", "1006"},
                            {"5", "ssssssssssssssss"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001006"}})
                      : CreateAttrs());
    }

    TOAuthChunk Chunk700107(bool withAttrs = false) {
        return CreateChunk(
            "700107",
            TOAuthError::ClientNotFound,
            withAttrs ? CreateAttrs({
                            {"4", "1007"},
                            {"5", "aaaaaaaaaaaaaaaa"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs(TAttributesChunk::TAttributesType{})
                      : CreateAttrs());
    }

    TOAuthChunk Chunk700109(bool withAttrs = false) {
        return CreateChunk(
            "700109",
            TOAuthError::TokenExpired,
            withAttrs ? CreateAttrs({
                            {"4", "1001"},
                            {"5", "wwwwwwwwwwwwwwww"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001001"}})
                      : CreateAttrs());
    }

    TOAuthChunk Chunk700110(bool withAttrs = false, bool withLoginId = false) {
        return CreateChunk(
            "700110",
            TOAuthError::Ok,
            withAttrs ? CreateAttrs({
                            {"4", "1001"},
                            {"5", "qqqqqqqqqqqqqqqq"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001001"}})
                      : CreateAttrs(),
            withLoginId ? "t:700110"
                        : "");
    }

    TOAuthChunk Chunk700111(bool withAttrs = false) {
        return CreateChunk(
            "700111",
            TOAuthError::TokenExpired,
            withAttrs ? CreateAttrs({
                            {"5", "zzzzzzzzzzzzzzzz"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs(TAttributesChunk::TAttributesType{})
                      : CreateAttrs());
    }

    TOAuthChunk Chunk700201(bool withAttrs = false, bool withLoginId = false) {
        return CreateChunk(
            "700201",
            TOAuthError::GLogout,
            withAttrs ? CreateAttrs({
                            {"4", "1001"},
                            {"5", "qqqqqqqqqqqqqqqq"},
                        })
                      : CreateAttrs(),
            withAttrs ? CreateAttrs({{"13", "1610001001"}})
                      : CreateAttrs(),
            withLoginId ? "t:700201"
                        : "");
    }

    Y_UNIT_TEST(procMissingUser) {

        const TConsumer consumer = CreateConsumer();
        NTest::TRequest request;
        TGetOAuthTokensProcessor proc(*Bb->Impl, request);

        request.Args["uid"] = "40000";
        request.Args["full_info"] = "yes";
        BB_EMPTY(proc.Process(consumer));
    }

    Y_UNIT_TEST(procCommonUser) {
#define BB_LOCAL_CHECK(ACTUAL, ...) \
    check(TStringBuilder() << __SOURCE_FILE__ << ":" << __LINE__, ACTUAL, __VA_ARGS__)

#define BB_LOCAL_EMPTY(ACTUAL) \
    check(TStringBuilder() << __SOURCE_FILE__ << ":" << __LINE__, ACTUAL)

        const TConsumer consumer = CreateConsumer();
        NTest::TRequest request;
        TGetOAuthTokensProcessor proc(*Bb->Impl, request);

        using TMapType = std::map<TString, TString>;
        bool withAttrs = false;
        auto check = [&](TString location, const std::map<TString, TString>& args, auto&&... chunks) {
            request.Args.clear();
            for (const auto& [k, v] : args) {
                request.Args[k] = v;
            }
            request.Args["uid"] = "40001";
            if (withAttrs) {
                AddAttrs(request);
            }

            checkResult(location, proc.Process(consumer), std::move(chunks)...);
        };

        for (bool wa : {false, true}) {
            withAttrs = wa;

            BB_LOCAL_CHECK({},
                           Chunk700101(withAttrs),
                           Chunk700103(withAttrs),
                           Chunk700105(withAttrs),
                           Chunk700110(withAttrs));
            BB_LOCAL_CHECK({
                               {"xtoken_only", "true"},
                           }, Chunk700110(withAttrs));
            BB_LOCAL_CHECK({
                               {"xtoken_only", "true"},
                               {"client_id", "1001"},
                           }, Chunk700110(withAttrs));
            BB_LOCAL_CHECK(TMapType({
                               {"client_id", "1001"},
                           }),
                           Chunk700101(withAttrs),
                           Chunk700110(withAttrs));

            BB_LOCAL_CHECK(TMapType({
                               {"full_info", "true"},
                           }),
                           Chunk700101(withAttrs),
                           Chunk700102(withAttrs),
                           Chunk700103(withAttrs),
                           Chunk700104(withAttrs),
                           Chunk700105(withAttrs),
                           Chunk700106(withAttrs),
                           Chunk700107(withAttrs),
                           Chunk700109(withAttrs),
                           Chunk700110(withAttrs),
                           Chunk700111(withAttrs));
            BB_LOCAL_CHECK(TMapType({
                               {"full_info", "true"},
                               {"client_id", "1002"},
                           }),
                           Chunk700102(withAttrs));

            BB_LOCAL_CHECK(TMapType({
                               {"client_id", "1001"},
                               {"full_info", "true"},
                           }),
                           Chunk700101(withAttrs),
                           Chunk700109(withAttrs),
                           Chunk700110(withAttrs),
                           Chunk700111(withAttrs));

            BB_LOCAL_EMPTY(TMapType({
                {"xtoken_only", "true"},
                {"client_id", "1002"},
            }));
        }

#undef BB_LOCAL_EMPTY
#undef BB_LOCAL_CHECK
    }

    Y_UNIT_TEST(procGLogoutedUser) {

        const TConsumer consumer = CreateConsumer();
        NTest::TRequest request;
        TGetOAuthTokensProcessor proc(*Bb->Impl, request);

        request.Args["uid"] = "40002";
        BB_EMPTY(proc.Process(consumer));

        request.Args["full_info"] = "yes";
        BB_CHECK(proc.Process(consumer),
                 Chunk700201());

        AddAttrs(request);
        BB_CHECK(proc.Process(consumer),
                 Chunk700201(true));

    }

    Y_UNIT_TEST(procWithLoginId) {

        const TConsumer consumer = CreateConsumer();
        NTest::TRequest request;
        TGetOAuthTokensProcessor proc(*Bb->Impl, request);

        // good user
        request.Args["uid"] = "40001";
        request.Args["get_login_id"] = "true";
        BB_CHECK(proc.Process(consumer),
                 Chunk700101(false, true),
                 Chunk700103(false, true),
                 Chunk700105(false, true),
                 Chunk700110(false, true));

        AddAttrs(request);
        BB_CHECK(proc.Process(consumer),
                 Chunk700101(true, true),
                 Chunk700103(true, true),
                 Chunk700105(true, true),
                 Chunk700110(true, true));

        request.Args.clear();

        // glogouted user
        request.Args["uid"] = "40002";
        request.Args["get_login_id"] = "true";
        BB_EMPTY(proc.Process(consumer));

        request.Args["full_info"] = "yes";
        BB_CHECK(proc.Process(consumer),
                 Chunk700201(false, true));

        AddAttrs(request);
        BB_CHECK(proc.Process(consumer),
                 Chunk700201(true, true));
    }

    Y_UNIT_TEST(exceptions) {

        TConsumer consumer;
        NTest::TRequest request;
        TGetOAuthTokensProcessor proc(*Bb->Impl, request);

        auto except = [&](TBlackboxError::EType type, const TString& err) {
            UNIT_ASSERT_EXCEPTION_SATISFIES(
                proc.Process(consumer),
                TBlackboxError,
                [=](const TBlackboxError& e) {
                    UNIT_ASSERT_STRING_CONTAINS(e.what(), err);
                    return e.Status() == type;
                });
        };

        // access denied
        except(TBlackboxError::EType::AccessDenied, "no grants for method=");
        consumer.SetAllow(TBlackboxMethods::GetOAuthTokens, true);

        request.Args["oauth_token_attributes"] = "asd";
        except(TBlackboxError::EType::AccessDenied, "no grants for oauth attributes");
        consumer.SetAllow(TBlackboxFlags::OAuthAttributes, true);

        request.Args["aliases"] = "all_with_hidden";
        except(TBlackboxError::EType::AccessDenied, "no grants to use aliases=all_with_hidden");
        consumer.SetAllow(TBlackboxFlags::GetHiddenAliases, true);

        request.Args["full_info"] = "true";
        except(TBlackboxError::EType::AccessDenied, "no grants for arg 'full_info'");
        consumer.SetAllow(TBlackboxFlags::FullInfo, true);

        // invalid params
        except(TBlackboxError::EType::InvalidParams, "Missing uid argument");
        request.Args["uid"] = "select * from";

        except(TBlackboxError::EType::InvalidParams, "invalid uid value:");
        request.Args["uid"] = "99999999999"; // no such uid in db

        except(TBlackboxError::EType::InvalidParams, "invalid oauth_token_attributes value");
        request.Args["oauth_token_attributes"] = "42";

        UNIT_ASSERT_NO_EXCEPTION(proc.Process(consumer));

        request.Args["client_id"] = "qwerqwr";
        except(TBlackboxError::EType::InvalidParams, "invalid client_id value:");

        request.Args["client_id"] = "465";
        UNIT_ASSERT_NO_EXCEPTION(proc.Process(consumer));
    }
}

template <>
void Out<TOAuthStatus>(IOutputStream& out, const TOAuthStatus& value) {
    out << value.Status();
}
