#include "util/generic/yexception.h"

#include <passport/infra/daemons/blackbox/src/blackbox_impl.h>
#include <passport/infra/daemons/blackbox/src/ip/ipacl.h>
#include <passport/infra/daemons/blackbox/src/ip/usernets_list.h>

#include <passport/infra/libs/cpp/request/test/request.h>

#include <library/cpp/testing/unittest/registar.h>

#include <stdexcept>

using namespace NPassport::NBb;
using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(PasspBbAclNewStyle) {

    Y_UNIT_TEST(wrongRate) {
        // work rate can't be > 100
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TUsernetsList(R"({
            "internal_explicit": [
                {
                    "ip": "2a02:6b8::/32",
                    "macro_name": "_YANDEXNETS_",
                    "work_rate": 101
                }
            ]
        })"),
            yexception,
            "Usernets: work rate is out of range: 101; json member 'internal_explicit'");

        // work rate can't be < 0
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TUsernetsList(R"({
            "internal_explicit": [
                {
                    "ip": "2a02:6b8::/32",
                    "macro_name": "_YANDEXNETS_",
                    "work_rate": -1
                }
            ]
        })"),
            yexception,
            "Usernets: broken usernets json member 'internal_explicit' expected int in 'work_rate'");

        // work_rate may be skipped
        std::unique_ptr<TUsernetsList> ul;
        UNIT_ASSERT_NO_EXCEPTION(
            ul = std::make_unique<TUsernetsList>(R"({
            "allow": [
                {
                    "ip": "2a02:6b8::/32",
                    "macro_name": "_YANDEXNETS_"
                }
            ],
            "internal_explicit": [],
            "robot_explicit": [],
            "deny": [],
            "robots": []
        })"));
        UNIT_ASSERT_VALUES_EQUAL(
            ENetworkKind::Internal,
            ul->FindNetwork(TIpAddr("2a02:6b8::1"), NPassport::NTest::TRequest{}));
    };

    Y_UNIT_TEST(networkKindTests) {

        TUsernetsList nets(R"({
            "internal_explicit": [
                {
                    "ip": "2a04:6b8:b111:8b00::2",
                    "macro_name": "_GUESTSNETS_"
                }
            ],
            "robot_explicit": [
                {
                    "ip": "2a04:6b8:b111:8b00::3",
                    "macro_name": "_GUESTSNETS_"
                }
            ],
            "allow": [
                {
                    "ip": "2a02:6b8::/32",
                    "macro_name": "_YANDEXNETS_",
                    "work_rate": 100
                },
                {
                    "ip": "2a02:6b9:0:1619::129",
                    "macro_name": "_CLOUD_YANDEX_CLIENTS_",
                    "work_rate": 100
                },
                {
                    "ip": "77.88.0.0/16",
                    "macro_name": "_SDCAMINEVKANETS_",
                    "work_rate": 100
                },
                {
                    "ip": "213.87.14.162",
                    "macro_name": "213.87.14.162",
                    "work_rate": 100
                }
            ],
            "deny": [
                {
                    "ip": "2a04:6b8:b111:8b00::/56",
                    "macro_name": "_TRBOSRVNETS_",
                    "work_rate": 100
                },
                {
                    "ip": "2a02:6b8:0:1619::129",
                    "macro_name": "racktables:guestsnets",
                    "work_rate": 100
                },
                {
                    "ip": "77.88.7.0/24",
                    "macro_name": "racktables:guestsnets",
                    "work_rate": 100
                },
                {
                    "ip": "77.88.193.89",
                    "macro_name": "racktables:guestsnets",
                    "work_rate": 100
                }
            ],
            "robots": [
                {
                    "ip": "2a02:6b8:a111:8b00::/56",
                    "macro_name": "_WIKI_SEARCH_NETS_",
                    "work_rate": 100
                },
                {
                    "ip": "2a02:6b8:a111:8b00::1",
                    "macro_name": "_NIRVANANETS_",
                    "work_rate": 100
                },
                {
                    "ip": "77.88.9.0/24",
                    "macro_name": "_SELENIUMGRIDNETS_",
                    "work_rate": 100
                },
                {
                    "ip": "77.88.60.90",
                    "macro_name": "_SELENIUMGRIDNETS_",
                    "work_rate": 100
                }
            ]
        })");

        const NPassport::NTest::TRequest req;
        // Allow:

        // IPv4:
        // single ip
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("213.87.14.162"), req));
        // mask
        // lower bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("77.88.0.0"), req));
        // middle
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("77.88.59.88"), req));
        // upper bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("77.88.63.255"), req));

        // IPv6:
        // single ip
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("2a02:6b9:0:1619::129"), req));
        // mask
        // lower bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("2a02:6b8::0"), req));
        // middle
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("2a02:6b8:0:1619::128"), req));
        // upper bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("2a02:6b8:ffff:ffff:ffff:ffff:ffff:ffff"), req));
        // explicit
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("2a04:6b8:b111:8b00::2"), req));

        // Deny:

        // IPv4:
        // single ip
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("77.88.193.89"), req));
        // mask
        // lower bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("77.88.7.0"), req));
        // middle
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("77.88.7.88"), req));
        // upper bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("77.88.7.255"), req));

        // IPv6:
        // single ip
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("2a02:6b8:0:1619::129"), req));
        // mask
        // lower bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("2a04:6b8:b111:8b00::"), req));
        // middle
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("2a04:6b8:b111:8b00::1"), req));
        // upper bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("2a04:6b8:b111:8bff:ffff:ffff:ffff:ffff"), req));

        // Robots:

        // IPv4:
        // single ip
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("77.88.60.90"), req));
        // mask
        // lower bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("77.88.9.0"), req));
        // middle
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("77.88.9.3"), req));
        // upper bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("77.88.9.255"), req));

        // IPv6:
        // single ip
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("2a02:6b8:a111:8b00::1"), req));
        // mask
        // lower bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("2a02:6b8:a111:8b00::"), req));
        // middle
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("2a02:6b8:a111:8b00::1"), req));
        // upper bound
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("2a02:6b8:a111:8bff:ffff:ffff:ffff:ffff"), req));
        // explicit
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Robot, nets.FindNetwork(TString("2a04:6b8:b111:8b00::3"), req));
    }

    Y_UNIT_TEST(turnedOffNetworkTests) {
        TUsernetsList nets(R"(
            {
                "internal_explicit": [],
                "robot_explicit": [],
                "allow": [
                    {
                        "ip": "77.88.60.90",
                        "macro_name": "_YANDEXNETS_",
                        "work_rate": 0
                    },
                    {
                        "ip": "2a02:6b9:0:1619::129",
                        "macro_name": "_CLOUD_YANDEX_CLIENTS_",
                        "work_rate": 100
                    },
                    {
                        "ip": "213.87.14.162",
                        "macro_name": "213.87.14.162",
                        "work_rate": 100
                    }
                ],
                "deny": [
                    {
                        "ip": "2a02:6b9:0:1619::129",
                        "macro_name": "_TRBOSRVNETS_",
                        "work_rate": 0
                    }
                ],
                "robots": [
                    {
                        "ip": "213.87.14.162",
                        "macro_name": "_WIKI_SEARCH_NETS_",
                        "work_rate": 0
                    }
                ]
            }
        )");

        const NPassport::NTest::TRequest req;

        // Turned off 'allowed'
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::External, nets.FindNetwork(TString("77.88.60.90"), req));

        // Turned off 'deny'
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("2a02:6b9:0:1619::129"), req));

        // Turned off 'robot'
        UNIT_ASSERT_VALUES_EQUAL(ENetworkKind::Internal, nets.FindNetwork(TString("213.87.14.162"), req));
    }

}
