#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/helpers/display_name_helper.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>
#include <passport/infra/daemons/blackbox/src/misc/dbfields_converter.h>
#include <passport/infra/daemons/blackbox/src/misc/hosts_list.h>
#include <passport/infra/daemons/blackbox/src/misc/shards_map.h>
#include <passport/infra/daemons/blackbox/src/misc/strings.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/testing/unittest/registar.h>

#include <map>

using namespace NPassport;
using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(TPasspBbFetcherTestSuite) {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    std::unique_ptr<TTestDbHolder> DB;

    TTestDbHolder& Db() {
        if (!DB) {
            DB = std::make_unique<TTestDbHolder>();
        }
        return *DB;
    }

    Y_UNIT_TEST(Test) {
        TLog::Error("Starting DbFetcher::test");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher);

        fetcher->AddAlias(TAlias::PORTAL_LOGIN);
        fetcher->AddAlias(TAlias::MAIL_LOGIN);
        fetcher->AddAlias(TAlias::PDD_MASTER_LOGIN);
        fetcher->AddSuid2();

        fetcher->AddAttr(TAttr::ACCOUNT_DISPLAY_NAME);
        TDbFieldsResult* firstname = converter->Add("userinfo.firstname.uid");

        TString uids[3] = {"543", "544", "545"};
        std::vector<TString> uidsSet(uids, uids + 3);
        fetcher->FetchByUids(uidsSet);

        const TDbProfile* profile = nullptr;
        UNIT_ASSERT((profile = fetcher->NextProfile()) != nullptr);
        UNIT_ASSERT_VALUES_EQUAL("Name", firstname->Value(profile));
        UNIT_ASSERT((profile = fetcher->NextProfile()) != nullptr);
        UNIT_ASSERT_VALUES_EQUAL("Name2", firstname->Value(profile));
        UNIT_ASSERT((profile = fetcher->NextProfile()) != nullptr);
        UNIT_ASSERT_VALUES_EQUAL("Name3", firstname->Value(profile));
        UNIT_ASSERT((fetcher->NextProfile()) == nullptr);
    }

    static void AddValue(TDbFieldsConverter& conv,
                         std::map<TString, TString>& expected,
                         std::map<TString, TDbFieldsResult*>& result,
                         const TString& name,
                         const TString& value) {
        expected.insert(std::make_pair(name, value));
        TDbFieldsResult* res = conv.Add(name);
        Y_ENSURE(res, "Cannot add dbfield");
        result.insert(std::make_pair(name, res));
    }

    static void FillMaliceFields(TDbFieldsConverter& conv,
                                 std::map<TString, TString>& expected,
                                 std::map<TString, TDbFieldsResult*>& result) {
        AddValue(conv, expected, result, "accounts.login.uid", "malice");
        AddValue(conv, expected, result, "accounts.ena.uid", "1");
        AddValue(conv, expected, result, "accounts.glogout.uid", "1");

        AddValue(conv, expected, result, "account_info.fio.uid", "Bartolome Denis");
        AddValue(conv, expected, result, "account_info.display_name.uid", "");
        AddValue(conv, expected, result, "account_info.sex.uid", "0");
        AddValue(conv, expected, result, "account_info.birth_date.uid", "1980-02-13");
        AddValue(conv, expected, result, "account_info.country.uid", "tr");
        AddValue(conv, expected, result, "account_info.city.uid", "Moscow");
        AddValue(conv, expected, result, "account_info.reg_date.uid", "2001-01-01 12:34:56");
        AddValue(conv, expected, result, "account_info.tz.uid", "Europe/Moscow");
        AddValue(conv, expected, result, "account_info.lang.uid", "ru");

        AddValue(conv, expected, result, "userinfo.firstname.uid", "Denis");
        AddValue(conv, expected, result, "userinfo.lastname.uid", "Bartolome");
        AddValue(conv, expected, result, "userinfo.display_name.uid", "");
        AddValue(conv, expected, result, "userinfo.sex.uid", "0");
        AddValue(conv, expected, result, "userinfo.birth_date.uid", "1980-02-13");
        AddValue(conv, expected, result, "userinfo.country.uid", "tr");
        AddValue(conv, expected, result, "userinfo.city.uid", "Moscow");
        AddValue(conv, expected, result, "userinfo.reg_date.uid", "2001-01-01 12:34:56");
        AddValue(conv, expected, result, "userinfo.tz.uid", "Europe/Moscow");
        AddValue(conv, expected, result, "userinfo.lang.uid", "ru");

        AddValue(conv, expected, result, "subscription.suid.2", "85052");
        AddValue(conv, expected, result, "subscription.login.2", "malice1");
        AddValue(conv, expected, result, "subscription.host_id.2", "1234");
        AddValue(conv, expected, result, "subscription.login_rule.2", "1");
        AddValue(conv, expected, result, "subscription.born_date.2", "2001-01-01 12:34:56");

        AddValue(conv, expected, result, "subscription.suid.16", "85052");
        AddValue(conv, expected, result, "subscription.login.16", "malice");
        AddValue(conv, expected, result, "subscription.host_id.16", "1234");
        AddValue(conv, expected, result, "subscription.login_rule.16", "1");
        AddValue(conv, expected, result, "subscription.born_date.16", "2001-01-01 12:34:56");

        AddValue(conv, expected, result, "subscription.suid.8", "1");
        AddValue(conv, expected, result, "subscription.login.8", "malice");
        AddValue(conv, expected, result, "subscription.login_rule.8", "1");
        AddValue(conv, expected, result, "subscription.born_date.8", "");

        // addValue(conv, expected, result, "subscription.suid.36", "");
        // addValue(conv, expected, result, "subscription.login.36", "");
        // addValue(conv, expected, result, "subscription.host_id.36", "");
        // addValue(conv, expected, result, "subscription.login_rule.36", "");

        AddValue(conv, expected, result, "subscription.suid.42", "1");
        AddValue(conv, expected, result, "subscription.login.42", "malice");
        AddValue(conv, expected, result, "subscription.host_id.42", "100");
        AddValue(conv, expected, result, "subscription.login_rule.42", "1");
        AddValue(conv, expected, result, "subscription.born_date.42", "");

        AddValue(conv, expected, result, "userinfo_safe.hintq.uid", "who am i");
        AddValue(conv, expected, result, "userinfo_safe.hinta.uid", "looser");

        AddValue(conv, expected, result, "password_quality.version.uid", "2");
        AddValue(conv, expected, result, "password_quality.quality.uid", "100");

        AddValue(conv, expected, result, "hosts.db_id.2", "testpg");

        AddValue(conv, expected, result, "adtarget.uid.uid", "");

        // addValue(conv, expected, result, "userphones.number.uid", "");
        // addValue(conv, expected, result, "userphones.confirmed.uid", "");
    }

    static void FillMalice2Fields(TDbFieldsConverter& conv,
                                  std::map<TString, TString>& expected,
                                  std::map<TString, TDbFieldsResult*>& result) {
        AddValue(conv, expected, result, "accounts.login.uid", "malice2");
        AddValue(conv, expected, result, "accounts.ena.uid", "1");
        AddValue(conv, expected, result, "accounts.glogout.uid", "1");

        AddValue(conv, expected, result, "account_info.fio.uid", "Malice None Нет");
        AddValue(conv, expected, result, "account_info.display_name.uid", "");
        AddValue(conv, expected, result, "account_info.sex.uid", "2");
        AddValue(conv, expected, result, "account_info.birth_date.uid", "");
        AddValue(conv, expected, result, "account_info.country.uid", "ru");
        AddValue(conv, expected, result, "account_info.city.uid", "Рязань");
        AddValue(conv, expected, result, "account_info.reg_date.uid", "2000-08-02 23:40:09");
        AddValue(conv, expected, result, "account_info.tz.uid", "Europe/Moscow");
        AddValue(conv, expected, result, "account_info.lang.uid", "ru");

        AddValue(conv, expected, result, "userinfo.firstname.uid", "None Нет");
        AddValue(conv, expected, result, "userinfo.lastname.uid", "Malice");
        AddValue(conv, expected, result, "userinfo.display_name.uid", "");
        AddValue(conv, expected, result, "userinfo.sex.uid", "2");
        AddValue(conv, expected, result, "userinfo.birth_date.uid", "");
        AddValue(conv, expected, result, "userinfo.country.uid", "ru");
        AddValue(conv, expected, result, "userinfo.city.uid", "Рязань");
        AddValue(conv, expected, result, "userinfo.reg_date.uid", "2000-08-02 23:40:09");
        AddValue(conv, expected, result, "userinfo.tz.uid", "Europe/Moscow");
        AddValue(conv, expected, result, "userinfo.lang.uid", "ru");

        AddValue(conv, expected, result, "subscription.suid.2", "281986");
        AddValue(conv, expected, result, "subscription.login.2", "malice");
        AddValue(conv, expected, result, "subscription.host_id.2", "1234");
        AddValue(conv, expected, result, "subscription.login_rule.2", "1");
        AddValue(conv, expected, result, "subscription.born_date.2", "2000-08-02 23:40:09");

        AddValue(conv, expected, result, "subscription.suid.16", "281986");
        AddValue(conv, expected, result, "subscription.login.16", "malice2");
        AddValue(conv, expected, result, "subscription.host_id.16", "1234");
        AddValue(conv, expected, result, "subscription.login_rule.16", "1");
        AddValue(conv, expected, result, "subscription.born_date.16", "2000-08-02 23:40:09");

        AddValue(conv, expected, result, "subscription.suid.8", "1");
        AddValue(conv, expected, result, "subscription.login.8", "Malice2");
        AddValue(conv, expected, result, "subscription.login_rule.8", "1");
        AddValue(conv, expected, result, "subscription.born_date.8", "");

        // addValue(conv, expected, result, "subscription.suid.36", "1");
        // addValue(conv, expected, result, "subscription.login.36", "malice2");
        // addValue(conv, expected, result, "subscription.host_id.36", "6789");
        // addValue(conv, expected, result, "subscription.login_rule.36", "1");

        AddValue(conv, expected, result, "subscription.suid.42", "");
        AddValue(conv, expected, result, "subscription.login.42", "");
        AddValue(conv, expected, result, "subscription.host_id.42", "");
        AddValue(conv, expected, result, "subscription.login_rule.42", "");
        AddValue(conv, expected, result, "subscription.born_date.42", "");

        AddValue(conv, expected, result, "userinfo_safe.hintq.uid", "");
        AddValue(conv, expected, result, "userinfo_safe.hinta.uid", "");

        AddValue(conv, expected, result, "password_quality.version.uid", "");
        AddValue(conv, expected, result, "password_quality.quality.uid", "");

        AddValue(conv, expected, result, "hosts.db_id.2", "testpg");

        AddValue(conv, expected, result, "adtarget.uid.uid", "126020");

        // addValue(conv, expected, result, "userphones.number.uid", "79161236789");
        // addValue(conv, expected, result, "userphones.confirmed.uid", "2005-12-31 23:58:53");
    }

    Y_UNIT_TEST(Convert) {
        TLog::Error("Starting DbFetcher::testConvert");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher);

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        FillMaliceFields(*converter, expected, result);

        std::vector<TString> uids;
        uids.push_back("57155");
        fetcher->FetchByUids(uids);

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("57155", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first + " : " + it->second + "<->" + pair.second->Value(profile));
        }
    }

    Y_UNIT_TEST(ConvertWithoutAttr) {
        TLog::Error("Starting DbFetcher::testConvertWithoutAttr");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher);

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        AddValue(*converter, expected, result, "accounts.login.uid", "malice");

        std::vector<TString> uids;
        uids.push_back("57155");
        fetcher->FetchByUids(uids);

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("57155", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first);
        }
    }

    Y_UNIT_TEST(ConvertByPortalLogin) {
        TLog::Error("Starting DbFetcher::testConvertByPortalLogin");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher);

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        FillMaliceFields(*converter, expected, result);

        fetcher->FetchByLogin("malice", TStrings::EMPTY, {}, TStrings::EMPTY);

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("57155", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first);
        }
    }

    Y_UNIT_TEST(ConvertByMailLogin) {
        TLog::Error("Starting DbFetcher::testConvertByMailLogin");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::vector<TString> sids(1, "2");
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher, sids);

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        FillMaliceFields(*converter, expected, result);

        fetcher->FetchByLogin("malice1", TStrings::EMPTY, sids, TStrings::EMPTY);

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("57155", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first);
        }
    }

    Y_UNIT_TEST(ConvertByNarodMailLogin) {
        TLog::Error("Starting DbFetcher::testConvertByNarodMailLogin");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::vector<TString> sids(1, "16");
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher, sids);

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        FillMaliceFields(*converter, expected, result);

        fetcher->FetchByLogin("malice", TStrings::EMPTY, sids, TStrings::EMPTY);

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("57155", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first);
        }
    }

    Y_UNIT_TEST(ConvertBySuid2) {
        TLog::Error("Starting DbFetcher::testConvertBySuid2");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher);

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        FillMaliceFields(*converter, expected, result);

        fetcher->FetchBySuid("85052", "2");

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("57155", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first);
        }
    }

    Y_UNIT_TEST(LoginMultipleSids) {
        TLog::Error("Starting DbFetcher::testLoginMultipleSids");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::vector<TString> sids(2);
        sids[0] = "2";
        sids[1] = "16";
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher, sids);

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        FillMalice2Fields(*converter, expected, result);
        fetcher->FetchByLogin("malice", TStrings::EMPTY, sids, TStrings::EMPTY);

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("126020", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first);
        }
    }

    Y_UNIT_TEST(LoginMultipleSids2) {
        TLog::Error("Starting DbFetcher::testLoginMultipleSids2");
        std::unique_ptr<TTestDbFetcher> fetcher = Db().CreateFetcher();
        std::unique_ptr<TDbFieldsConverter> converter = Db().CreateConverter(*fetcher);
        std::vector<TString> sids(2);
        sids[0] = "16";
        sids[1] = "2";

        std::map<TString, TString> expected;
        std::map<TString, TDbFieldsResult*> result;

        FillMaliceFields(*converter, expected, result);

        fetcher->FetchByLogin("malice", TStrings::EMPTY, sids, TStrings::EMPTY);

        const TDbProfile* profile = fetcher->NextProfile();

        UNIT_ASSERT(nullptr != profile);
        UNIT_ASSERT_VALUES_EQUAL("57155", profile->Uid());

        for (const auto& pair : result) {
            std::map<TString, TString>::iterator it = expected.find(pair.first);
            UNIT_ASSERT(expected.end() != it);
            UNIT_ASSERT_VALUES_EQUAL_C(it->second, pair.second->Value(profile), pair.first);
        }
    }

    Y_UNIT_TEST(GetAttrsCondition) {
        TDbProfile::TAttrs attrs;

        attrs["1001"];
        attrs["1010"];
        attrs["1024"];

        UNIT_ASSERT_VALUES_EQUAL(" AND type < 30", TTestDbFetcher::GetAttrsCondition(attrs));

        for (int num = 0; num < 30; ++num) {
            attrs[IntToString<10>(num)];
            UNIT_ASSERT_VALUES_EQUAL(" AND type < 30", TTestDbFetcher::GetAttrsCondition(attrs));
        }

        attrs["101"];
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 30 OR type IN (101))", TTestDbFetcher::GetAttrsCondition(attrs));

        attrs["34"];
        attrs["35"];
        attrs["36"];
        attrs["39"];
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 30 OR type IN (101,39,34))", TTestDbFetcher::GetAttrsCondition(attrs));

        for (int num = 40; num < 47; ++num) {
            attrs[IntToString<10>(num)];
        }
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 30 OR type IN (101,46,45,44,43,42,41,40,39,34))",
                                 TTestDbFetcher::GetAttrsCondition(attrs));
        for (int num = 47; num < 50; ++num) {
            attrs[IntToString<10>(num)];
        }
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 41 OR type IN (101,49,48,47,46,45,44,43,42,41))",
                                 TTestDbFetcher::GetAttrsCondition(attrs));

        attrs["50"];
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 42 OR type IN (101,50,49,48,47,46,45,44,43,42))",
                                 TTestDbFetcher::GetAttrsCondition(attrs));

        for (int num = 51; num < 60; ++num) {
            attrs[IntToString<10>(num)];
        }
        UNIT_ASSERT_VALUES_EQUAL("", TTestDbFetcher::GetAttrsCondition(attrs));
    }

    Y_UNIT_TEST(GetAttrsConditionCustom) {
        TDbProfile::TAttrs attrs;

        attrs["1001"];
        attrs["1010"];
        attrs["1024"];

        UNIT_ASSERT_VALUES_EQUAL(" AND type < 5", TTestDbFetcher::GetAttrsCondition(attrs, 5));
        UNIT_ASSERT_VALUES_EQUAL(" AND type < 100", TTestDbFetcher::GetAttrsCondition(attrs, 100));

        attrs["1"];
        attrs["2"];
        attrs["5"];
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 3 OR type IN (5))", TTestDbFetcher::GetAttrsCondition(attrs, 3));
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 5 OR type IN (5))", TTestDbFetcher::GetAttrsCondition(attrs, 5));
        UNIT_ASSERT_VALUES_EQUAL(" AND type < 10", TTestDbFetcher::GetAttrsCondition(attrs, 10));

        attrs["101"];
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 10 OR type IN (101))", TTestDbFetcher::GetAttrsCondition(attrs, 10));
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 100 OR type IN (101))", TTestDbFetcher::GetAttrsCondition(attrs, 100));

        attrs["34"];
        attrs["35"];
        attrs["36"];
        attrs["39"];
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 6 OR type IN (101,39,34))", TTestDbFetcher::GetAttrsCondition(attrs, 6));

        for (int num = 40; num < 47; ++num) {
            attrs[IntToString<10>(num)];
        }
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 7 OR type IN (101,46,45,44,43,42,41,40,39,34))",
                                 TTestDbFetcher::GetAttrsCondition(attrs, 7));
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 44 OR type IN (101,46,45,44))",
                                 TTestDbFetcher::GetAttrsCondition(attrs, 44));
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 50 OR type IN (101))", TTestDbFetcher::GetAttrsCondition(attrs, 50));

        for (int num = 47; num < 50; ++num) {
            attrs[IntToString<10>(num)];
        }
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 41 OR type IN (101,49,48,47,46,45,44,43,42,41))",
                                 TTestDbFetcher::GetAttrsCondition(attrs, 21));
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 44 OR type IN (101,49,48,47,46,45,44))",
                                 TTestDbFetcher::GetAttrsCondition(attrs, 44));
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 50 OR type IN (101))", TTestDbFetcher::GetAttrsCondition(attrs, 50));

        attrs["50"];
        UNIT_ASSERT_VALUES_EQUAL(" AND (type < 42 OR type IN (101,50,49,48,47,46,45,44,43,42))",
                                 TTestDbFetcher::GetAttrsCondition(attrs, 40));

        for (int num = 1; num < 53; ++num) {
            attrs[IntToString<10>(num)];
        }
        UNIT_ASSERT_VALUES_EQUAL("", TTestDbFetcher::GetAttrsCondition(attrs, 1));
        UNIT_ASSERT_VALUES_EQUAL("", TTestDbFetcher::GetAttrsCondition(attrs, 5));
        UNIT_ASSERT_VALUES_EQUAL("", TTestDbFetcher::GetAttrsCondition(attrs, 10));
        UNIT_ASSERT_VALUES_EQUAL("", TTestDbFetcher::GetAttrsCondition(attrs, 100));
    }
}
