#include <passport/infra/daemons/blackbox/ut/common/common.h>

#include <passport/infra/daemons/blackbox/src/misc/db_profile.h>
#include <passport/infra/daemons/blackbox/src/misc/db_types.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NBb;

Y_UNIT_TEST_SUITE(TPasspDbProfileTest) {
    Y_UNIT_TEST(DbValue) {
        TDbValue v1;
        UNIT_ASSERT(!v1.Exists);
        UNIT_ASSERT(v1.Value.empty());
        UNIT_ASSERT(!v1.AsBoolean());
        UNIT_ASSERT_VALUES_EQUAL(0, v1.AsTime());

        TDbValue v2("100");
        UNIT_ASSERT(v2.Exists);
        UNIT_ASSERT_VALUES_EQUAL("100", v2.Value);
        UNIT_ASSERT(v2.AsBoolean());
        UNIT_ASSERT_VALUES_EQUAL(100, v2.AsTime());
    }

    static void CheckEmptyProfile(const TTestDbProfile& p) {
        UNIT_ASSERT(p.Empty_);

        UNIT_ASSERT_VALUES_EQUAL("", p.Uid());
        UNIT_ASSERT_VALUES_EQUAL("", p.Sid());
        UNIT_ASSERT(!p.Pdd());
        UNIT_ASSERT(!p.CatchAll());
        UNIT_ASSERT_VALUES_EQUAL("", p.PddDomId());
        UNIT_ASSERT_VALUES_EQUAL("", p.PddDomain());
        UNIT_ASSERT_VALUES_EQUAL("", p.PddDomainUtf());
        UNIT_ASSERT(!p.PddDomItem().IsFound());
        UNIT_ASSERT_EXCEPTION_CONTAINS(p.DomainList().size(), yexception, "DbProfile: missing domain cache for pdd user");

        UNIT_ASSERT_VALUES_EQUAL("", p.AltDomId());
        UNIT_ASSERT_VALUES_EQUAL("", p.AltDomain());

        UNIT_ASSERT(p.Aliases().empty());
        UNIT_ASSERT(p.AliasesPdd().empty());
        UNIT_ASSERT(p.AliasesOldPublic().empty());
        UNIT_ASSERT(!p.NeedAliases());
        UNIT_ASSERT(!p.HasAlias(TAlias::PORTAL_LOGIN));
        UNIT_ASSERT(!p.HasAlias(TAlias::MAIL_LOGIN));
        UNIT_ASSERT(!p.HasAlias(TAlias::PDD_ALIAS_LOGIN));
        UNIT_ASSERT(!p.HasAlias(TAlias::OLDPUBLICID));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::PORTAL_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::MAIL_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::PDD_ALIAS_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::OLDPUBLICID));

        UNIT_ASSERT(p.Attrs().empty());
        UNIT_ASSERT(!p.Suid2_.Exists);
        UNIT_ASSERT(p.ExtendedPhoneAttrs().empty());
        UNIT_ASSERT(p.ExtendedEmailAttrs().empty());

        UNIT_ASSERT(!p.NeedPhoneOperations_);
        UNIT_ASSERT(p.PhoneOperations().empty());
        UNIT_ASSERT(p.PhoneBindings().empty());

        UNIT_ASSERT_EXCEPTION_CONTAINS(p.Get(-1), yexception, "Incorrect hidden index");
        UNIT_ASSERT_EXCEPTION_CONTAINS(p.Get(0), yexception, "Incorrect hidden index");
        UNIT_ASSERT_EXCEPTION_CONTAINS(p.Get(1), yexception, "Incorrect hidden index");

        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::NoAlias, (int)p.MatchesLogin("", TAlias::PORTAL_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::NoAlias, (int)p.MatchesLogin("", TAlias::PDD_ALIAS_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::NoAlias, (int)p.MatchesLogin("", TAlias::OLDPUBLICID));

        const TString strProfile =
            "DbProfile for uid '':\n"
            "Sid = '', pdd domain_id='', alternative domain id='':\n"
            "Aliases:\n"
            "Attrs:\n"
            "Suid2:<NULL>\n"
            "Extended phone attrs:\ntype=1:\n"
            "Extended email attrs:\ntype=2:\n"
            "Extended webauthn attrs:\ntype=3:\n"
            "Phone operations:\n"
            "Index:\n";

        UNIT_ASSERT_VALUES_EQUAL(strProfile, p.ToString());

        UNIT_ASSERT_VALUES_EQUAL("", p.DefaultPhoneId());
    }

    static void CheckSampleProfile(const TTestDbProfile& p) {
        UNIT_ASSERT(!p.Empty_);

        UNIT_ASSERT_VALUES_EQUAL("70501", p.Uid());
        UNIT_ASSERT_VALUES_EQUAL("2", p.Sid());
        UNIT_ASSERT(!p.Pdd());
        UNIT_ASSERT(p.CatchAll());
        UNIT_ASSERT_VALUES_EQUAL("111", p.PddDomId());
        UNIT_ASSERT_VALUES_EQUAL("xn-----6kccl6ap0agkc7a2i.xn--p1ai", p.PddDomain());
        UNIT_ASSERT_VALUES_EQUAL("рога-и-копыта.рф", p.PddDomainUtf());
        UNIT_ASSERT(p.PddDomItem().IsFound());
        UNIT_ASSERT_VALUES_EQUAL(0, p.DomainList().size());

        UNIT_ASSERT_VALUES_EQUAL("3", p.AltDomId());
        UNIT_ASSERT_VALUES_EQUAL("mail.ru", p.AltDomain());

        UNIT_ASSERT_VALUES_EQUAL(3, p.Aliases().size());
        UNIT_ASSERT(!p.AliasesPdd().empty());
        UNIT_ASSERT(!p.AliasesOldPublic().empty());
        UNIT_ASSERT(p.NeedAliases());
        const TDbProfile::TMultiAlias& pddAliases = p.AliasesPdd();
        UNIT_ASSERT_VALUES_EQUAL(3, pddAliases.size());
        UNIT_ASSERT(pddAliases.end() != pddAliases.find("admin@рога-и-копыта.рф"));
        UNIT_ASSERT(pddAliases.end() != pddAliases.find("glavbukh@рога-и-копыта.рф"));
        UNIT_ASSERT(pddAliases.end() != pddAliases.find("janitor@рога-и-копыта.рф"));
        const TDbProfile::TMultiAlias& oldPublic = p.AliasesOldPublic();
        UNIT_ASSERT_VALUES_EQUAL(2, oldPublic.size());
        UNIT_ASSERT(oldPublic.end() != oldPublic.find("old_public_id_1"));
        UNIT_ASSERT(oldPublic.end() != oldPublic.find("old_public_id_2"));

        UNIT_ASSERT(p.HasAlias(TAlias::PORTAL_LOGIN));
        UNIT_ASSERT(!p.HasAlias(TAlias::MAIL_LOGIN));
        UNIT_ASSERT(!p.HasAlias(TAlias::LITE_LOGIN));
        UNIT_ASSERT(p.HasAlias(TAlias::PDD_MASTER_LOGIN));
        UNIT_ASSERT(p.HasAlias(TAlias::PDD_ALIAS_LOGIN));
        UNIT_ASSERT(p.HasAlias(TAlias::OLDPUBLICID));
        UNIT_ASSERT_VALUES_EQUAL("test-user", p.GetAlias(TAlias::PORTAL_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::MAIL_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::LITE_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("gendir@рога-и-копыта.рф", p.GetAlias(TAlias::PDD_MASTER_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::PDD_ALIAS_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL("", p.GetAlias(TAlias::OLDPUBLICID));

        UNIT_ASSERT_VALUES_EQUAL(5, p.Attrs().size());
        UNIT_ASSERT(p.Attrs().at(TAttr::ACCOUNT_HAVE_HINT).Exists);
        UNIT_ASSERT_VALUES_EQUAL("0", p.Attrs().at(TAttr::ACCOUNT_HAVE_HINT).Value);
        UNIT_ASSERT(p.Attrs().at(TAttr::ACCOUNT_HAVE_PASSWORD).Exists);
        UNIT_ASSERT_VALUES_EQUAL("1", p.Attrs().at(TAttr::ACCOUNT_HAVE_PASSWORD).Value);
        UNIT_ASSERT(p.Attrs().at(TAttr::KARMA_VALUE).Exists);
        UNIT_ASSERT_VALUES_EQUAL("100", p.Attrs().at(TAttr::KARMA_VALUE).Value);
        UNIT_ASSERT(!p.Attrs().at(TAttr::ACCOUNT_BETATESTER).Exists);
        UNIT_ASSERT_VALUES_EQUAL("", p.Attrs().at(TAttr::ACCOUNT_BETATESTER).Value);
        UNIT_ASSERT(p.Attrs().at(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME).Exists);
        UNIT_ASSERT_VALUES_EQUAL("1", p.Attrs().at(TAttr::ACCOUNT_GLOBAL_LOGOUT_DATETIME).Value);
        UNIT_ASSERT_EXCEPTION_CONTAINS(p.Attrs().at(TAttr::PERSON_LANGUAGE), std::out_of_range, "key not found");

        UNIT_ASSERT(p.Suid2().Exists);
        UNIT_ASSERT_VALUES_EQUAL("12341234", p.Suid2().Value);

        UNIT_ASSERT_VALUES_EQUAL(2, p.ExtendedPhoneAttrs().size());
        UNIT_ASSERT_VALUES_EQUAL(2, p.ExtendedPhoneAttrs().at("").size());
        UNIT_ASSERT_VALUES_EQUAL(2, p.ExtendedPhoneAttrs().at("5").size());
        UNIT_ASSERT_VALUES_EQUAL("79161234567", p.ExtendedPhoneAttrs().at("5").at(TPhoneAttr::NUMBER).Value);
        UNIT_ASSERT_VALUES_EQUAL("1414141414", p.ExtendedPhoneAttrs().at("5").at(TPhoneAttr::CONFIRMED).Value);

        UNIT_ASSERT_VALUES_EQUAL(2, p.ExtendedEmailAttrs().size());
        UNIT_ASSERT_VALUES_EQUAL(3, p.ExtendedEmailAttrs().at("").size());
        UNIT_ASSERT_VALUES_EQUAL(3, p.ExtendedEmailAttrs().at("2").size());
        UNIT_ASSERT_VALUES_EQUAL("user@gmail.com", p.ExtendedEmailAttrs().at("2").at(TEmailAttr::ADDRESS).Value);
        UNIT_ASSERT_VALUES_EQUAL("1212121212", p.ExtendedEmailAttrs().at("2").at(TEmailAttr::CONFIRMED).Value);
        UNIT_ASSERT_VALUES_EQUAL("1", p.ExtendedEmailAttrs().at("2").at(TEmailAttr::IS_UNSAFE).Value);

        UNIT_ASSERT_VALUES_EQUAL(2, p.ExtendedWebauthnAttrs().size());
        UNIT_ASSERT_VALUES_EQUAL(2, p.ExtendedWebauthnAttrs().at("").size());
        UNIT_ASSERT_VALUES_EQUAL(2, p.ExtendedWebauthnAttrs().at("10").size());
        UNIT_ASSERT_VALUES_EQUAL("asdfasdfasdf", p.ExtendedWebauthnAttrs().at("10").at(TWebauthnAttr::EXTERNAL_ID).Value);
        UNIT_ASSERT_VALUES_EQUAL("1616161616", p.ExtendedWebauthnAttrs().at("10").at(TWebauthnAttr::CREATED).Value);

        UNIT_ASSERT(p.NeedPhoneOperations_);
        UNIT_ASSERT_VALUES_EQUAL(2, p.PhoneOperations().size());
        UNIT_ASSERT_VALUES_EQUAL("1,2,3,4", p.PhoneOperations().at("77"));
        UNIT_ASSERT_VALUES_EQUAL("51,52,53,,,54", p.PhoneOperations().at("177"));

        UNIT_ASSERT_VALUES_EQUAL((int)EPhoneBindingsType::Current, (int)p.PhoneBindingsType_);
        UNIT_ASSERT_VALUES_EQUAL(1, p.PhoneBindings().size());
        UNIT_ASSERT_VALUES_EQUAL("current", p.PhoneBindings()[0].Type);
        UNIT_ASSERT_VALUES_EQUAL("79162222222", p.PhoneBindings()[0].Number);
        UNIT_ASSERT_VALUES_EQUAL("1", p.PhoneBindings()[0].PhoneId);
        UNIT_ASSERT_VALUES_EQUAL("70501", p.PhoneBindings()[0].Uid);
        UNIT_ASSERT_VALUES_EQUAL("1400554360", p.PhoneBindings()[0].Bound);
        UNIT_ASSERT_VALUES_EQUAL("1", p.PhoneBindings()[0].Flags);

        UNIT_ASSERT_EXCEPTION_CONTAINS(p.Get(-1), yexception, "Incorrect hidden index");
        UNIT_ASSERT_VALUES_EQUAL("test-user", p.Get(0)->Value);
        UNIT_ASSERT_VALUES_EQUAL("", p.Get(6)->Value);

        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::NoAlias, (int)p.MatchesLogin("", TAlias::LITE_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::NoMatch, (int)p.MatchesLogin("s", TAlias::PORTAL_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::Match, (int)p.MatchesLogin("test-user", TAlias::PORTAL_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::NoMatch, (int)p.MatchesLogin("admin", TAlias::PDD_ALIAS_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::Match, (int)p.MatchesLogin("admin@рога-и-копыта.рф", TAlias::PDD_ALIAS_LOGIN));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::NoMatch, (int)p.MatchesLogin("old_public_id", TAlias::OLDPUBLICID));
        UNIT_ASSERT_VALUES_EQUAL((int)TDbProfile::Match, (int)p.MatchesLogin("old_public_id_1", TAlias::OLDPUBLICID));

        const TString strProfile =
            "DbProfile for uid '70501':\n"
            "Sid = '2', pdd domain_id='111', alternative domain id='3':\n"
            "Aliases:\n"
            "\t'1' : 'test-user' (index=0)\n"
            "\t'2' : '' (index=1)\n"
            "\t'7' : 'gendir@рога-и-копыта.рф' (index=2)\n"
            "\t'8' : 'admin@рога-и-копыта.рф'\n"
            "\t'8' : 'glavbukh@рога-и-копыта.рф'\n"
            "\t'8' : 'janitor@рога-и-копыта.рф'\n"
            "\t'20' : 'old_public_id_1'\n"
            "\t'20' : 'old_public_id_2'\n"
            "Attrs:\n"
            "\t'1005' : '1' (index=4)\n"
            "\t'1006' : '0' (index=3)\n"
            "\t'17' : '100' (index=5)\n"
            "\t'4' : '1' (index=7)\n"
            "\t'8' : '<NULL>' (index=6)\n"
            "Suid2:12341234\n"
            "Extended phone attrs:\ntype=1:\n"
            "id= {\n\t'1' : '<NULL>'\n\t'4' : '<NULL>'\n}\n"
            "id=5 {\n"
            "\t'1' : '79161234567'\n"
            "\t'4' : '1414141414'\n}\n"
            "Extended email attrs:\ntype=2:\n"
            "id= {\n\t'1' : '<NULL>'\n\t'3' : '<NULL>'\n\t'6' : '<NULL>'\n}\n"
            "id=2 {\n\t'1' : 'user@gmail.com'\n\t'3' : '1212121212'\n\t'6' : '1'\n}\n"
            "Extended webauthn attrs:\ntype=3:\n"
            "id= {\n\t'1' : '<NULL>'\n\t'5' : '<NULL>'\n}\n"
            "id=10 {\n\t'1' : 'asdfasdfasdf'\n\t'5' : '1616161616'\n}\n"
            "Phone operations:\n"
            "\t'177' : '51,52,53,,,54'\n"
            "\t'77' : '1,2,3,4'\n"
            "Index:\n"
            "\t 0 : 'test-user'\n"
            "\t 1 : ''\n"
            "\t 2 : 'gendir@рога-и-копыта.рф'\n"
            "\t 3 : '0'\n"
            "\t 4 : '1'\n"
            "\t 5 : '100'\n"
            "\t 6 : '<NULL>'\n"
            "\t 7 : '1'\n"
            "\t 8 : '12341234'\n";

        UNIT_ASSERT_VALUES_EQUAL(strProfile, p.ToString());
        UNIT_ASSERT_VALUES_EQUAL("5", p.DefaultPhoneId());
    }

    Y_UNIT_TEST(DbProfileConstructors) {
        TTestDbProfile p1;
        CheckEmptyProfile(p1);

        p1.FillWithSampleData();
        CheckSampleProfile(p1);

        TTestDbProfile p2(p1);
        CheckSampleProfile(p2);
    }
};
